using UnityEditor;
using UnityEditor.Build;
using UnityEngine;

/// <summary>
/// Correction spécifique pour l'erreur CS2006 après switch vers WebGL
/// </summary>
public class FixWebGLDefines
{
    [MenuItem("Tools/Fix WebGL Defines (CS2006)")]
    public static void FixWebGLDefinesError()
    {
        Debug.Log("🔧 Correction des erreurs CS2006 pour WebGL...");
        
        // Cibler spécifiquement WebGL
        BuildTargetGroup webGL = BuildTargetGroup.WebGL;
        
        try
        {
#if UNITY_2023_1_OR_NEWER
            var namedTarget = NamedBuildTarget.WebGL;
            
            // Obtenir les symboles actuels
            string currentDefines = PlayerSettings.GetScriptingDefineSymbols(namedTarget);
            Debug.Log($"Symboles WebGL actuels: '{currentDefines}'");
            
            // Nettoyer complètement les symboles pour WebGL
            PlayerSettings.SetScriptingDefineSymbols(namedTarget, "");
            Debug.Log("✓ Symboles WebGL nettoyés");
            
#else
            // Unity 2022 et antérieur
            string currentDefines = PlayerSettings.GetScriptingDefineSymbolsForGroup(webGL);
            Debug.Log($"Symboles WebGL actuels: '{currentDefines}'");
            
            // Nettoyer complètement
            PlayerSettings.SetScriptingDefineSymbolsForGroup(webGL, "");
            Debug.Log("✓ Symboles WebGL nettoyés");
#endif
            
            // Forcer la sauvegarde
            AssetDatabase.SaveAssets();
            AssetDatabase.Refresh(ImportAssetOptions.ForceUpdate);
            
            Debug.Log("✅ CORRECTION TERMINÉE !");
            Debug.Log("👉 FERMEZ Unity et supprimez le dossier Library/ScriptAssemblies");
            Debug.Log("👉 Puis rouvrez Unity");
        }
        catch (System.Exception e)
        {
            Debug.LogError($"❌ Erreur: {e.Message}");
        }
    }
    
    [MenuItem("Tools/Clear ALL Defines (All Platforms)")]
    public static void ClearAllDefines()
    {
        Debug.Log("🧹 Nettoyage de TOUS les symboles pour TOUTES les plateformes...");
        
        BuildTargetGroup[] allTargets = new BuildTargetGroup[]
        {
            BuildTargetGroup.Standalone,
            BuildTargetGroup.WebGL,
            BuildTargetGroup.Android,
            BuildTargetGroup.iOS,
            BuildTargetGroup.WSA
        };
        
        foreach (var target in allTargets)
        {
            try
            {
#if UNITY_2023_1_OR_NEWER
                var namedTarget = NamedBuildTarget.FromBuildTargetGroup(target);
                PlayerSettings.SetScriptingDefineSymbols(namedTarget, "");
#else
                PlayerSettings.SetScriptingDefineSymbolsForGroup(target, "");
#endif
                Debug.Log($"  ✓ {target}: nettoyé");
            }
            catch (System.Exception e)
            {
                Debug.LogWarning($"  ⚠ {target}: {e.Message}");
            }
        }
        
        AssetDatabase.SaveAssets();
        AssetDatabase.Refresh(ImportAssetOptions.ForceUpdate);
        
        Debug.Log("✅ Tous les symboles nettoyés !");
        Debug.Log("👉 Fermez Unity, supprimez Library/ScriptAssemblies, et rouvrez");
    }
    
    [MenuItem("Tools/Diagnose Define Symbols")]
    public static void DiagnoseDefines()
    {
        Debug.Log("=== DIAGNOSTIC SCRIPTING DEFINE SYMBOLS ===");
        
        BuildTargetGroup[] allTargets = new BuildTargetGroup[]
        {
            BuildTargetGroup.Standalone,
            BuildTargetGroup.WebGL,
            BuildTargetGroup.Android,
            BuildTargetGroup.iOS
        };
        
        foreach (var target in allTargets)
        {
            try
            {
#if UNITY_2023_1_OR_NEWER
                var namedTarget = NamedBuildTarget.FromBuildTargetGroup(target);
                string defines = PlayerSettings.GetScriptingDefineSymbols(namedTarget);
#else
                string defines = PlayerSettings.GetScriptingDefineSymbolsForGroup(target);
#endif
                
                bool isEmpty = string.IsNullOrWhiteSpace(defines);
                bool hasEmptyTokens = !isEmpty && defines.Contains(";;");
                bool startsWithSemicolon = !isEmpty && defines.StartsWith(";");
                bool endsWithSemicolon = !isEmpty && defines.EndsWith(";");
                
                string status = isEmpty ? "✓ VIDE (OK)" : 
                               hasEmptyTokens || startsWithSemicolon || endsWithSemicolon ? "❌ PROBLÈME DÉTECTÉ" : 
                               "✓ OK";
                
                Debug.Log($"{target}: {status}");
                if (!isEmpty)
                {
                    Debug.Log($"  Valeur: '{defines}'");
                    if (hasEmptyTokens) Debug.Log("  ⚠ Contient des tokens vides (;;)");
                    if (startsWithSemicolon) Debug.Log("  ⚠ Commence par un point-virgule");
                    if (endsWithSemicolon) Debug.Log("  ⚠ Se termine par un point-virgule");
                }
            }
            catch (System.Exception e)
            {
                Debug.LogWarning($"{target}: ⚠ Erreur - {e.Message}");
            }
        }
        
        Debug.Log("============================================");
    }
}


