# 🍎 Fix : Voile Blanc sur Vidéos (Mac)

## 🎯 Problème

Sur Mac, un **voile blanc** apparaît par-dessus les vidéos de dialogue, rendant le jeu injouable.

**Cause** : 
- Problème de rendu Metal (API graphique Mac)
- Image UI blanche avec sortingOrder élevé qui passe devant la vidéo
- Gestion incorrecte de l'alpha sur Mac

---

## ✅ SOLUTION RAPIDE

### **Étape 1 : Ajouter MacCompatibilityFix à la Scène Player**

1. **Ouvrir la scène `Player`** dans Unity

2. **Créer un GameObject vide** :
   - Clic droit Hierarchy → Create Empty
   - Nommer : `MacCompatibilityManager`

3. **Ajouter le script** :
   - Sélectionner `MacCompatibilityManager`
   - Inspector → **Add Component**
   - Rechercher `MacCompatibilityFix`
   - Ajouter

4. **Configurer** :
   - ✅ `Enable Diagnostic` : **true** (pour voir les logs)
   - ✅ `Fix White Veil` : **true** ← **IMPORTANT**
   - ✅ `Fix Missing Assets` : **true**
   - ✅ `Fix Canvas Order` : **true**

5. **Sauvegarder la scène** : Ctrl+S

### **Étape 2 : Build & Test**

1. **File** → **Build and Run**
2. Lancer sur Mac
3. **Le voile devrait disparaître** ✅

---

## 🔍 Comment Ça Marche

Le script `MacCompatibilityFix.cs` :

1. ✅ **Détecte les images blanches** suspectes (>= 50000 sortingOrder)
2. ✅ **Met leur alpha à 0** pour les rendre transparentes
3. ✅ **Corrige l'ordre des Canvas** pour Mac
4. ✅ **Configure SSL** pour le chargement des assets

**Code clé** (dans `FixWhiteVeilIssue()`) :
```csharp
// Si c'est une Image sans sprite et blanche
if (img.sprite == null && img.gameObject.name.Contains("Panel"))
{
    Color c = img.color;
    c.a = 0f; // Complètement transparent
    img.color = c;
}
```

---

## 🧪 Vérifier que Ça Marche

### **Dans les Logs Build Mac** :

Chercher dans `~/Library/Logs/[Company]/[Product]/Player.log` :

```
[MacFix] 🔧 Recherche du voile blanc...
[MacFix] Canvas trouvé: DialogueCanvas, sortingOrder: 50000
[MacFix] ⚠️ Image blanche suspecte trouvée: OverlayPanel
[MacFix] ✅ Correction de l'alpha de OverlayPanel
```

Si vous voyez ces logs → Le fix fonctionne ! ✅

---

## 🔧 Debug en Direct (Dans le Build Mac)

Le script permet de **relancer le fix** pendant que le jeu tourne :

1. Lancer le build Mac
2. Pendant un dialogue (avec voile blanc)
3. **Appuyer sur F11** 🔑
4. Le fix se relance
5. Le voile devrait disparaître

---

## ⚠️ Si Ça Ne Marche Toujours Pas

### **Solution Alternative : Forcer l'Alpha du Dialogue**

Créez un nouveau script `DialogueOverlayFix.cs` :

```csharp
using UnityEngine;
using UnityEngine.UI;
using System.Collections;

public class DialogueOverlayFix : MonoBehaviour
{
    void Start()
    {
        StartCoroutine(FixOverlayLoop());
    }
    
    IEnumerator FixOverlayLoop()
    {
        while (true)
        {
            yield return new WaitForSeconds(0.5f);
            
            // Trouver toutes les images blanches
            Image[] allImages = FindObjectsByType<Image>(FindObjectsSortMode.None);
            
            foreach (Image img in allImages)
            {
                // Si l'image est blanche, sans sprite, et dans un Canvas de dialogue
                if (img.color.r > 0.9f && img.color.g > 0.9f && img.color.b > 0.9f && 
                    img.sprite == null && img.GetComponentInParent<Canvas>() != null)
                {
                    Canvas canvas = img.GetComponentInParent<Canvas>();
                    
                    // Si le canvas est un overlay de dialogue (sortingOrder élevé)
                    if (canvas.sortingOrder >= 40000)
                    {
                        Color c = img.color;
                        c.a = 0f; // Transparent
                        img.color = c;
                        
                        Debug.Log($"[OverlayFix] Image blanche transparente: {img.name}");
                    }
                }
            }
        }
    }
}
```

**Ajouter ce script** sur le même GameObject `MacCompatibilityManager`.

---

## 🎨 Solution Ultime : Modifier DialoguePlayer

Si les solutions précédentes ne marchent pas, le problème vient du `DialoguePlayer` lui-même.

### **Localiser le Problème**

Le `DialoguePlayer` crée probablement un **Panel blanc** pour overlay. Sur Mac, ce panel reste visible.

**Dans le code `DialoguePlayer`**, chercher :
```csharp
// Création d'un panel overlay blanc
overlayImage = ...
overlayImage.color = Color.white; // ← PROBLÈME
```

**Changer en** :
```csharp
#if UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX
    overlayImage.color = Color.clear; // Transparent sur Mac
#else
    overlayImage.color = Color.white;
#endif
```

---

## 📋 Checklist Complète

**Configuration** :
- [ ] Script `MacCompatibilityFix` ajouté à la scène Player
- [ ] `Fix White Veil` = true
- [ ] `Enable Diagnostic` = true (pour logs)
- [ ] Scène sauvegardée

**Test Build** :
- [ ] Build Mac créé
- [ ] Lancé sur Mac
- [ ] Dialogue testé
- [ ] Voile blanc disparu ? ✅

**Si problème persiste** :
- [ ] F11 pressé pendant le dialogue (relance fix)
- [ ] Logs consultés (`~/Library/Logs/.../Player.log`)
- [ ] Logs `[MacFix]` présents ?
- [ ] Script `DialogueOverlayFix` ajouté (alternative)

---

## 💡 Pourquoi Ce Problème Existe

**Sur Windows/Linux** :
- DirectX/OpenGL gèrent correctement les overlays transparents
- Pas de problème

**Sur Mac** :
- Metal (API graphique) a un comportement différent
- Les overlays blancs ne deviennent pas transparents automatiquement
- Besoin de forcer alpha = 0 manuellement

C'est un bug connu d'Unity avec Metal sur Mac ! 🍎

---

## 🆘 Support

Si le voile persiste après avoir suivi toutes les étapes :

1. **Partager les logs** `[MacFix]` du Player.log
2. **Préciser** :
   - Le voile apparaît sur quel type de dialogue ? (BEFORE/AFTER/SUCCESS/FAIL)
   - Le voile couvre toute la vidéo ou juste une partie ?
   - F11 fait-il disparaître le voile temporairement ?

---

**Date** : 21 Octobre 2025
**Problème** : Voile blanc sur vidéos dialogue Mac
**Solution** : Script MacCompatibilityFix actif dans scène Player

