using UnityEngine;

/// <summary>
/// Système de logs centralisé avec niveaux de verbosité
/// </summary>
public static class DebugLogger
{
    public enum LogLevel
    {
        None = 0,      // Aucun log
        Error = 1,     // Seulement les erreurs critiques
        Warning = 2,   // Erreurs + warnings
        Info = 3,      // Erreurs + warnings + infos importantes
        Debug = 4,     // Tout (mode debug complet)
    }

    // 🔧 CONFIGUREZ ICI LE NIVEAU DE LOG GLOBAL
    public static LogLevel GlobalLogLevel = LogLevel.Warning;

    // Logs par catégorie
    public static LogLevel GameManagerLogLevel = LogLevel.Warning;
    public static LogLevel MapManagerLogLevel = LogLevel.Warning;
    public static LogLevel LevelManagerLogLevel = LogLevel.Info; // Info pour voir les chargements importants
    public static LogLevel DialogueLogLevel = LogLevel.Warning;
    public static LogLevel CalculatorLogLevel = LogLevel.Warning;
    public static LogLevel LoadingLogLevel = LogLevel.Warning;
    public static LogLevel UILogLevel = LogLevel.Warning;
    public static LogLevel GameConfigLoaderLogLevel = LogLevel.Debug; // Debug temporaire pour écran noir
    public static LogLevel GameLauncherLogLevel = LogLevel.Debug; // Debug temporaire pour écran noir
    public static LogLevel LevelCardLogLevel = LogLevel.Debug; // Debug pour le lancement depuis menu
    public static LogLevel ManagersInitializerLogLevel = LogLevel.Info; // Info pour voir l'initialisation

    // Méthodes publiques
    public static void Log(string message, string category = "General")
    {
        if (ShouldLog(LogLevel.Debug, category))
            Debug.Log($"[{category}] {message}");
    }

    public static void LogInfo(string message, string category = "General")
    {
        if (ShouldLog(LogLevel.Info, category))
            Debug.Log($"[{category}] {message}");
    }

    public static void LogWarning(string message, string category = "General")
    {
        if (ShouldLog(LogLevel.Warning, category))
            Debug.LogWarning($"[{category}] {message}");
    }

    public static void LogError(string message, string category = "General")
    {
        if (ShouldLog(LogLevel.Error, category))
            Debug.LogError($"[{category}] {message}");
    }

    private static bool ShouldLog(LogLevel messageLevel, string category)
    {
        LogLevel categoryLevel = GetCategoryLogLevel(category);
        return (int)messageLevel <= (int)categoryLevel;
    }

    private static LogLevel GetCategoryLogLevel(string category)
    {
        switch (category.ToLower())
        {
            case "gamemanager":
            case "game": return GameManagerLogLevel;
            
            case "mapmanager":
            case "map": return MapManagerLogLevel;
            
            case "levelmanager":
            case "level": return LevelManagerLogLevel;
            
            case "dialogue":
            case "dialogueplayer":
            case "subtitle": return DialogueLogLevel;
            
            case "calculator":
            case "calculatorgamemanager": return CalculatorLogLevel;
            
            case "loading":
            case "loadingmanager": return LoadingLogLevel;
            
            case "ui":
            case "menu":
            case "mainmenu": return UILogLevel;
            
            case "gameconfigloader":
            case "configloader": return GameConfigLoaderLogLevel;
            
            case "gamelauncher":
            case "launcher": return GameLauncherLogLevel;
            
            case "levelcard":
            case "card": return LevelCardLogLevel;
            
            case "managersinitializer":
            case "initializer": return ManagersInitializerLogLevel;
            
            default: return GlobalLogLevel;
        }
    }

    // Méthode pour activer le mode debug complet (utile pour debug)
    public static void EnableDebugMode()
    {
        GlobalLogLevel = LogLevel.Debug;
        GameManagerLogLevel = LogLevel.Debug;
        MapManagerLogLevel = LogLevel.Debug;
        LevelManagerLogLevel = LogLevel.Debug;
        DialogueLogLevel = LogLevel.Debug;
        CalculatorLogLevel = LogLevel.Debug;
        LoadingLogLevel = LogLevel.Debug;
        UILogLevel = LogLevel.Debug;
        GameConfigLoaderLogLevel = LogLevel.Debug;
        GameLauncherLogLevel = LogLevel.Debug;
        LevelCardLogLevel = LogLevel.Debug;
        ManagersInitializerLogLevel = LogLevel.Debug;
        Debug.Log("=== MODE DEBUG ACTIVÉ ===");
    }

    // Méthode pour désactiver tous les logs sauf erreurs
    public static void SetSilentMode()
    {
        GlobalLogLevel = LogLevel.Error;
        GameManagerLogLevel = LogLevel.Error;
        MapManagerLogLevel = LogLevel.Error;
        LevelManagerLogLevel = LogLevel.Error;
        DialogueLogLevel = LogLevel.Error;
        CalculatorLogLevel = LogLevel.Error;
        LoadingLogLevel = LogLevel.Error;
        UILogLevel = LogLevel.Error;
        GameConfigLoaderLogLevel = LogLevel.Error;
        GameLauncherLogLevel = LogLevel.Error;
        LevelCardLogLevel = LogLevel.Error;
        ManagersInitializerLogLevel = LogLevel.Error;
        Debug.Log("=== MODE SILENCIEUX ACTIVÉ (Erreurs seulement) ===");
    }
}

