using UnityEngine;

public class TargetZone : MonoBehaviour
{
    private AnswerWithZone answer;
    private TargetZoneData zoneData;
    // Référence de résolution pour la conversion JSON -> écran (ex: 1920x1080)
    private static int referenceWidth = 1920;
    private static int referenceHeight = 1080;

    public static void SetReferenceResolution(int width, int height)
    {
        if (width > 0) referenceWidth = width;
        if (height > 0) referenceHeight = height;
    }
    
    public void Initialize(AnswerWithZone answerData)
    {
        answer = answerData;
        zoneData = new TargetZoneData
        {
            id = answerData.zoneId,
            x = answerData.zone.x,
            y = answerData.zone.y,
            width = answerData.zone.width,
            height = answerData.zone.height
        };
        
        Debug.Log($"TargetZone initialisée: {answer.text} - Zone: ({zoneData.x},{zoneData.y},{zoneData.width},{zoneData.height})");
    }
    
    // Méthode principale pour vérifier si un point écran est dans cette zone
    public bool IsPointInScreenZone(Vector2 screenPoint)
    {
        if (zoneData == null) return false;
        
        // DEBUG MAC : Logs verbeux pour diagnostiquer le problème de zones
        bool isMac = Application.platform == RuntimePlatform.OSXEditor || Application.platform == RuntimePlatform.OSXPlayer;
        
        if (isMac)
        {
            Debug.Log($"[TargetZone/MAC] 🍎 Vérification zone '{answer?.text}' pour point écran {screenPoint}");
            Debug.Log($"[TargetZone/MAC] Écran: {Screen.width}x{Screen.height}, Référence: {referenceWidth}x{referenceHeight}");
        }
        
        // CORRECTION PLEIN ÉCRAN : Prendre en compte les letterbars/pillarboxes
        // Calculer le ratio d'aspect actuel vs référence
        float currentAspect = (float)Screen.width / Screen.height;
        float referenceAspect = (float)referenceWidth / referenceHeight;
        
        float scaleX, scaleY;
        float offsetX = 0, offsetY = 0;
        
        if (currentAspect > referenceAspect)
        {
            // Écran plus large que la référence → pillarboxes (barres verticales)
            scaleY = (float)Screen.height / referenceHeight;
            scaleX = scaleY; // Même échelle pour préserver l'aspect ratio
            
            // Calculer l'offset horizontal des barres noires
            float scaledWidth = referenceWidth * scaleX;
            offsetX = (Screen.width - scaledWidth) / 2f;
            
            if (isMac) Debug.Log($"[TargetZone/MAC] Mode: Pillarboxes (écran large), offsetX={offsetX:F1}, scale={scaleX:F3}");
        }
        else if (currentAspect < referenceAspect)
        {
            // Écran plus haut que la référence → letterboxes (barres horizontales)
            scaleX = (float)Screen.width / referenceWidth;
            scaleY = scaleX; // Même échelle pour préserver l'aspect ratio
            
            // Calculer l'offset vertical des barres noires
            float scaledHeight = referenceHeight * scaleY;
            offsetY = (Screen.height - scaledHeight) / 2f;
            
            if (isMac) Debug.Log($"[TargetZone/MAC] Mode: Letterboxes (écran haut), offsetY={offsetY:F1}, scale={scaleX:F3}");
        }
        else
        {
            // Aspect ratio identique → pas de barres
            scaleX = (float)Screen.width / referenceWidth;
            scaleY = (float)Screen.height / referenceHeight;
            
            if (isMac) Debug.Log($"[TargetZone/MAC] Mode: Ratio identique, scaleX={scaleX:F3}, scaleY={scaleY:F3}");
        }
        
        // Conversion des coordonnées écran vers l'espace JSON (origine haut-gauche)
        // En retirant les offsets des barres noires
        float jsonX = (screenPoint.x - offsetX) / scaleX;
        float jsonY = ((Screen.height - screenPoint.y) - offsetY) / scaleY;

        if (isMac)
        {
            Debug.Log($"[TargetZone/MAC] Conversion: Screen({screenPoint.x:F1},{screenPoint.y:F1}) -> JSON({jsonX:F1},{jsonY:F1})");
            Debug.Log($"[TargetZone/MAC] Zone '{answer?.text}': x={zoneData.x}-{zoneData.x+zoneData.width}, y={zoneData.y}-{zoneData.y+zoneData.height}");
        }

        // Vérifier si le point est dans les limites de la zone
        bool inZone = jsonX >= zoneData.x && 
                     jsonX <= (zoneData.x + zoneData.width) &&
                     jsonY >= zoneData.y && 
                     jsonY <= (zoneData.y + zoneData.height);
        
        if (isMac)
        {
            Debug.Log($"[TargetZone/MAC] Résultat: {(inZone ? "✅ DANS LA ZONE" : "❌ HORS ZONE")}");
        }
        
        if (inZone && !isMac)
        {
            //Debug.Log($"Clic/Survol dans zone {answer.text}: Screen({screenPoint.x},{screenPoint.y}) -> JSON({jsonX},{jsonY}) [Scale: {scaleX:F2}x{scaleY:F2}, Offset: {offsetX:F0}x{offsetY:F0}]");
        }
        
        return inZone;
    }
    
    public string GetAnswerText()
    {
        return answer?.text ?? "Réponse inconnue";
    }
    
    public bool IsCorrectAnswer()
    {
        return answer?.isCorrect ?? false;
    }
    
    public AnswerWithZone GetAnswer()
    {
        return answer;
    }
    
    // Méthode pour débugger les limites de la zone
    public void DebugZoneLimits()
    {
        if (answer != null && zoneData != null)
        {
            Debug.Log($"Zone '{answer.text}' limites JSON: X[{zoneData.x} à {zoneData.x + zoneData.width}] Y[{zoneData.y} à {zoneData.y + zoneData.height}]");
            Debug.Log($"Zone '{answer.text}' centre: ({zoneData.x + zoneData.width/2f}, {zoneData.y + zoneData.height/2f})");
        }
        else
        {
            Debug.LogWarning("DebugZoneLimits: answer ou zoneData est null");
        }
    }
}