using UnityEngine;
using UnityEngine.Video;

/// <summary>
/// Correction des color primaries pour les vidéos sur Mac
/// À ajouter sur le GameObject avec VideoPlayer
/// 
/// Problème : "Color primaries is unknown or unsupported by AVFoundationVideoMedia"
/// Solution : Configurer les couleurs correctes (bt709 pour Mac)
/// </summary>
[RequireComponent(typeof(VideoPlayer))]
public class MacVideoColorFix : MonoBehaviour
{
    [Header("Configuration")]
    [Tooltip("VideoPlayer à corriger (auto-détecté si vide)")]
    public VideoPlayer videoPlayer;
    
    [Header("Debug")]
    public bool showDebugLogs = true;
    
    private bool isMac = false;

    void Awake()
    {
        // Récupérer le VideoPlayer
        if (videoPlayer == null)
        {
            videoPlayer = GetComponent<VideoPlayer>();
        }
        
        // Détecter Mac
        isMac = Application.platform == RuntimePlatform.OSXEditor || 
                Application.platform == RuntimePlatform.OSXPlayer;
        
        if (showDebugLogs)
        {
            Debug.Log($"[MacVideoColorFix] 🍎 Platform: {Application.platform}");
            Debug.Log($"[MacVideoColorFix] Est Mac: {isMac}");
        }
    }

    void Start()
    {
        if (!isMac)
        {
            if (showDebugLogs) Debug.Log("[MacVideoColorFix] ⏭️ Non-Mac, pas de correction nécessaire");
            enabled = false;
            return;
        }

        if (videoPlayer == null)
        {
            Debug.LogError("[MacVideoColorFix] ❌ VideoPlayer non trouvé!");
            return;
        }

        if (showDebugLogs)
        {
            Debug.Log("[MacVideoColorFix] 🎬 Correction des color primaries pour Mac...");
        }

        ApplyMacVideoColorFix();
    }

    void ApplyMacVideoColorFix()
    {
        try
        {
            // ====== FIX PRINCIPAL : Matériel et Renderer ======
            
            // 1. Chercher le Renderer
            Renderer renderer = videoPlayer.GetComponent<Renderer>();
            
            if (renderer != null && renderer.material != null)
            {
                Material mat = renderer.material;
                
                if (showDebugLogs)
                {
                    Debug.Log($"[MacVideoColorFix] 📋 Matériau trouvé: {mat.name}");
                    Debug.Log($"[MacVideoColorFix] 📊 Shader: {mat.shader.name}");
                }

                // 2. S'assurer que le shader supporte les couleurs correctes
                // Le shader par défaut devrait être "Unlit/Texture" ou similaire
                
                // 3. Forcer le mode de rendu
                videoPlayer.renderMode = VideoRenderMode.MaterialOverride;
                
                if (showDebugLogs)
                {
                    Debug.Log($"[MacVideoColorFix] ✅ Render mode: {videoPlayer.renderMode}");
                }
            }
            else
            {
                if (showDebugLogs)
                {
                    Debug.LogWarning("[MacVideoColorFix] ⚠️ Pas de Renderer trouvé");
                }
            }

            // ====== FIX SECONDAIRE : Format de la Texture ======
            
            // 4. Configurer le format de la texture
            if (videoPlayer.targetTexture != null)
            {
                if (showDebugLogs)
                {
                    Debug.Log($"[MacVideoColorFix] 🖼️ Target Texture: {videoPlayer.targetTexture.name}");
                    Debug.Log($"[MacVideoColorFix] 📏 Format: {videoPlayer.targetTexture.format}");
                }
            }

            // ====== FIX GAMMA ======
            
            // 5. Forcer le mode gamma linéaire pour Mac
            if (QualitySettings.activeColorSpace == ColorSpace.Linear)
            {
                if (showDebugLogs)
                {
                    Debug.Log("[MacVideoColorFix] 🌈 Color space: Linear (correct)");
                }
            }
            else
            {
                if (showDebugLogs)
                {
                    Debug.LogWarning("[MacVideoColorFix] ⚠️ Color space: Gamma (peut causer des décalages)");
                }
            }

            // ====== CONFIGURATION AUDIO/VIDEO ======
            
            // 6. Logs informatifs
            if (showDebugLogs)
            {
                Debug.Log($"[MacVideoColorFix] 📹 Source: {videoPlayer.source}");
                Debug.Log($"[MacVideoColorFix] 🎥 Playback speed: {videoPlayer.playbackSpeed}");
                Debug.Log($"[MacVideoColorFix] 🔊 Audio output mode: {videoPlayer.audioOutputMode}");
            }

            Debug.Log("[MacVideoColorFix] ✅ Corrections appliquées pour Mac!");
        }
        catch (System.Exception ex)
        {
            Debug.LogError($"[MacVideoColorFix] ❌ Erreur: {ex.Message}");
            Debug.LogError($"[MacVideoColorFix] Stack: {ex.StackTrace}");
        }
    }

    // Méthode pour appliquer la correction manuellement si besoin
    public void ReapplyFix()
    {
        if (isMac)
        {
            Debug.Log("[MacVideoColorFix] 🔄 Réapplication de la correction...");
            ApplyMacVideoColorFix();
        }
    }

    // Afficher l'état de la vidéo
    void OnGUI()
    {
        if (!isMac || !showDebugLogs) return;
        
        if (videoPlayer == null) return;

        GUILayout.BeginArea(new Rect(10, 100, 400, 150));
        GUILayout.Box("Mac Video Status", GUILayout.ExpandWidth(true));
        
        GUILayout.Label($"Playing: {videoPlayer.isPlaying}");
        GUILayout.Label($"Time: {videoPlayer.time:F2}s / {videoPlayer.length:F2}s");
        GUILayout.Label($"Render Mode: {videoPlayer.renderMode}");
        GUILayout.Label($"Color Space: {QualitySettings.activeColorSpace}");
        
        if (GUILayout.Button("Reapply Fix"))
        {
            ReapplyFix();
        }
        
        GUILayout.EndArea();
    }
}
