using UnityEngine;
using UnityEngine.Video;
using UnityEngine.UI;

/// <summary>
/// Diagnostic ultra-détaillé pour Mac video color fix
/// À ajouter sur n'importe quel GameObject de la scène Player
/// </summary>
public class MacVideoColorFixDiagnostic : MonoBehaviour
{
    void Start()
    {
        Debug.Log("═══════════════════════════════════════════════════════════");
        Debug.Log("[DIAGNOSTIC] 🔍 VÉRIFICATION MAC VIDEO COLOR FIX");
        Debug.Log("═══════════════════════════════════════════════════════════");
        
        // ÉTAPE 1 : Platform
        Debug.Log($"✓ Platform: {Application.platform}");
        bool isMac = Application.platform == RuntimePlatform.OSXEditor || 
                     Application.platform == RuntimePlatform.OSXPlayer;
        Debug.Log($"✓ Est Mac: {isMac}");
        
        // ÉTAPE 2 : Chercher TOUS les VideoPlayer dans la scène
        Debug.Log("\n📺 RECHERCHE DE VIDEOPLAYER...");
        VideoPlayer[] allPlayers = FindObjectsByType<VideoPlayer>(FindObjectsSortMode.None);
        Debug.Log($"✓ Nombre de VideoPlayer trouvés: {allPlayers.Length}");
        
        if (allPlayers.Length == 0)
        {
            Debug.LogError("❌ AUCUN VideoPlayer trouvé dans la scène!");
            Debug.LogError("❌ Vous devez ajouter un VideoPlayer à la scène!");
            return;
        }
        
        // ÉTAPE 3 : Analyser chaque VideoPlayer
        for (int i = 0; i < allPlayers.Length; i++)
        {
            VideoPlayer vp = allPlayers[i];
            Debug.Log($"\n📺 VideoPlayer #{i}:");
            Debug.Log($"  ├─ GameObject: {vp.gameObject.name}");
            Debug.Log($"  ├─ Active: {vp.isPlaying}");
            Debug.Log($"  ├─ Source: {vp.source}");
            Debug.Log($"  ├─ Render Mode: {vp.renderMode}");
            Debug.Log($"  ├─ Target Texture: {(vp.targetTexture != null ? vp.targetTexture.name : "NULL")}");
            
            // Chercher Renderer
            Renderer renderer = vp.GetComponent<Renderer>();
            if (renderer != null)
            {
                Debug.Log($"  ├─ Renderer: ✓ Trouvé");
                Debug.Log($"  │  ├─ Material: {renderer.material.name}");
                Debug.Log($"  │  └─ Shader: {renderer.material.shader.name}");
            }
            else
            {
                Debug.Log($"  ├─ Renderer: ❌ PAS TROUVÉ");
            }
            
            // Chercher Canvas
            Canvas canvas = vp.GetComponent<Canvas>();
            if (canvas != null)
            {
                Debug.Log($"  ├─ Canvas: ✓ Trouvé (Sorting Order: {canvas.sortingOrder})");
            }
            
            // Chercher RawImage (si dans UI)
            RawImage rawImage = vp.GetComponent<RawImage>();
            if (rawImage != null)
            {
                Debug.Log($"  ├─ RawImage: ✓ Trouvé");
                Debug.Log($"  │  └─ Texture: {(rawImage.texture != null ? rawImage.texture.name : "NULL")}");
            }
            
            Debug.Log($"  └─ URL: {(vp.source == VideoSource.Url ? vp.url : "N/A")}");
        }
        
        // ÉTAPE 4 : Chercher les scripts MacVideoColorFix
        Debug.Log("\n🔧 RECHERCHE DE SCRIPTS...");
        MacVideoColorFix[] fixScripts = FindObjectsByType<MacVideoColorFix>(FindObjectsSortMode.None);
        Debug.Log($"✓ Scripts MacVideoColorFix trouvés: {fixScripts.Length}");
        
        for (int i = 0; i < fixScripts.Length; i++)
        {
            MacVideoColorFix fix = fixScripts[i];
            Debug.Log($"  ├─ Script #{i}: {fix.gameObject.name}");
            Debug.Log($"  │  └─ VideoPlayer assigné: {(fix.GetComponent<VideoPlayer>() != null ? "✓ OUI" : "❌ NON")}");
        }
        
        // ÉTAPE 5 : Configuration système
        Debug.Log("\n⚙️ CONFIGURATION SYSTÈME...");
        Debug.Log($"✓ Color Space: {QualitySettings.activeColorSpace}");
        Debug.Log($"✓ Graphics API: {SystemInfo.graphicsDeviceType}");
        Debug.Log($"✓ GPU: {SystemInfo.graphicsDeviceName}");
        
        // ÉTAPE 6 : Résumé
        Debug.Log("\n" + "═══════════════════════════════════════════════════════════");
        Debug.Log("[DIAGNOSTIC] 📋 RÉSUMÉ");
        Debug.Log("═══════════════════════════════════════════════════════════");
        
        if (allPlayers.Length == 0)
        {
            Debug.LogError("❌ PROBLÈME: Aucun VideoPlayer trouvé!");
            Debug.LogError("❌ SOLUTION: Créer un VideoPlayer ou attendre qu'un dialogue le crée");
        }
        else if (fixScripts.Length == 0)
        {
            Debug.LogError("❌ PROBLÈME: Aucun script MacVideoColorFix trouvé!");
            Debug.LogError("❌ SOLUTION: Ajouter le script à un GameObject persistant");
        }
        else
        {
            Debug.Log("✅ Configuration semble correcte");
            Debug.Log("✅ Vérifier les logs du gameplay pour [MacVideoColorFix]");
        }
        
        Debug.Log("═══════════════════════════════════════════════════════════\n");
    }
}
