using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Video;
using TMPro;
using System.Collections;

public class MapPopup : MonoBehaviour
{
    private RectTransform rectTransform;
    private CanvasGroup canvasGroup;
    private VideoPlayer videoPlayer;
    private RenderTexture renderTexture;
    private RawImage videoDisplay;
    private TextMeshProUGUI popupText;
    private Button clickButton;
    private Image circularMask;
    private GameObject videoContainer;

    private MapZone currentZone;
    private MapManager mapManager;
    private MapPopupConfig config;

    public void Initialize(MapPopupConfig popupConfig, MapManager manager)
    {
        rectTransform = GetComponent<RectTransform>();
        canvasGroup = GetComponent<CanvasGroup>();
        config = popupConfig;
        mapManager = manager;
        canvasGroup.alpha = 0f;
        // NE PAS désactiver ici
    }

    public void SetupPopupContent(string videoUrl, string text, string levelId, bool clickable, string buttonText)
    {
        // Créer un MapZone temporaire
        currentZone = new MapZone
        {
            popupVideo = videoUrl,
            popupText = text,
            targetLevelId = levelId,
            popupClickable = clickable,
            buttonText = buttonText
        };

        // Créer le contenu directement (pas d'appel récursif)
        foreach (Transform child in transform)
        {
            Destroy(child.gameObject);
        }

        CreateCircularBackground();
        CreateVideoPlayer();
        CreateTextArea();

        if (currentZone.popupClickable)
        {
            CreatePlayButton();
        }

        else
        {
            Debug.Log($"[SetupPopupContent] Zone NON clickable, pas de bouton");
        }



    }

    public void Show()
    {
        if (currentZone == null) return;

        // Ne PAS réactiver ici, déjà actif
        StartCoroutine(ShowPopupCoroutine(Vector2.zero));
    }

    IEnumerator ShowPopupCoroutine(Vector2 screenPosition)
    {
        yield return new WaitForSeconds(config.showDelay);

        float elapsed = 0f;
        while (elapsed < config.fadeSpeed)
        {
            elapsed += Time.deltaTime;
            canvasGroup.alpha = Mathf.Lerp(0f, 1f, elapsed / config.fadeSpeed);
            yield return null;
        }
        canvasGroup.alpha = 1f;
    }





    void CreateCircularBackground()
    {
        GameObject bgObj = new GameObject("CircularBackground");
        bgObj.transform.SetParent(transform, false);

        RectTransform bgRect = bgObj.AddComponent<RectTransform>();
        bgRect.anchorMin = Vector2.zero;
        bgRect.anchorMax = Vector2.one;
        bgRect.offsetMin = Vector2.zero;
        bgRect.offsetMax = Vector2.zero;

        circularMask = bgObj.AddComponent<Image>();

        Texture2D circleTexture = CreateCircleTexture(512);
        Sprite circleSprite = Sprite.Create(circleTexture,
            new Rect(0, 0, circleTexture.width, circleTexture.height),
            new Vector2(0.5f, 0.5f));

        circularMask.sprite = circleSprite;
        circularMask.raycastTarget = false;

        if (ColorUtility.TryParseHtmlString(config.backgroundColor, out Color bgColor))
        {
            circularMask.color = bgColor;
        }

        Outline outline = bgObj.AddComponent<Outline>();
        if (ColorUtility.TryParseHtmlString(config.borderColor, out Color borderColor))
        {
            outline.effectColor = borderColor;
        }
        outline.effectDistance = new Vector2(config.borderWidth, -config.borderWidth);
    }

    Texture2D CreateCircleTexture(int size)
    {
        Texture2D texture = new Texture2D(size, size, TextureFormat.RGBA32, false);
        Color[] pixels = new Color[size * size];

        float center = size / 2f;
        float radius = size / 2f;

        for (int y = 0; y < size; y++)
        {
            for (int x = 0; x < size; x++)
            {
                float distance = Vector2.Distance(new Vector2(x, y), new Vector2(center, center));

                if (distance <= radius)
                {
                    float alpha = 1f - Mathf.Clamp01((distance - radius + 2f) / 2f);
                    pixels[y * size + x] = new Color(1f, 1f, 1f, alpha);
                }
                else
                {
                    pixels[y * size + x] = Color.clear;
                }
            }
        }

        texture.SetPixels(pixels);
        texture.Apply();
        texture.filterMode = FilterMode.Bilinear;

        return texture;
    }

    void CreateTextArea()
    {
        if (string.IsNullOrEmpty(currentZone.popupText))
        {
            return;
        }

        GameObject maskContainer = new GameObject("TextMaskContainer");
        maskContainer.transform.SetParent(transform, false);

        RectTransform maskRect = maskContainer.AddComponent<RectTransform>();
        maskRect.anchorMin = Vector2.zero;
        maskRect.anchorMax = Vector2.one;
        maskRect.offsetMin = Vector2.zero;
        maskRect.offsetMax = Vector2.zero;

        Mask mask = maskContainer.AddComponent<Mask>();
        mask.showMaskGraphic = false;

        Image maskImage = maskContainer.AddComponent<Image>();
        Texture2D circleTex = CreateCircleTexture(512);
        Sprite circleSprite = Sprite.Create(circleTex,
            new Rect(0, 0, circleTex.width, circleTex.height),
            new Vector2(0.5f, 0.5f));
        maskImage.sprite = circleSprite;
        maskImage.color = Color.white;

        GameObject textContainer = new GameObject("TextContainer");
        textContainer.transform.SetParent(maskContainer.transform, false);

        RectTransform textRect = textContainer.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.zero;
        textRect.pivot = new Vector2(0.5f, 1f);

        float popupSize = rectTransform.sizeDelta.x;
        float textWidth = popupSize * 0.9f;

        // NOUVEAU : Utiliser textHeight depuis config
        textRect.sizeDelta = new Vector2(textWidth, config.textHeight);
        textRect.anchoredPosition = new Vector2(popupSize * 0.5f, popupSize - 10f);

        Image textBg = textContainer.AddComponent<Image>();
        if (ColorUtility.TryParseHtmlString(config.textBackgroundColor, out Color textBgColor))
        {
            textBg.color = textBgColor;
        }

        GameObject textObj = new GameObject("PopupText");
        textObj.transform.SetParent(textContainer.transform, false);

        RectTransform textObjRect = textObj.AddComponent<RectTransform>();
        textObjRect.anchorMin = Vector2.zero;
        textObjRect.anchorMax = Vector2.one;

        // NOUVEAU : Padding configurable
        textObjRect.offsetMin = new Vector2(config.textPaddingLeft, config.textPaddingBottom);
        textObjRect.offsetMax = new Vector2(-config.textPaddingRight, -config.textPaddingTop);

        popupText = textObj.AddComponent<TextMeshProUGUI>();
        popupText.text = currentZone.popupText;
        popupText.fontSize = config.textFontSize;
        popupText.alignment = TextAlignmentOptions.Center;
        popupText.textWrappingMode = TMPro.TextWrappingModes.Normal;

        if (ColorUtility.TryParseHtmlString(config.textColor, out Color textColor))
        {
            popupText.color = textColor;
        }
    }

    void CreateVideoPlayer()
    {
        videoContainer = new GameObject("VideoContainer");
        videoContainer.transform.SetParent(transform, false);

        RectTransform videoRect = videoContainer.AddComponent<RectTransform>();
        videoRect.anchorMin = new Vector2(0.5f, 0.5f);
        videoRect.anchorMax = new Vector2(0.5f, 0.5f);
        videoRect.pivot = new Vector2(0.5f, 0.5f);

        float videoSize = rectTransform.sizeDelta.x * 1f;
        videoRect.sizeDelta = new Vector2(videoSize, videoSize);

        float yOffset = rectTransform.sizeDelta.y * 0f;
        videoRect.anchoredPosition = new Vector2(0f, yOffset);

        Mask videoMask = videoContainer.AddComponent<Mask>();
        videoMask.showMaskGraphic = false;

        Image maskImage = videoContainer.AddComponent<Image>();
        Texture2D circleTex = CreateCircleTexture(512);
        Sprite circleSprite = Sprite.Create(circleTex,
            new Rect(0, 0, circleTex.width, circleTex.height),
            new Vector2(0.5f, 0.5f));
        maskImage.sprite = circleSprite;
        maskImage.raycastTarget = false;
        maskImage.color = Color.white;

        int rtSize = 1024;
        renderTexture = new RenderTexture(rtSize, rtSize, 0);
        renderTexture.Create();

        GameObject videoPlayerObj = new GameObject("VideoPlayer");
        videoPlayerObj.transform.SetParent(videoContainer.transform, false);

        videoPlayer = videoPlayerObj.AddComponent<VideoPlayer>();
        videoPlayer.playOnAwake = false;
        videoPlayer.renderMode = VideoRenderMode.RenderTexture;
        videoPlayer.targetTexture = renderTexture;
        videoPlayer.isLooping = true;
        videoPlayer.aspectRatio = VideoAspectRatio.FitInside;
        videoPlayer.audioOutputMode = VideoAudioOutputMode.None;
        videoPlayer.url = currentZone.popupVideo;
        videoPlayer.Prepare();
        videoPlayer.prepareCompleted += OnVideoPrepared;

        GameObject rawImageObj = new GameObject("VideoDisplay");
        rawImageObj.transform.SetParent(videoContainer.transform, false);

        RectTransform rawImageRect = rawImageObj.AddComponent<RectTransform>();
        rawImageRect.anchorMin = new Vector2(0.5f, 0.5f);
        rawImageRect.anchorMax = new Vector2(0.5f, 0.5f);
        rawImageRect.pivot = new Vector2(0.5f, 0.5f);

        float displaySize = videoSize * 1.0f;
        rawImageRect.sizeDelta = new Vector2(displaySize, displaySize);
        rawImageRect.anchoredPosition = Vector2.zero;

        videoDisplay = rawImageObj.AddComponent<RawImage>();
        videoDisplay.texture = renderTexture;
    }

    void OnVideoPrepared(VideoPlayer source)
    {
        videoPlayer.Play();

        if (videoDisplay != null)
        {
            RectTransform rt = videoDisplay.GetComponent<RectTransform>();
            float size = rt.sizeDelta.x;
            rt.sizeDelta = new Vector2(size, size);
        }
    }

    void CreatePlayButton()
    {
        GameObject buttonObj = new GameObject("PlayButton");
        buttonObj.transform.SetParent(transform, false);

        RectTransform buttonRect = buttonObj.AddComponent<RectTransform>();
        buttonRect.anchorMin = new Vector2(0.5f, 0f);
        buttonRect.anchorMax = new Vector2(0.5f, 0f);
        buttonRect.pivot = new Vector2(0.5f, 0f);
        buttonRect.sizeDelta = new Vector2(config.buttonWidth, config.buttonHeight);
        buttonRect.anchoredPosition = new Vector2(0, config.buttonBottomMargin);

        Image buttonImage = buttonObj.AddComponent<Image>();
        if (ColorUtility.TryParseHtmlString(config.buttonBackgroundColor, out Color btnBgColor))
        {
            buttonImage.color = btnBgColor;
        }

        Button button = buttonObj.AddComponent<Button>();
        button.targetGraphic = buttonImage;
        button.onClick.AddListener(() =>
        {
            OnPopupClick();
        });

        GameObject textObj = new GameObject("ButtonText");
        textObj.transform.SetParent(buttonObj.transform, false);

        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.offsetMin = Vector2.zero;
        textRect.offsetMax = Vector2.zero;

        TextMeshProUGUI buttonText = textObj.AddComponent<TextMeshProUGUI>();
        Debug.Log($"[CreatePlayButton] buttonText zone: '{currentZone.buttonText}', config: '{config.buttonText}'");

        string textToDisplay = !string.IsNullOrEmpty(currentZone.buttonText) ? currentZone.buttonText : config.buttonText;
        buttonText.text = textToDisplay;

        Debug.Log($"[CreatePlayButton] Texte affiché: '{textToDisplay}'");

        buttonText.text = textToDisplay;
        buttonText.fontSize = config.buttonFontSize;
        buttonText.alignment = TextAlignmentOptions.Center;
        buttonText.color = Color.white;

        if (ColorUtility.TryParseHtmlString(config.buttonTextColor, out Color textColor))
        {
            buttonText.color = textColor;
        }
    }
    void OnPopupClick()
    {
        if (mapManager != null && currentZone != null)
        {
            Destroy(gameObject);
            mapManager.LaunchLevel(currentZone.targetLevelId);
        }
    }

    public void Hide()
    {
        if (!gameObject.activeInHierarchy)
        {
            return;
        }

        StartCoroutine(HidePopupCoroutine());
    }

    IEnumerator HidePopupCoroutine()
    {
        float elapsed = 0f;
        while (elapsed < config.fadeSpeed)
        {
            elapsed += Time.deltaTime;
            canvasGroup.alpha = Mathf.Lerp(1f, 0f, elapsed / config.fadeSpeed);
            yield return null;
        }
        canvasGroup.alpha = 0f;

        if (videoPlayer != null)
        {
            videoPlayer.Stop();
        }

        gameObject.SetActive(false);
    }

    void OnDestroy()
    {
        if (renderTexture != null)
        {
            renderTexture.Release();
            Destroy(renderTexture);
        }
    }
}