using UnityEngine;
using UnityEngine.SceneManagement;
using System.Collections;

public class SceneTransitionManager : MonoBehaviour
{
    private static SceneTransitionManager instance;
    private static GameObject transitionOverlay;
    
    void Awake()
    {
        // Singleton pattern
        if (instance == null)
        {
            instance = this;
            DontDestroyOnLoad(gameObject);
        }
        else
        {
            Destroy(gameObject);
        }
    }
    
    public static void LoadSceneWithTransition(string sceneName)
    {
        Debug.Log($"[SceneTransitionManager] Demande de chargement de la scène: {sceneName}");
        if (instance != null)
        {
            Debug.Log($"[SceneTransitionManager] Instance trouvée, lancement transition vers {sceneName}");
            instance.StartCoroutine(instance.TransitionToScene(sceneName));
        }
        else
        {
            Debug.LogError("[SceneTransitionManager] Instance introuvable !");
        }
    }
    
    IEnumerator TransitionToScene(string sceneName)
    {
        Debug.Log($"[SceneTransitionManager] Début transition vers {sceneName}");
        // Créer écran noir IMMÉDIATEMENT
        CreatePersistentOverlay();
        
        // Charger la scène de façon asynchrone
        Debug.Log($"[SceneTransitionManager] Chargement asynchrone de {sceneName}");
        AsyncOperation asyncLoad = SceneManager.LoadSceneAsync(sceneName);
        
        // Attendre que la scène soit complètement chargée
        while (!asyncLoad.isDone)
        {
            yield return null;
        }
        
        // Attendre encore un peu pour que tout soit initialisé
        yield return new WaitForSeconds(0.5f);
        
        // Faire disparaître l'overlay
        if (transitionOverlay != null)
        {
            StartCoroutine(FadeOutOverlay());
        }
    }
    
    void CreatePersistentOverlay()
    {
        if (transitionOverlay != null)
        {
            Destroy(transitionOverlay);
        }
        
        // Créer un overlay qui survit au changement de scène
        transitionOverlay = new GameObject("SceneTransitionOverlay");
        DontDestroyOnLoad(transitionOverlay);
        
        // Utiliser un SpriteRenderer pour être indépendant des Canvas
        SpriteRenderer sr = transitionOverlay.AddComponent<SpriteRenderer>();
        
        // Texture noire
        Texture2D blackTexture = new Texture2D(1, 1);
        blackTexture.SetPixel(0, 0, Color.black);
        blackTexture.Apply();
        
        sr.sprite = Sprite.Create(blackTexture, new Rect(0, 0, 1, 1), Vector2.one * 0.5f);
        sr.sortingOrder = 32767; // Valeur maximale
        
        // Couvrir tout l'écran
        Camera mainCam = Camera.main;
        if (mainCam != null)
        {
            float height = mainCam.orthographicSize * 2f;
            float width = height * mainCam.aspect;
            transitionOverlay.transform.localScale = new Vector3(width * 2, height * 2, 1);
            transitionOverlay.transform.position = new Vector3(0, 0, -5);
        }
        
        Debug.Log("Overlay de transition créé");
    }
    
    IEnumerator FadeOutOverlay()
    {
        SpriteRenderer sr = transitionOverlay.GetComponent<SpriteRenderer>();
        
        float fadeTime = 0.5f;
        float elapsedTime = 0f;
        
        while (elapsedTime < fadeTime)
        {
            elapsedTime += Time.deltaTime;
            float alpha = 1f - (elapsedTime / fadeTime);
            sr.color = new Color(0, 0, 0, alpha);
            yield return null;
        }
        
        Destroy(transitionOverlay);
        transitionOverlay = null;
    }
}