using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Networking;
using UnityEngine.SceneManagement;
using TMPro;
using System.Collections;
using System.Collections.Generic;

/// <summary>
/// Onglet Quêtes : Affiche la liste des maps disponibles sur la moitié gauche du panneau
/// </summary>
public class QuetesTab : SettingsTab
{
    [Header("Layout")]
    private RectTransform leftPanel;
    private RectTransform rightPanel;
    private ScrollRect scrollRect;
    private RectTransform contentContainer;
    
    [Header("Style")]
    private Color normalColor = new Color(0.92f, 0.89f, 0.85f, 1f);
    private Color hoverColor = new Color(0.82f, 0.67f, 0.89f, 1f); // Violet clair
    private Color selectedColor = new Color(0.70f, 0.50f, 0.85f, 1f); // Violet
    
    private List<MapInfo> maps = new List<MapInfo>();
    private int selectedMapIndex = -1;
    private List<GameObject> questItems = new List<GameObject>();
    
    [System.Serializable]
    private class MapsRegistry
    {
        public List<MapInfo> maps;
    }
    
    [System.Serializable]
    private class MapInfo
    {
        public string id;
        public string name;
        public string configUrl;
        public string description;
    }
    
    protected override void Awake()
    {
        base.Awake();
        tabName = "QUÊTES";
        tabOrder = 2; if (contentContainer == null) CreateLayout();
    }
    
    protected override void OnEnable()
    {
        // Appelé quand l'onglet devient actif
        Debug.Log("========================================");
        Debug.Log("[QuetesTab] ★ OnEnable appelé ★"); Debug.LogError("[QuetesTab] ⚠️ ONENABLE CALLED - CHECK CONSOLE!");
        Debug.Log($"[QuetesTab] contentContainer null ? {contentContainer == null}");
        Debug.Log($"[QuetesTab] Nombre de maps : {maps.Count}");
        Debug.Log($"[QuetesTab] Nombre d'items : {questItems.Count}");
        Debug.Log("========================================");
        
        if (contentContainer == null)
        {
            Debug.Log("[QuetesTab] → Création du layout (première fois)");
            CreateLayout();
        }
        
        // TOUJOURS recharger les maps si la liste est vide (pour déboguer)
        if (questItems.Count == 0)
        {
            Debug.Log("[QuetesTab] → Liste vide, chargement des maps...");
            StartCoroutine(LoadMapsRegistry());
        }
        else
        {
            Debug.Log($"[QuetesTab] → Liste déjà remplie avec {questItems.Count} items");
            
            // CORRECTION : Vérifier et corriger chaque item pour forcer la visibilité
            StartCoroutine(DiagnoseAndFixItemsDelayed());
        }
    }
    
    private void CreateLayout()
    {
        Debug.Log("[QuetesTab] CreateLayout() début");
        
        CanvasGroup cg = gameObject.AddComponent<CanvasGroup>();
        cg.blocksRaycasts = true;
        cg.interactable = true;
        Debug.Log("[QuetesTab] CanvasGroup ajouté pour bloquer raycasts");
        
        // IMPORTANT : Ajouter une Image au QuetesTab lui-même pour bloquer TOUS les raycasts
        if (gameObject.GetComponent<Image>() == null)
        {
            Image tabBg = gameObject.AddComponent<Image>();
            tabBg.color = new Color(0f, 0f, 0f, 0.01f); // Presque transparent mais bloque les clics
            tabBg.raycastTarget = true;
            Debug.Log("[QuetesTab] Image de fond ajoutée au QuetesTab pour bloquer les raycasts");
        }
        
        // CORRECTION : Ajouter un Canvas au QuetesTab pour forcer l'ordre de rendu
        if (gameObject.GetComponent<Canvas>() == null)
        {
            Canvas tabCanvas = gameObject.AddComponent<Canvas>();
            tabCanvas.overrideSorting = true;
            tabCanvas.sortingOrder = 1; // Au-dessus du Canvas parent (qui est à 500)
            Debug.Log("[QuetesTab] Canvas ajouté avec sortingOrder=1 pour forcer le rendu au-dessus");
        }
        
        // Créer le panneau gauche (liste des quêtes)
        GameObject leftObj = new GameObject("LeftPanel");
        leftObj.transform.SetParent(transform, false);
        leftPanel = leftObj.AddComponent<RectTransform>();
        leftPanel.anchorMin = new Vector2(0, 0);
        leftPanel.anchorMax = new Vector2(0.48f, 1);
        leftPanel.offsetMin = new Vector2(20, 20);
        leftPanel.offsetMax = new Vector2(-10, -20);
        
        // Ajouter une image de fond BIEN VISIBLE avec raycastTarget = true
        Image leftBg = leftObj.AddComponent<Image>();
        leftBg.color = new Color(0.95f, 0.92f, 0.88f, 1f); // Beige clair, complètement opaque
        leftBg.raycastTarget = true; // IMPORTANT : Bloquer les raycasts
        
        // S'assurer que le panneau est actif
        leftObj.SetActive(true);
        Debug.Log($"[QuetesTab] LeftPanel créé et activé: {leftObj.activeInHierarchy}");
        
        // Créer le panneau droit (pour détails futurs)
        GameObject rightObj = new GameObject("RightPanel");
        rightObj.transform.SetParent(transform, false);
        rightPanel = rightObj.AddComponent<RectTransform>();
        rightPanel.anchorMin = new Vector2(0.52f, 0);
        rightPanel.anchorMax = new Vector2(1, 1);
        rightPanel.offsetMin = new Vector2(10, 20);
        rightPanel.offsetMax = new Vector2(-20, -20);
        
        // Ajouter une image de fond BIEN VISIBLE
        Image rightBg = rightObj.AddComponent<Image>();
        rightBg.color = new Color(0.88f, 0.90f, 0.92f, 1f); // Bleu-gris clair, complètement opaque
        rightBg.raycastTarget = true; // IMPORTANT : Bloquer les raycasts
        
        // Texte temporaire pour le panneau droit
        GameObject rightTextObj = new GameObject("InfoText");
        rightTextObj.transform.SetParent(rightPanel, false);
        RectTransform rightTextRect = rightTextObj.AddComponent<RectTransform>();
        rightTextRect.anchorMin = Vector2.zero;
        rightTextRect.anchorMax = Vector2.one;
        rightTextRect.offsetMin = new Vector2(20, 20);
        rightTextRect.offsetMax = new Vector2(-20, -20);
        
        TextMeshProUGUI rightText = rightTextObj.AddComponent<TextMeshProUGUI>();
        rightText.text = "PERSONNAGES\n\nLES FORMATIONS\n\nBADGE OBTENU";
        rightText.fontSize = 16;
        rightText.alignment = TextAlignmentOptions.Top;
        rightText.color = new Color(0.4f, 0.3f, 0.5f, 1f); // Complètement opaque
        
        // Créer le ScrollRect pour la liste de gauche
        GameObject scrollObj = new GameObject("ScrollView");
        scrollObj.transform.SetParent(leftPanel, false);
        
        RectTransform scrollRect_rt = scrollObj.AddComponent<RectTransform>();
        scrollRect_rt.anchorMin = Vector2.zero;
        scrollRect_rt.anchorMax = Vector2.one;
        scrollRect_rt.offsetMin = Vector2.zero;
        scrollRect_rt.offsetMax = Vector2.zero;
        
        scrollRect = scrollObj.AddComponent<ScrollRect>();
        scrollRect.horizontal = false;
        scrollRect.vertical = true;
        scrollRect.scrollSensitivity = 30f;
        
        // Viewport
        GameObject viewportObj = new GameObject("Viewport");
        viewportObj.transform.SetParent(scrollObj.transform, false);
        RectTransform viewportRect = viewportObj.AddComponent<RectTransform>();
        viewportRect.anchorMin = Vector2.zero;
        viewportRect.anchorMax = Vector2.one;
        viewportRect.offsetMin = Vector2.zero;
        viewportRect.offsetMax = Vector2.zero;
        Image viewportImage = viewportObj.AddComponent<Image>();
        viewportImage.color = new Color(1, 1, 1, 0); // Transparent
        viewportImage.raycastTarget = true; // IMPORTANT : Bloquer les raycasts
        viewportObj.AddComponent<Mask>().showMaskGraphic = false;
        scrollRect.viewport = viewportRect;
        
        // S'assurer que le viewport est actif
        viewportObj.SetActive(true);
        Debug.Log($"[QuetesTab] Viewport créé et activé: {viewportObj.activeInHierarchy}");
        
        // Content
        GameObject contentObj = new GameObject("Content");
        contentObj.transform.SetParent(viewportObj.transform, false);
        contentContainer = contentObj.AddComponent<RectTransform>();
        contentContainer.anchorMin = new Vector2(0, 1);
        contentContainer.anchorMax = new Vector2(1, 1);
        contentContainer.pivot = new Vector2(0.5f, 1);
        contentContainer.offsetMin = Vector2.zero;
        contentContainer.offsetMax = Vector2.zero;
        
        // Layout Group
        VerticalLayoutGroup vlg = contentObj.AddComponent<VerticalLayoutGroup>();
        vlg.spacing = 10f;
        vlg.padding = new RectOffset(10, 10, 10, 10);
        vlg.childControlWidth = true;
        vlg.childControlHeight = false;
        vlg.childForceExpandWidth = true;
        vlg.childForceExpandHeight = false;
        
        ContentSizeFitter csf = contentObj.AddComponent<ContentSizeFitter>();
        csf.verticalFit = ContentSizeFitter.FitMode.PreferredSize;
        
        scrollRect.content = contentContainer;
        
        // S'assurer que tout est actif
        scrollObj.SetActive(true);
        contentObj.SetActive(true);
        Debug.Log($"[QuetesTab] ScrollView actif: {scrollObj.activeInHierarchy}, Content actif: {contentObj.activeInHierarchy}");
        Debug.Log($"[QuetesTab] VerticalLayoutGroup présent: {vlg != null}, spacing: {vlg.spacing}");
        Debug.Log($"[QuetesTab] ContentSizeFitter présent: {csf != null}, verticalFit: {csf.verticalFit}");
        
        Debug.Log("[QuetesTab] CreateLayout() terminé - contentContainer créé");
        
        // Vérifier les tailles
        Debug.Log($"[QuetesTab] 📏 LeftPanel size: {leftPanel.rect.width} x {leftPanel.rect.height}");
        Debug.Log($"[QuetesTab] 📏 ContentContainer size: {contentContainer.rect.width} x {contentContainer.rect.height}");
        Debug.Log($"[QuetesTab] 📏 LeftPanel anchors: Min({leftPanel.anchorMin.x}, {leftPanel.anchorMin.y}) Max({leftPanel.anchorMax.x}, {leftPanel.anchorMax.y})");
    }
    
    private IEnumerator LoadMapsRegistry()
    {
        Debug.Log("[QuetesTab] ========== DÉBUT LoadMapsRegistry ==========");
        
        // Vérifier que GeneralConfigManager existe
        if (GeneralConfigManager.Instance == null)
        {
            Debug.LogError("[QuetesTab] ❌ GeneralConfigManager.Instance est NULL !");
            Debug.LogError("[QuetesTab] ❌ Vérifiez qu'il y a un GameObject avec GeneralConfigManager dans la scène !");
            yield break;
        }
        
        Debug.Log("[QuetesTab] ✅ GeneralConfigManager.Instance trouvé !");
        
        string registryUrl = GeneralConfigManager.Instance.GetMapsRegistryUrl();
        Debug.Log($"[QuetesTab] 📂 URL du registry: {registryUrl}");
        
        using (UnityWebRequest www = UnityWebRequest.Get(registryUrl))
        {
            Debug.Log("[QuetesTab] ⏳ Envoi de la requête...");
            yield return www.SendWebRequest();
            
            Debug.Log($"[QuetesTab] 📡 Requête terminée. Résultat: {www.result}");
            
            if (www.result != UnityWebRequest.Result.Success)
            {
                Debug.LogError($"[QuetesTab] ❌ Erreur chargement: {www.error}");
                Debug.LogError($"[QuetesTab] ❌ Code HTTP: {www.responseCode}");
                Debug.LogError($"[QuetesTab] ❌ URL: {registryUrl}");
                yield break;
            }
            
            Debug.Log("[QuetesTab] ✅ Requête réussie");
            Debug.Log($"[QuetesTab] 📄 Taille du JSON: {www.downloadHandler.text.Length} caractères");
            
            try
            {
                string jsonText = www.downloadHandler.text;
                Debug.Log($"[QuetesTab] 📝 Contenu JSON (100 premiers car): {jsonText.Substring(0, Mathf.Min(100, jsonText.Length))}...");
                
                MapsRegistry registry = JsonUtility.FromJson<MapsRegistry>(jsonText);
                
                if (registry == null)
                {
                    Debug.LogError("[QuetesTab] ❌ Registry est null après parsing !");
                    yield break;
                }
                
                if (registry.maps == null)
                {
                    Debug.LogError("[QuetesTab] ❌ registry.maps est null !");
                    yield break;
                }
                
                maps = registry.maps;
                Debug.Log($"[QuetesTab] ✅ {maps.Count} maps chargées avec succès");
                
                // Afficher les noms des maps
                for (int i = 0; i < Mathf.Min(3, maps.Count); i++)
                {
                    Debug.Log($"[QuetesTab]   → Map {i}: {maps[i].name}");
                }
                
                Debug.Log("[QuetesTab] 🏗️ Appel de PopulateQuestsList()...");
                PopulateQuestsList();
                Debug.Log("[QuetesTab] ========== FIN LoadMapsRegistry ==========");
            }
            catch (System.Exception e)
            {
                Debug.LogError($"[QuetesTab] ❌ Erreur parsing: {e.Message}");
                Debug.LogError($"[QuetesTab] ❌ Stack trace: {e.StackTrace}");
            }
        }
    }
    
    private void PopulateQuestsList()
    {
        Debug.Log($"[QuetesTab] PopulateQuestsList() - {maps.Count} maps à afficher");
        
        // Nettoyer la liste existante
        foreach (GameObject item in questItems)
        {
            Destroy(item);
        }
        questItems.Clear();
        
        // Créer un item pour chaque map
        for (int i = 0; i < maps.Count; i++)
        {
            int index = i; // Capture pour le closure
            MapInfo map = maps[i];
            
            Debug.Log($"[QuetesTab] Création item {i}: {map.name}");
            GameObject itemObj = CreateQuestItem(map, index);
            questItems.Add(itemObj);
        }
        
        Debug.Log($"[QuetesTab] PopulateQuestsList() terminé - {questItems.Count} items créés");
        
        // Diagnostic immédiat des items créés
        DiagnoseItems();
        
        // IMPORTANT : Lancer une coroutine pour forcer le recalcul après un frame
        StartCoroutine(ForceLayoutRebuildNextFrame());
    }
    
    private IEnumerator DiagnoseAndFixItemsDelayed()
    {
        // Attendre un frame pour que tous les éléments soient initialisés
        yield return null;
        DiagnoseAndFixItems();
    }
    
    private void DiagnoseAndFixItems()
    {
        Debug.Log($"[QuetesTab] 🔧 DIAGNOSTIC ET CORRECTION des {questItems.Count} items :");
        
        // Vérifier que le parent est actif
        if (leftPanel != null)
        {
            bool leftPanelActive = leftPanel.gameObject.activeInHierarchy;
            Debug.Log($"[QuetesTab] LeftPanel actif ? {leftPanelActive}");
            Debug.Log($"[QuetesTab] LeftPanel scale : {leftPanel.localScale}");
            Debug.Log($"[QuetesTab] LeftPanel size : {leftPanel.rect.width} x {leftPanel.rect.height}");
            
            if (!leftPanelActive)
            {
                Debug.LogError("[QuetesTab] ❌ LeftPanel est désactivé ! Activation...");
                leftPanel.gameObject.SetActive(true);
            }
            
            if (leftPanel.localScale == Vector3.zero)
            {
                Debug.LogError("[QuetesTab] ❌ LeftPanel a une scale de 0 ! Correction...");
                leftPanel.localScale = Vector3.one;
            }
        }
        
        // Vérifier le contentContainer
        if (contentContainer != null)
        {
            Debug.Log($"[QuetesTab] ContentContainer actif ? {contentContainer.gameObject.activeInHierarchy}");
            Debug.Log($"[QuetesTab] ContentContainer scale : {contentContainer.localScale}");
            Debug.Log($"[QuetesTab] ContentContainer size : {contentContainer.rect.width} x {contentContainer.rect.height}");
            Debug.Log($"[QuetesTab] ContentContainer position : {contentContainer.anchoredPosition}");
            Debug.Log($"[QuetesTab] ContentContainer enfants : {contentContainer.childCount}");
            
            if (!contentContainer.gameObject.activeInHierarchy)
            {
                Debug.LogError("[QuetesTab] ❌ ContentContainer désactivé ! Activation...");
                contentContainer.gameObject.SetActive(true);
            }
        }
        
        // Vérifier chaque item
        for (int i = 0; i < questItems.Count; i++)
        {
            GameObject item = questItems[i];
            if (item == null)
            {
                Debug.LogError($"[QuetesTab] ❌ Item {i} est NULL !");
                continue;
            }
            
            bool wasActive = item.activeInHierarchy;
            RectTransform rect = item.GetComponent<RectTransform>();
            Image img = item.GetComponent<Image>();
            CanvasGroup cg = item.GetComponent<CanvasGroup>();
            
            Debug.Log($"[QuetesTab] Item {i} ({item.name}) :");
            Debug.Log($"  Active: {wasActive}, Scale: {rect.localScale}, Position: {rect.anchoredPosition}");
            Debug.Log($"  Size: {rect.sizeDelta}, Color: {img?.color}");
            if (cg != null) Debug.Log($"  CanvasGroup alpha: {cg.alpha}, blocksRaycasts: {cg.blocksRaycasts}");
            
            // CORRECTIONS FORCÉES
            bool needsFix = false;
            
            if (!wasActive)
            {
                Debug.LogWarning($"[QuetesTab] ⚠️ Item {i} était désactivé ! Activation...");
                item.SetActive(true);
                needsFix = true;
            }
            
            if (rect.localScale == Vector3.zero || rect.localScale.x < 0.01f)
            {
                Debug.LogWarning($"[QuetesTab] ⚠️ Item {i} a une scale invalide ! Correction...");
                rect.localScale = Vector3.one;
                needsFix = true;
            }
            
            if (img != null && img.color.a < 0.01f)
            {
                Debug.LogWarning($"[QuetesTab] ⚠️ Item {i} est transparent (alpha={img.color.a}) ! Correction...");
                Color newColor = img.color;
                newColor.a = 1f;
                img.color = newColor;
                needsFix = true;
            }
            
            if (cg != null && cg.alpha < 0.01f)
            {
                Debug.LogWarning($"[QuetesTab] ⚠️ Item {i} a un CanvasGroup alpha={cg.alpha} ! Correction...");
                cg.alpha = 1f;
                needsFix = true;
            }
            
            // Vérifier les TextMeshPro enfants
            TextMeshProUGUI[] texts = item.GetComponentsInChildren<TextMeshProUGUI>();
            foreach (var text in texts)
            {
                if (text.color.a < 0.01f)
                {
                    Debug.LogWarning($"[QuetesTab] ⚠️ Item {i} texte '{text.text}' transparent ! Correction...");
                    Color textColor = text.color;
                    textColor.a = 1f;
                    text.color = textColor;
                    needsFix = true;
                }
            }
            
            if (needsFix)
            {
                Debug.Log($"[QuetesTab] ✅ Item {i} corrigé !");
            }
        }
        
        // Forcer le recalcul du layout
        if (contentContainer != null)
        {
            Debug.Log("[QuetesTab] 🔄 Forçage du recalcul du layout...");
            UnityEngine.UI.LayoutRebuilder.ForceRebuildLayoutImmediate(contentContainer);
            Canvas.ForceUpdateCanvases();
        }
        
        Debug.Log("[QuetesTab] 🔧 Diagnostic et correction terminés !");
    }
    
    private void DiagnoseItems()
    {
        Debug.Log($"[QuetesTab] 🔍 DIAGNOSTIC des {questItems.Count} items créés :");
        
        for (int i = 0; i < questItems.Count; i++)
        {
            GameObject item = questItems[i];
            if (item == null)
            {
                Debug.LogError($"[QuetesTab] ❌ Item {i} est NULL !");
                continue;
            }
            
            RectTransform rect = item.GetComponent<RectTransform>();
            Image img = item.GetComponent<Image>();
            Button btn = item.GetComponent<Button>();
            
            Debug.Log($"[QuetesTab] Item {i}: {item.name}");
            Debug.Log($"[QuetesTab]   → Active: {item.activeInHierarchy}");
            Debug.Log($"[QuetesTab]   → Position: {rect.anchoredPosition}");
            Debug.Log($"[QuetesTab]   → Size: {rect.sizeDelta}");
            Debug.Log($"[QuetesTab]   → Image color: {img.color}");
            Debug.Log($"[QuetesTab]   → Image raycastTarget: {img.raycastTarget}");
            Debug.Log($"[QuetesTab]   → Button interactable: {btn.interactable}");
            
            // Vérifier les enfants
            int childCount = item.transform.childCount;
            Debug.Log($"[QuetesTab]   → Enfants: {childCount}");
            
            for (int j = 0; j < childCount; j++)
            {
                Transform child = item.transform.GetChild(j);
                TextMeshProUGUI text = child.GetComponent<TextMeshProUGUI>();
                if (text != null)
                {
                    Debug.Log($"[QuetesTab]     → Enfant {j}: {child.name}, texte: '{text.text}', couleur: {text.color}");
                }
            }
        }
    }
    
    private IEnumerator ForceLayoutRebuildNextFrame()
    {
        Debug.Log("[QuetesTab] 🔄 Attente d'un frame pour le recalcul du layout...");
        
        // Attendre la fin du frame pour que tous les enfants soient bien positionnés
        yield return new WaitForEndOfFrame();
        
        if (contentContainer != null)
        {
            Debug.Log("[QuetesTab] 🔄 Forçage du recalcul du layout...");
            Debug.Log($"[QuetesTab] 📏 ContentContainer AVANT: {contentContainer.rect.width} x {contentContainer.rect.height}");
            
            // Forcer le recalcul du layout
            UnityEngine.UI.LayoutRebuilder.ForceRebuildLayoutImmediate(contentContainer);
            Canvas.ForceUpdateCanvases();
            
            // Attendre encore un frame
            yield return null;
            
            Debug.Log($"[QuetesTab] 📏 ContentContainer APRÈS: {contentContainer.rect.width} x {contentContainer.rect.height}");
            Debug.Log("[QuetesTab] ✅ Layout recalculé");
            
            // Vérifier les positions des items
            for (int i = 0; i < questItems.Count; i++)
            {
                if (questItems[i] != null)
                {
                    RectTransform itemRect = questItems[i].GetComponent<RectTransform>();
                    Debug.Log($"[QuetesTab] 📍 Item {i} position finale: {itemRect.anchoredPosition}");
                }
            }
            
            // Si la hauteur est toujours 0, il y a un problème plus profond
            if (contentContainer.rect.height <= 0)
            {
                Debug.LogError("[QuetesTab] ❌ La hauteur du ContentContainer est toujours 0 !");
                Debug.LogError("[QuetesTab] ❌ Vérifiez que les items ont des LayoutElement avec des hauteurs définies");
                
                // Essayer une approche alternative : positionner manuellement
                Debug.Log("[QuetesTab] 🔧 Tentative de positionnement manuel...");
                float yOffset = 0;
                for (int i = 0; i < questItems.Count; i++)
                {
                    if (questItems[i] != null)
                    {
                        RectTransform itemRect = questItems[i].GetComponent<RectTransform>();
                        itemRect.anchoredPosition = new Vector2(0, -yOffset);
                        yOffset += 110; // 100 + 10 de spacing
                        Debug.Log($"[QuetesTab] 🔧 Item {i} positionné manuellement à: {itemRect.anchoredPosition}");
                    }
                }
                
                // Mettre à jour la taille du content
                contentContainer.sizeDelta = new Vector2(0, yOffset);
                Debug.Log($"[QuetesTab] 🔧 ContentContainer taille mise à jour: {contentContainer.sizeDelta}");
            }
        }
    }
    
    private GameObject CreateQuestItem(MapInfo map, int index)
    {
        Debug.Log($"[QuetesTab] CreateQuestItem() - Création de {map.name}");
        
        // Vérifier que contentContainer existe
        if (contentContainer == null)
        {
            Debug.LogError("[QuetesTab] ❌ contentContainer est NULL dans CreateQuestItem !");
            return null;
        }
        
        // Créer le GameObject de base
        GameObject itemObj = new GameObject($"QuestItem_{map.id}");
        itemObj.transform.SetParent(contentContainer, false);
        
        RectTransform itemRect = itemObj.AddComponent<RectTransform>();
        itemRect.sizeDelta = new Vector2(0, 100); // Hauteur fixe
        
        Debug.Log($"[QuetesTab]   → Item créé: {itemObj.name}, parent: {itemObj.transform.parent.name}");
        Debug.Log($"[QuetesTab]   → Item position: {itemRect.anchoredPosition}, active: {itemObj.activeInHierarchy}");
        
        // Background avec couleur BIEN VISIBLE
        Image bgImage = itemObj.AddComponent<Image>();
        bgImage.color = new Color(0.8f, 0.7f, 0.6f, 1f); // Couleur plus visible
        bgImage.sprite = null;
        bgImage.raycastTarget = true; // IMPORTANT : Bloquer les raycasts
        
        // Button
        Button button = itemObj.AddComponent<Button>();
        button.targetGraphic = bgImage;
        button.onClick.AddListener(() => OnQuestClicked(index, map));
        button.interactable = true; // Assurez-vous que le bouton est interactable
        
        // Layout Group pour organiser le contenu
        VerticalLayoutGroup vlg = itemObj.AddComponent<VerticalLayoutGroup>();
        vlg.padding = new RectOffset(15, 15, 10, 10);
        vlg.spacing = 5f;
        vlg.childAlignment = TextAnchor.UpperLeft;
        vlg.childControlWidth = true;
        vlg.childControlHeight = false;
        vlg.childForceExpandWidth = true;
        vlg.childForceExpandHeight = false;
        
        // Titre avec couleur BIEN VISIBLE
        GameObject titleObj = new GameObject("Title");
        titleObj.transform.SetParent(itemObj.transform, false);
        TextMeshProUGUI titleText = titleObj.AddComponent<TextMeshProUGUI>();
        titleText.text = map.name.ToUpper();
        titleText.fontSize = 18;
        titleText.fontStyle = FontStyles.Bold;
        titleText.color = new Color(0.1f, 0.1f, 0.1f, 1f); // Noir pour être bien visible
        titleText.alignment = TextAlignmentOptions.Left;
        
        LayoutElement titleLayout = titleObj.AddComponent<LayoutElement>();
        titleLayout.preferredHeight = 25;
        
        // Description (si disponible) avec couleur BIEN VISIBLE
        if (!string.IsNullOrEmpty(map.description))
        {
            GameObject descObj = new GameObject("Description");
            descObj.transform.SetParent(itemObj.transform, false);
            TextMeshProUGUI descText = descObj.AddComponent<TextMeshProUGUI>();
            descText.text = map.description;
            descText.fontSize = 12;
            descText.color = new Color(0.2f, 0.2f, 0.2f, 1f); // Gris foncé pour être bien visible
            descText.alignment = TextAlignmentOptions.Left;
            descText.enableWordWrapping = true;
            
            LayoutElement descLayout = descObj.AddComponent<LayoutElement>();
            descLayout.preferredHeight = 50;
            descLayout.flexibleHeight = 1;
        }
        
        // Ajouter des effets de hover
        AddHoverEffects(button, bgImage, index);
        
        Debug.Log($"[QuetesTab]   ✅ Item {map.name} créé avec succès");
        
        return itemObj;
    }
    
    private void AddHoverEffects(Button button, Image bgImage, int index)
    {
        UnityEngine.EventSystems.EventTrigger trigger = button.gameObject.AddComponent<UnityEngine.EventSystems.EventTrigger>();
        
        // Hover Enter
        UnityEngine.EventSystems.EventTrigger.Entry entryEnter = new UnityEngine.EventSystems.EventTrigger.Entry();
        entryEnter.eventID = UnityEngine.EventSystems.EventTriggerType.PointerEnter;
        entryEnter.callback.AddListener((data) => 
        {
            if (selectedMapIndex != index)
            {
                bgImage.color = hoverColor;
            }
        });
        trigger.triggers.Add(entryEnter);
        
        // Hover Exit
        UnityEngine.EventSystems.EventTrigger.Entry entryExit = new UnityEngine.EventSystems.EventTrigger.Entry();
        entryExit.eventID = UnityEngine.EventSystems.EventTriggerType.PointerExit;
        entryExit.callback.AddListener((data) => 
        {
            if (selectedMapIndex != index)
            {
                bgImage.color = normalColor;
            }
        });
        trigger.triggers.Add(entryExit);
    }
    
    private void OnQuestClicked(int index, MapInfo map)
    {
        Debug.Log($"[QuetesTab] Quête cliquée: {map.name} ({map.configUrl})");
        
        // Mettre à jour la sélection visuelle
        UpdateSelection(index);
        
        // Sauvegarder la map sélectionnée dans PlayerPrefs
        PlayerPrefs.SetString("SelectedMapId", map.id);
        PlayerPrefs.SetString("SelectedMapConfig", map.configUrl);
        PlayerPrefs.Save();
        
        // Fermer le panneau de paramètres
        if (SettingsManager.Instance != null)
        {
            SettingsManager.Instance.CloseSettings();
        }
        
        // Charger la scène de la map après un court délai
        StartCoroutine(LoadMapSceneDelayed(map));
    }
    
    private void UpdateSelection(int index)
    {
        selectedMapIndex = index;
        
        for (int i = 0; i < questItems.Count; i++)
        {
            Image bgImage = questItems[i].GetComponent<Image>();
            if (bgImage != null)
            {
                bgImage.color = (i == index) ? selectedColor : normalColor;
            }
        }
    }
    
    private IEnumerator LoadMapSceneDelayed(MapInfo map)
    {
        yield return new WaitForSeconds(0.3f);
        
        Debug.Log($"[QuetesTab] Chargement de la scène Map avec config: {map.configUrl}");
        
        // Déclencher l'affichage du loading screen
        UnifiedLoadingManager.ShowLoading("Chargement de la carte...");
        
        // Charger la scène
        SceneManager.LoadScene("Map");
    }
}
