#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Script pour fusionner les 3 fichiers de dialogues (intro, success, fail) 
en un seul fichier par jeu
"""

import json
import os
from pathlib import Path

def merge_dialogue_files(json_folder):
    """
    Fusionne les fichiers de dialogues intro, success, fail en un seul fichier
    """
    json_path = Path(json_folder)
    
    # Trouver tous les fichiers *_intro.json
    intro_files = list(json_path.glob("*_intro.json"))
    
    print(f"[OK] Trouve {len(intro_files)} fichiers intro a traiter\n")
    
    merged_count = 0
    
    for intro_file in intro_files:
        # Extraire le nom de base (ex: Q0J1 depuis Q0J1_intro.json)
        base_name = intro_file.stem.replace("_intro", "")
        
        # Chemins des 3 fichiers
        success_file = json_path / f"{base_name}_success.json"
        fail_file = json_path / f"{base_name}_fail.json"
        
        # Vérifier que les 3 fichiers existent
        if not success_file.exists() or not fail_file.exists():
            print(f"[SKIP] {base_name} - Fichiers incomplets")
            continue
        
        try:
            # Charger les 3 fichiers
            with open(intro_file, 'r', encoding='utf-8') as f:
                intro_data = json.load(f)
            
            with open(success_file, 'r', encoding='utf-8') as f:
                success_data = json.load(f)
            
            with open(fail_file, 'r', encoding='utf-8') as f:
                fail_data = json.load(f)
            
            # Créer la nouvelle structure
            merged_data = {
                "dialogues": {
                    "intro": intro_data.get("dialogue", {}),
                    "success": success_data.get("dialogue", {}),
                    "fail": fail_data.get("dialogue", {})
                }
            }
            
            # Sauvegarder le fichier fusionné
            output_file = json_path / f"{base_name}_dialogues.json"
            with open(output_file, 'w', encoding='utf-8') as f:
                json.dump(merged_data, f, indent=4, ensure_ascii=False)
            
            print(f"[OK] {base_name} -> {base_name}_dialogues.json")
            merged_count += 1
            
        except Exception as e:
            print(f"[ERREUR] {base_name}: {e}")
    
    print(f"\n[OK] {merged_count} fichiers fusionnes")
    return merged_count

def update_game_configs(json_folder):
    """
    Met à jour les fichiers de configuration des jeux pour pointer vers les nouveaux fichiers de dialogues
    """
    json_path = Path(json_folder)
    
    # Trouver tous les fichiers Q*.json (sauf les dialogues)
    game_files = [f for f in json_path.glob("Q*.json") 
                  if not any(x in f.name for x in ['_intro', '_success', '_fail', '_dialogues'])]
    
    print(f"\n[OK] Mise a jour de {len(game_files)} fichiers de jeu\n")
    
    updated_count = 0
    
    for game_file in game_files:
        try:
            # Lire le fichier
            with open(game_file, 'r', encoding='utf-8') as f:
                data = json.load(f)
            
            # Vérifier si le fichier a une config de dialogues
            if 'gameConfig' in data:
                game_config = data['gameConfig']
                
                # Extraire le nom de base du jeu (ex: Q0J1)
                base_name = game_file.stem
                
                # Si les anciennes URLs existent, les remplacer par la nouvelle
                if any(key in game_config for key in ['dialogueBeforeUrl', 'dialogueSuccessUrl', 'dialogueFailUrl']):
                    # Remplacer par le nouveau système
                    game_config['dialoguesUrl'] = f"{base_name}_dialogues.json"
                    
                    # Supprimer les anciennes clés
                    game_config.pop('dialogueBeforeUrl', None)
                    game_config.pop('dialogueSuccessUrl', None)
                    game_config.pop('dialogueFailUrl', None)
                    
                    # Sauvegarder
                    with open(game_file, 'w', encoding='utf-8') as f:
                        json.dump(data, f, indent=4, ensure_ascii=False)
                    
                    print(f"[OK] {game_file.name} -> dialoguesUrl: {base_name}_dialogues.json")
                    updated_count += 1
        
        except Exception as e:
            print(f"[ERREUR] {game_file.name}: {e}")
    
    print(f"\n[OK] {updated_count} fichiers de jeu mis a jour")

if __name__ == "__main__":
    json_folder = r"d:\UJSA_IA\septembre\Assets\StreamingAssets\json"
    
    print("="*60)
    print("   FUSION DES FICHIERS DE DIALOGUES")
    print("="*60)
    print()
    
    # Étape 1 : Fusionner les fichiers de dialogues
    merge_dialogue_files(json_folder)
    
    # Étape 2 : Mettre à jour les fichiers de configuration des jeux
    update_game_configs(json_folder)
    
    print()
    print("="*60)
    print("   TERMINE")
    print("="*60)
    print()
    print("Prochaines etapes :")
    print("1. Verifier les nouveaux fichiers *_dialogues.json")
    print("2. Tester dans Unity")
    print("3. Si OK, supprimer les anciens fichiers *_intro/success/fail.json")

