using UnityEngine;
using UnityEngine.Video;
using UnityEngine.UI;

/// <summary>
/// Fix pour les vidéos du DialoguePlayer
/// Copie l'approche réussie de GameManager.backgroundVideo
/// 
/// Problème : DialoguePlayer affiche un voile blanc sur les vidéos
/// Solution : Forcer le même comportement que backgroundVideo (qui fonctionne)
/// </summary>
public class DialoguePlayerVideoFix : MonoBehaviour
{
    void Awake()
    {
        // LOG ULTRA-PRÉCOCE - Dans Awake avant Start
        Debug.Log("[DialogueVideoFix] 🔥 AWAKE - Script détecté!");
        Debug.Log($"[DialogueVideoFix] GameObject actif: {gameObject.activeInHierarchy}");
        Debug.Log($"[DialogueVideoFix] Script actif: {enabled}");
        Debug.Log($"[DialogueVideoFix] Scène: {UnityEngine.SceneManagement.SceneManager.GetActiveScene().name}");
    }
    
    void Start()
    {
        // LOG IMMÉDIAT - AVANT TOUT
        Debug.Log("[DialogueVideoFix] ========== SCRIPT EXÉCUTÉ ==========");
        Debug.Log($"[DialogueVideoFix] Platform: {Application.platform}");
        Debug.Log($"[DialogueVideoFix] Scène: {UnityEngine.SceneManagement.SceneManager.GetActiveScene().name}");
        Debug.Log($"[DialogueVideoFix] GameObject: {gameObject.name}");
        Debug.Log("[DialogueVideoFix] ====================================");
        
        Debug.Log("[DialogueVideoFix] 🎬 Démarrage du fix vidéo dialogue...");
        
        // Détecter Mac
        bool isMac = Application.platform == RuntimePlatform.OSXEditor || 
                     Application.platform == RuntimePlatform.OSXPlayer;
        
        if (!isMac)
        {
            Debug.Log("[DialogueVideoFix] ⏭️ Non-Mac, pas nécessaire");
            return;
        }

        // Attendre que le DialoguePlayer crée son VideoPlayer
        StartCoroutine(FixDialogueVideoWhenReady());
    }

    System.Collections.IEnumerator FixDialogueVideoWhenReady()
    {
        Debug.Log("[DialogueVideoFix] 🔍 En attente du VideoPlayer du dialogue...");
        
        // Attendre jusqu'à 5 secondes pour que le VideoPlayer du dialogue apparaisse
        float timeout = 5f;
        float elapsed = 0f;
        
        while (elapsed < timeout)
        {
            // Chercher TOUS les VideoPlayer
            VideoPlayer[] allPlayers = FindObjectsByType<VideoPlayer>(FindObjectsSortMode.None);
            
            foreach (VideoPlayer vp in allPlayers)
            {
                // Chercher le VideoPlayer du DialoguePlayer
                // (généralement pas attaché à la caméra, pas en 3D, utilisé en UI)
                
                if (vp.isPlaying || vp.source == VideoSource.Url)
                {
                    Debug.Log($"[DialogueVideoFix] ✅ VideoPlayer trouvé: {vp.gameObject.name}");
                    
                    // Appliquer les mêmes paramètres que backgroundVideo
                    ApplyVideoFix(vp);
                    yield break; // Arrêter la coroutine
                }
            }
            
            elapsed += Time.deltaTime;
            yield return new WaitForSeconds(0.1f);
        }
        
        Debug.Log("[DialogueVideoFix] ⏱️ Timeout - VideoPlayer non trouvé dans le temps imparti");
    }

    void ApplyVideoFix(VideoPlayer videoPlayer)
    {
        try
        {
            Debug.Log("[DialogueVideoFix] 🔧 Application du fix vidéo...");
            
            // 1. Chercher la caméra principale
            Camera mainCam = Camera.main;
            if (mainCam == null)
            {
                Debug.LogError("[DialogueVideoFix] ❌ Caméra principale non trouvée!");
                return;
            }
            
            // 2. Appliquer les paramètres comme backgroundVideo (GameManager)
            videoPlayer.renderMode = VideoRenderMode.CameraNearPlane;
            videoPlayer.targetCamera = mainCam;
            videoPlayer.targetCameraAlpha = 1f; // Opaque
            
            Debug.Log($"[DialogueVideoFix] ✅ Render Mode: {videoPlayer.renderMode}");
            Debug.Log($"[DialogueVideoFix] ✅ Target Camera: {mainCam.name}");
            Debug.Log($"[DialogueVideoFix] ✅ Alpha: 1.0 (opaque)");
            
            // 3. Vérifier/corriger les overlays UI blancs
            FixUIOverlays();
            
            Debug.Log("[DialogueVideoFix] ✅ Fix appliqué avec succès!");
        }
        catch (System.Exception ex)
        {
            Debug.LogError($"[DialogueVideoFix] ❌ Erreur: {ex.Message}");
        }
    }

    void FixUIOverlays()
    {
        Debug.Log("[DialogueVideoFix] 🎨 Nettoyage des overlays UI...");
        
        // Trouver TOUS les Canvas et panels blancs du dialogue
        Canvas[] allCanvases = FindObjectsByType<Canvas>(FindObjectsSortMode.None);
        int cleanedCount = 0;
        
        foreach (Canvas canvas in allCanvases)
        {
            // Si c'est un canvas de dialogue (sortingOrder élevé)
            if (canvas.sortingOrder >= 40000)
            {
                Debug.Log($"[DialogueVideoFix] 📍 Canvas dialogue: {canvas.name} (order: {canvas.sortingOrder})");
                
                // Chercher les images blanches sans sprite
                Image[] images = canvas.GetComponentsInChildren<Image>();
                
                foreach (Image img in images)
                {
                    if (img.sprite == null && 
                        img.color.r > 0.8f && img.color.g > 0.8f && img.color.b > 0.8f && 
                        img.color.a > 0.1f)
                    {
                        // C'est probablement un fond blanc du DialoguePlayer
                        Debug.Log($"[DialogueVideoFix] 🎨 Image blanche trouvée: {img.name}");
                        
                        // La rendre complètement transparente
                        Color c = img.color;
                        c.a = 0f;
                        img.color = c;
                        
                        cleanedCount++;
                        Debug.Log($"[DialogueVideoFix] ✅ Rendue transparente");
                    }
                }
            }
        }
        
        Debug.Log($"[DialogueVideoFix] 🎨 {cleanedCount} overlay(s) nettoyé(s)");
    }
}
