using System.Collections;
using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Networking;
using UnityEngine.Video;
using TMPro;

/// <summary>
/// Gère la mise en page de la scène calculator selon les mêmes spécifications que shooting :
/// - Écran 1920x1080
/// - Image de fond chargée depuis URL (visible autour de la zone de jeu)
/// - Bandeau supérieur : 1480x160, à 40px du haut, coins arrondis
/// - Zone de jeu centrale : 1480x835, à 40px du bas, coins arrondis (contient vidéo + calculatrice)
/// </summary>
public class CalculatorGameLayout : MonoBehaviour
{
    [Header("Layout Configuration")]
    [SerializeField] private string backgroundImageUrl = "";
    
    [Header("Layout Dimensions")]
    [SerializeField] private float screenWidth = 1920f;
    [SerializeField] private float screenHeight = 1080f;
    [SerializeField] private float contentWidth = 1480f;
    [SerializeField] private float headerHeight = 160f;
    [SerializeField] private float gameAreaHeight = 835f;
    [SerializeField] private float topMargin = 40f;
    [SerializeField] private float bottomMargin = 40f;
    [SerializeField] private float cornerRadius = 20f;         // Coins arrondis du bandeau
    [SerializeField] private float gameAreaCornerRadius = 20f; // Coins arrondis du cadre central
    
    [Header("Colors")]
    [SerializeField] private Color headerBackgroundColor = new Color(0.98f, 0.94f, 0.92f, 1f); // Sera écrasé par general-config.json
    
    [Header("Debug")]
    [SerializeField] private bool showDebugGizmos = false;
    
    // Shadow config
    private BandShadowConfig shadowConfig;
    
    // Références UI créées
    private RawImage backgroundImage;
    private GameObject headerPanel;
    private GameObject gameAreaPanel;
    private GameObject videoContainer;
    private RawImage videoDisplay;
    private Transform ledContainer;
    private TextMeshProUGUI questionText;
    
    // RenderTexture pour la vidéo
    private RenderTexture videoRenderTexture;
    
    // Références internes
    private Canvas mainCanvas;
    private CanvasScaler canvasScaler;
    private Texture2D loadedBackgroundTexture;
    private bool isInitialized = false;
    
    // Propriétés publiques pour accès par le CalculatorGameManager
    public GameObject HeaderPanel => headerPanel;
    public GameObject GameAreaPanel => gameAreaPanel;
    public GameObject VideoContainer => videoContainer;
    public TextMeshProUGUI QuestionText => questionText;
    public Transform LedContainer => ledContainer;
    public bool IsInitialized => isInitialized;
    public RectTransform GameAreaRect => gameAreaPanel?.GetComponent<RectTransform>();
    public RenderTexture VideoRenderTexture => videoRenderTexture;
    public RawImage VideoDisplay => videoDisplay;
    public float ContentWidth => contentWidth;
    public float GameAreaHeight => gameAreaHeight;
    public float CornerRadius => cornerRadius;
    
    // Singleton pour accès facile
    public static CalculatorGameLayout Instance { get; private set; }
    
    void Awake()
    {
        if (Instance == null)
        {
            Instance = this;
        }
        else
        {
            Destroy(gameObject);
            return;
        }
        
        SetupCanvas();
        
        // Charger les paramètres depuis general-config.json AVANT de créer le layout
        LoadConfigFromGeneralConfig();
        
        CreateVideoRenderTexture();
        CreateLayoutImmediate();
    }
    
    /// <summary>
    /// Charge les paramètres du bandeau depuis general-config.json
    /// </summary>
    void LoadConfigFromGeneralConfig()
    {
        if (GeneralConfigManager.Instance == null)
        {
            Debug.LogWarning("[CalculatorGameLayout] GeneralConfigManager non disponible, utilisation des valeurs par défaut");
            return;
        }
        
        if (!GeneralConfigManager.Instance.IsConfigLoaded())
        {
            Debug.LogWarning("[CalculatorGameLayout] Config pas encore chargée, utilisation des valeurs par défaut");
            return;
        }
        
        var defaultUIConfig = GeneralConfigManager.Instance.GetDefaultUIConfig();
        if (defaultUIConfig?.bands != null)
        {
            // Hauteur du bandeau
            if (defaultUIConfig.bands.bandHeight > 0)
            {
                headerHeight = defaultUIConfig.bands.bandHeight;
                Debug.Log($"[CalculatorGameLayout] ✅ Hauteur bandeau depuis config: {headerHeight}px");
            }
            
            // Couleur du bandeau
            if (!string.IsNullOrEmpty(defaultUIConfig.bands.bandColor))
            {
                if (ColorUtility.TryParseHtmlString(defaultUIConfig.bands.bandColor, out Color parsedColor))
                {
                    parsedColor.a = defaultUIConfig.bands.bandAlpha;
                    headerBackgroundColor = parsedColor;
                    Debug.Log($"[CalculatorGameLayout] ✅ Couleur bandeau depuis config: {defaultUIConfig.bands.bandColor}");
                }
            }
            
            // Corner radius (bandeau)
            if (defaultUIConfig.bands.cornerRadius > 0)
            {
                cornerRadius = defaultUIConfig.bands.cornerRadius;
                Debug.Log($"[CalculatorGameLayout] ✅ Corner radius bandeau depuis config: {cornerRadius}px");
            }
            
            // Corner radius (cadre central / zone de jeu)
            if (defaultUIConfig.bands.gameAreaCornerRadius > 0)
            {
                gameAreaCornerRadius = defaultUIConfig.bands.gameAreaCornerRadius;
                Debug.Log($"[CalculatorGameLayout] ✅ Corner radius zone de jeu depuis config: {gameAreaCornerRadius}px");
            }
            
            // Shadow config
            shadowConfig = defaultUIConfig.bands.shadow;
            if (shadowConfig != null && shadowConfig.enabled)
            {
                Debug.Log($"[CalculatorGameLayout] ✅ Shadow configurée: {shadowConfig.color}");
            }
            
            // Recalculer la hauteur de la zone de jeu
            float spacing = 5f;
            gameAreaHeight = screenHeight - topMargin - bottomMargin - headerHeight - spacing;
            Debug.Log($"[CalculatorGameLayout] ✅ Zone de jeu recalculée: {gameAreaHeight}px");
        }
    }
    
    void Start()
    {
        // Charger l'image de fond depuis general-config ou URL spécifiée
        LoadBackgroundFromConfig();
        
        // Si la config n'était pas disponible dans Awake(), réessayer maintenant
        StartCoroutine(TryLoadConfigDelayed());
    }
    
    /// <summary>
    /// Essaie de charger la config après un court délai si elle n'était pas disponible dans Awake
    /// </summary>
    IEnumerator TryLoadConfigDelayed()
    {
        // Attendre que GeneralConfigManager soit disponible et chargé
        float timeout = 3f;
        float elapsed = 0f;
        
        while ((GeneralConfigManager.Instance == null || !GeneralConfigManager.Instance.IsConfigLoaded()) && elapsed < timeout)
        {
            yield return new WaitForSeconds(0.1f);
            elapsed += 0.1f;
        }
        
        if (GeneralConfigManager.Instance != null && GeneralConfigManager.Instance.IsConfigLoaded())
        {
            var defaultUIConfig = GeneralConfigManager.Instance.GetDefaultUIConfig();
            if (defaultUIConfig?.bands != null)
            {
                // Mettre à jour le corner radius du cadre central si nécessaire
                if (defaultUIConfig.bands.gameAreaCornerRadius > 0 && defaultUIConfig.bands.gameAreaCornerRadius != gameAreaCornerRadius)
                {
                    gameAreaCornerRadius = defaultUIConfig.bands.gameAreaCornerRadius;
                    Debug.Log($"[CalculatorGameLayout] ✅ (Start) Corner radius zone de jeu mis à jour: {gameAreaCornerRadius}px");
                    
                    // Mettre à jour les coins arrondis du gameAreaPanel
                    UpdateGameAreaCornerRadius();
                }
                
                // Mettre à jour le corner radius du bandeau si nécessaire
                if (defaultUIConfig.bands.cornerRadius > 0 && defaultUIConfig.bands.cornerRadius != cornerRadius)
                {
                    cornerRadius = defaultUIConfig.bands.cornerRadius;
                    Debug.Log($"[CalculatorGameLayout] ✅ (Start) Corner radius bandeau mis à jour: {cornerRadius}px");
                    
                    // Mettre à jour les coins arrondis du header
                    if (headerPanel != null)
                    {
                        var roundedCorners = headerPanel.GetComponent<RoundedCornersImage>();
                        if (roundedCorners != null)
                        {
                            roundedCorners.cornerRadius = cornerRadius;
                        }
                    }
                }
            }
        }
    }
    
    /// <summary>
    /// Met à jour les coins arrondis de la zone de jeu
    /// </summary>
    void UpdateGameAreaCornerRadius()
    {
        if (gameAreaPanel != null)
        {
            var roundedCorners = gameAreaPanel.GetComponent<RoundedCornersImage>();
            if (roundedCorners != null)
            {
                roundedCorners.cornerRadius = gameAreaCornerRadius;
                Debug.Log($"[CalculatorGameLayout] ✅ GameAreaPanel coins arrondis mis à jour: {gameAreaCornerRadius}px");
            }
            else
            {
                // Ajouter le composant si absent
                roundedCorners = gameAreaPanel.AddComponent<RoundedCornersImage>();
                roundedCorners.cornerRadius = gameAreaCornerRadius;
                Debug.Log($"[CalculatorGameLayout] ✅ RoundedCornersImage ajouté à GameAreaPanel: {gameAreaCornerRadius}px");
            }
        }
        else
        {
            Debug.LogWarning("[CalculatorGameLayout] GameAreaPanel non trouvé pour mise à jour des coins arrondis");
        }
    }
    
    void LoadBackgroundFromConfig()
    {
        string bgUrl = backgroundImageUrl;
        
        // Si pas d'URL spécifiée, utiliser celle de general-config
        if (string.IsNullOrEmpty(bgUrl) && GeneralConfigManager.Instance != null)
        {
            // Utiliser le chemin du fond d'écran principal depuis la config
            var config = GeneralConfigManager.Instance.GetConfig();
            if (config?.assetsPaths != null && !string.IsNullOrEmpty(config.assetsPaths.backgroundImagePath))
            {
                bgUrl = config.assetsPaths.backgroundImagePath + "main_background.jpg";
            }
        }
        
        if (!string.IsNullOrEmpty(bgUrl))
        {
            StartCoroutine(LoadBackgroundImageAsync(bgUrl));
        }
    }
    
    void SetupCanvas()
    {
        mainCanvas = GetComponentInParent<Canvas>();
        if (mainCanvas == null)
        {
            mainCanvas = FindFirstObjectByType<Canvas>();
        }
        
        if (mainCanvas != null)
        {
            canvasScaler = mainCanvas.GetComponent<CanvasScaler>();
            if (canvasScaler != null)
            {
                canvasScaler.uiScaleMode = CanvasScaler.ScaleMode.ScaleWithScreenSize;
                canvasScaler.referenceResolution = new Vector2(screenWidth, screenHeight);
                canvasScaler.matchWidthOrHeight = 0.5f;
            }
        }
    }
    
    void CreateVideoRenderTexture()
    {
        int width = Mathf.RoundToInt(contentWidth);
        int height = Mathf.RoundToInt(gameAreaHeight);
        
        videoRenderTexture = new RenderTexture(width, height, 0);
        videoRenderTexture.Create();
        
        Debug.Log($"[CalculatorGameLayout] RenderTexture créée: {width}x{height}");
    }
    
    void CreateLayoutImmediate()
    {
        if (mainCanvas == null)
        {
            Debug.LogError("[CalculatorGameLayout] Canvas introuvable!");
            return;
        }
        
        // 1. Créer l'image de fond
        CreateBackgroundImage();
        
        // 2. Créer le header (bandeau supérieur avec LEDs et question)
        CreateHeaderPanel();
        
        // 3. Créer la zone de jeu centrale (pour vidéo + calculatrice)
        CreateGameAreaPanel();
        
        isInitialized = true;
        Debug.Log("[CalculatorGameLayout] Layout créé avec succès");
    }
    
    void CreateBackgroundImage()
    {
        GameObject bgObj = new GameObject("BackgroundImage");
        bgObj.transform.SetParent(mainCanvas.transform, false);
        bgObj.transform.SetAsFirstSibling();
        
        RectTransform bgRect = bgObj.AddComponent<RectTransform>();
        bgRect.anchorMin = Vector2.zero;
        bgRect.anchorMax = Vector2.one;
        bgRect.sizeDelta = Vector2.zero;
        bgRect.anchoredPosition = Vector2.zero;
        
        backgroundImage = bgObj.AddComponent<RawImage>();
        backgroundImage.color = new Color(0.2f, 0.2f, 0.2f, 1f); // Gris foncé par défaut
        backgroundImage.raycastTarget = false;
        
        // Mettre derrière tout
        Canvas bgCanvas = bgObj.AddComponent<Canvas>();
        bgCanvas.overrideSorting = true;
        bgCanvas.sortingOrder = -10;
        
        Debug.Log("[CalculatorGameLayout] Background créé");
    }
    
    void CreateHeaderPanel()
    {
        headerPanel = new GameObject("HeaderPanel");
        headerPanel.transform.SetParent(mainCanvas.transform, false);
        
        RectTransform headerRect = headerPanel.AddComponent<RectTransform>();
        headerRect.anchorMin = new Vector2(0.5f, 1f);
        headerRect.anchorMax = new Vector2(0.5f, 1f);
        headerRect.pivot = new Vector2(0.5f, 1f);
        headerRect.sizeDelta = new Vector2(contentWidth, headerHeight);
        headerRect.anchoredPosition = new Vector2(0, -topMargin);
        
        // Image de fond du header
        Image headerImage = headerPanel.AddComponent<Image>();
        headerImage.color = headerBackgroundColor;
        headerImage.raycastTarget = false;
        
        // Ajouter les coins arrondis
        AddRoundedCorners(headerPanel, cornerRadius);
        
        // Ajouter l'ombre si configurée
        if (shadowConfig != null && shadowConfig.enabled)
        {
            Shadow shadow = headerPanel.AddComponent<Shadow>();
            
            Color shadowColor = new Color(0, 0, 0, 0.25f);
            if (ColorUtility.TryParseHtmlString(shadowConfig.color, out Color parsedShadowColor))
            {
                shadowColor = parsedShadowColor;
            }
            
            shadow.effectColor = shadowColor;
            shadow.effectDistance = new Vector2(shadowConfig.offsetX, -shadowConfig.offsetY);
            
            Debug.Log($"[CalculatorGameLayout] ✅ Shadow ajoutée au header: {shadowConfig.color}");
        }
        
        // Canvas pour le sorting order
        Canvas headerCanvas = headerPanel.AddComponent<Canvas>();
        headerCanvas.overrideSorting = true;
        headerCanvas.sortingOrder = 100;
        headerPanel.AddComponent<GraphicRaycaster>();
        
        // Créer le contenu du header
        CreateHeaderContent();
        
        Debug.Log($"[CalculatorGameLayout] Header créé: {contentWidth}x{headerHeight}, couleur: {headerBackgroundColor}");
    }
    
    void CreateHeaderContent()
    {
        // === CONTENEUR POUR LES LEDs (à gauche) ===
        GameObject ledContainerObj = new GameObject("LEDContainer");
        ledContainerObj.transform.SetParent(headerPanel.transform, false);
        
        RectTransform ledRect = ledContainerObj.AddComponent<RectTransform>();
        ledRect.anchorMin = new Vector2(0f, 0f);
        ledRect.anchorMax = new Vector2(0f, 1f);
        ledRect.pivot = new Vector2(0f, 0.5f);
        ledRect.sizeDelta = new Vector2(400f, 0f);
        ledRect.anchoredPosition = new Vector2(20f, 0f);
        
        ledContainer = ledContainerObj.transform;
        
        // === ZONE DE TEXTE POUR LA QUESTION ===
        GameObject questionObj = new GameObject("QuestionText");
        questionObj.transform.SetParent(headerPanel.transform, false);
        
        RectTransform questionRect = questionObj.AddComponent<RectTransform>();
        questionRect.anchorMin = new Vector2(0f, 0f);
        questionRect.anchorMax = new Vector2(1f, 1f);
        questionRect.offsetMin = new Vector2(420f, 20f);
        questionRect.offsetMax = new Vector2(-40f, -20f);
        
        questionText = questionObj.AddComponent<TextMeshProUGUI>();
        questionText.text = "Chargement de la question...";
        questionText.fontSize = 32f;
        questionText.fontStyle = FontStyles.Normal;
        questionText.color = new Color(0.25f, 0.2f, 0.3f, 1f);
        questionText.alignment = TextAlignmentOptions.Left;
        questionText.textWrappingMode = TextWrappingModes.Normal;
        questionText.overflowMode = TextOverflowModes.Ellipsis;
        questionText.enableAutoSizing = true;
        questionText.fontSizeMin = 20f;
        questionText.fontSizeMax = 38f;
        
        Debug.Log("[CalculatorGameLayout] Contenu header créé");
    }
    
    void CreateGameAreaPanel()
    {
        // === CONTENEUR PRINCIPAL AVEC COINS ARRONDIS (zone de jeu) ===
        gameAreaPanel = new GameObject("GameAreaPanel");
        gameAreaPanel.transform.SetParent(mainCanvas.transform, false);
        
        RectTransform gameRect = gameAreaPanel.AddComponent<RectTransform>();
        gameRect.anchorMin = new Vector2(0.5f, 0f);
        gameRect.anchorMax = new Vector2(0.5f, 0f);
        gameRect.pivot = new Vector2(0.5f, 0f);
        gameRect.sizeDelta = new Vector2(contentWidth, gameAreaHeight);
        gameRect.anchoredPosition = new Vector2(0, bottomMargin);
        
        // Canvas pour contrôler l'ordre d'affichage
        Canvas gameCanvas = gameAreaPanel.AddComponent<Canvas>();
        gameCanvas.overrideSorting = true;
        gameCanvas.sortingOrder = 10;
        gameAreaPanel.AddComponent<GraphicRaycaster>();
        
        // Image de fond du conteneur (couleur de fallback visible)
        Image containerBg = gameAreaPanel.AddComponent<Image>();
        containerBg.color = new Color(0.15f, 0.15f, 0.2f, 1f); // Bleu-gris foncé comme fallback
        containerBg.raycastTarget = true;
        
        // Ajouter les coins arrondis au conteneur (zone de jeu)
        AddRoundedCorners(gameAreaPanel, gameAreaCornerRadius);
        
        // Masque pour découper le contenu aux coins arrondis
        Mask containerMask = gameAreaPanel.AddComponent<Mask>();
        containerMask.showMaskGraphic = true; // Afficher le fond !
        
        // Le videoContainer est maintenant le même que gameAreaPanel
        videoContainer = gameAreaPanel;
        
        // Créer le display vidéo à l'intérieur
        CreateVideoDisplay();
        
        Debug.Log($"[CalculatorGameLayout] Game Area créée: {contentWidth}x{gameAreaHeight}");
    }
    
    void CreateVideoDisplay()
    {
        // === VIDEO DISPLAY (enfant du gameAreaPanel, remplit tout) ===
        GameObject videoObj = new GameObject("VideoDisplay");
        videoObj.transform.SetParent(gameAreaPanel.transform, false);
        
        RectTransform videoRect = videoObj.AddComponent<RectTransform>();
        videoRect.anchorMin = Vector2.zero;
        videoRect.anchorMax = Vector2.one;
        videoRect.sizeDelta = Vector2.zero;
        videoRect.anchoredPosition = Vector2.zero;
        
        videoDisplay = videoObj.AddComponent<RawImage>();
        videoDisplay.texture = videoRenderTexture;
        videoDisplay.color = Color.white;
        videoDisplay.raycastTarget = false;
        
        Debug.Log("[CalculatorGameLayout] Video display créé dans GameAreaPanel");
    }
    
    void AddRoundedCorners(GameObject obj, float radius)
    {
        var roundedCorners = obj.AddComponent<RoundedCornersImage>();
        if (roundedCorners != null)
        {
            roundedCorners.cornerRadius = radius;
        }
    }
    
    IEnumerator LoadBackgroundImageAsync(string url)
    {
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                loadedBackgroundTexture = DownloadHandlerTexture.GetContent(request);
                if (backgroundImage != null)
                {
                    backgroundImage.texture = loadedBackgroundTexture;
                    backgroundImage.color = Color.white;
                }
                Debug.Log($"[CalculatorGameLayout] Image de fond chargée: {url}");
            }
            else
            {
                Debug.LogWarning($"[CalculatorGameLayout] Erreur chargement fond: {request.error}");
            }
        }
    }
    
    /// <summary>
    /// Configure le VideoPlayer pour utiliser la RenderTexture du layout
    /// </summary>
    public void SetupVideoPlayer(VideoPlayer videoPlayer)
    {
        if (videoPlayer != null && videoRenderTexture != null)
        {
            videoPlayer.renderMode = VideoRenderMode.RenderTexture;
            videoPlayer.targetTexture = videoRenderTexture;
            Debug.Log("[CalculatorGameLayout] VideoPlayer configuré avec RenderTexture");
        }
    }
    
    /// <summary>
    /// Obtient le conteneur pour placer la calculatrice
    /// </summary>
    public Transform GetCalculatorParent()
    {
        return videoContainer?.transform;
    }
    
    void OnDestroy()
    {
        if (videoRenderTexture != null)
        {
            videoRenderTexture.Release();
            Destroy(videoRenderTexture);
        }
        
        if (loadedBackgroundTexture != null)
        {
            Destroy(loadedBackgroundTexture);
        }
        
        if (Instance == this)
        {
            Instance = null;
        }
    }
}

