using UnityEngine;
using UnityEngine.SceneManagement;
using System.Collections;

/// <summary>
/// Gestionnaire pour la scène de chargement dédiée
/// Utilisée pour les transitions critiques entre scènes
/// </summary>
public class LoadingSceneManager : MonoBehaviour
{
    [Header("Configuration")]
    [SerializeField] private string targetSceneName = "";
    [SerializeField] private float minimumLoadingTime = 1.0f;
    [SerializeField] private bool showProgressBar = true;
    
    [Header("UI References")]
    [SerializeField] private UnityEngine.UI.Slider progressBar;
    [SerializeField] private TMPro.TextMeshProUGUI progressText;
    [SerializeField] private TMPro.TextMeshProUGUI loadingMessageText;
    
    [Header("Messages")]
    [SerializeField] private string[] loadingMessages = {
        "Chargement en cours...",
        "Préparation des ressources...",
        "Initialisation du jeu...",
        "Presque terminé..."
    };
    
    private AsyncOperation loadingOperation;
    private float loadingStartTime;
    private int currentMessageIndex = 0;
    private Coroutine messageRotationCoroutine;
    
    void Start()
    {
        // Récupérer le nom de la scène cible depuis PlayerPrefs
        targetSceneName = PlayerPrefs.GetString("LoadingTargetScene", "");
        
        if (string.IsNullOrEmpty(targetSceneName))
        {
            Debug.LogError("Aucune scène cible définie pour le chargement !");
            return;
        }
        
        Debug.Log($"[LoadingSceneManager] Chargement de la scène: {targetSceneName}");
        
        // Initialiser l'UI
        InitializeUI();
        
        // Démarrer le chargement
        StartCoroutine(LoadTargetScene());
    }
    
    void InitializeUI()
    {
        // Masquer la barre de progression si pas souhaitée
        if (progressBar != null)
        {
            progressBar.gameObject.SetActive(showProgressBar);
        }
        
        // Initialiser les messages
        if (loadingMessageText != null && loadingMessages.Length > 0)
        {
            loadingMessageText.text = loadingMessages[0];
        }
        
        // Démarrer la rotation des messages
        if (loadingMessages.Length > 1)
        {
            messageRotationCoroutine = StartCoroutine(RotateLoadingMessages());
        }
    }
    
    IEnumerator LoadTargetScene()
    {
        loadingStartTime = Time.time;
        
        // Démarrer le chargement asynchrone
        loadingOperation = SceneManager.LoadSceneAsync(targetSceneName);
        loadingOperation.allowSceneActivation = false; // On contrôle manuellement l'activation
        
        // Attendre que la scène soit prête (90% chargée)
        while (loadingOperation.progress < 0.9f)
        {
            UpdateProgressUI();
            yield return null;
        }
        
        // Attendre le temps minimum de chargement
        float elapsedTime = Time.time - loadingStartTime;
        if (elapsedTime < minimumLoadingTime)
        {
            yield return new WaitForSeconds(minimumLoadingTime - elapsedTime);
        }
        
        // Finaliser l'UI
        if (progressBar != null)
        {
            progressBar.value = 1.0f;
        }
        
        if (progressText != null)
        {
            progressText.text = "100%";
        }
        
        if (loadingMessageText != null)
        {
            loadingMessageText.text = "Chargement terminé !";
        }
        
        // Attendre un court délai avant l'activation
        yield return new WaitForSeconds(0.5f);
        
        // Activer la scène
        loadingOperation.allowSceneActivation = true;
    }
    
    void UpdateProgressUI()
    {
        if (loadingOperation == null) return;
        
        float progress = loadingOperation.progress;
        
        // Mettre à jour la barre de progression
        if (progressBar != null)
        {
            progressBar.value = progress;
        }
        
        // Mettre à jour le texte de progression
        if (progressText != null)
        {
            int percentage = Mathf.RoundToInt(progress * 100);
            progressText.text = $"{percentage}%";
        }
    }
    
    IEnumerator RotateLoadingMessages()
    {
        while (true)
        {
            yield return new WaitForSeconds(2.0f);
            
            if (loadingMessageText != null && loadingMessages.Length > 0)
            {
                currentMessageIndex = (currentMessageIndex + 1) % loadingMessages.Length;
                loadingMessageText.text = loadingMessages[currentMessageIndex];
            }
        }
    }
    
    void OnDestroy()
    {
        if (messageRotationCoroutine != null)
        {
            StopCoroutine(messageRotationCoroutine);
        }
    }
    
    /// <summary>
    /// Méthode statique pour charger une scène avec l'écran de chargement
    /// </summary>
    public static void LoadSceneWithLoadingScreen(string sceneName, float minLoadingTime = 1.0f)
    {
        PlayerPrefs.SetString("LoadingTargetScene", sceneName);
        PlayerPrefs.SetFloat("MinimumLoadingTime", minLoadingTime);
        SceneManager.LoadScene("LoadingScene");
    }
}
