using UnityEngine;
using UnityEngine.Networking;
using System;
using System.Text;
using System.Collections;

/// <summary>
/// Gestionnaire singleton pour stocker les données utilisateur après connexion
/// </summary>
public class UserDataManager : MonoBehaviour
{
    private static UserDataManager _instance;
    public static UserDataManager Instance
    {
        get
        {
            if (_instance == null)
            {
                GameObject obj = new GameObject("UserDataManager");
                _instance = obj.AddComponent<UserDataManager>();
                DontDestroyOnLoad(obj);
            }
            return _instance;
        }
    }

    [Header("Données utilisateur (nouveau format API)")]
    public string token;
    public int userId;
    public string firstName;
    public string lastName;
    public string email;
    public string position;
    public string companyName;
    public bool hasMembership;
    public int timeUsed; // Temps utilisé récupéré depuis l'API
    
    [Header("Données utilisateur (ancien format - compatibilité)")]
    public bool isNaproMembership;
    public bool isClMembership;
    public bool isAnnieMembership;
    public bool isUjsaMembership;
    public bool emailValidated;
    public CompanyData company;
    public bool profilCompleted;
    public string aboutMe;
    
    [Header("Temps de jeu local")]
    public float playTime; // Temps de jeu en secondes (compteur local)
    
    [Header("Préférences de quêtes")]
    // Dictionnaire pour stocker le niveau de difficulté par questId
    private System.Collections.Generic.Dictionary<string, string> questDifficulties = new System.Collections.Generic.Dictionary<string, string>();

    [System.Serializable]
    public class CompanyData
    {
        public int id;
        public string name;
    }

    [System.Serializable]
    public class FreeTimeUsedRequest
    {
        public int time_used;
    }
    
    // === Classes pour le nouveau format API ===
    
    [System.Serializable]
    public class UserApiResponse
    {
        public string status;
        public string message;
        public UserApiData data;
    }
    
    [System.Serializable]
    public class UserApiData
    {
        public string first_name;
        public string last_name;
        public string email;
        public string position;
        public string company;
        public bool has_membership;
        public int time_used;
    }

    /// <summary>
    /// Stocke les données utilisateur après une connexion réussie
    /// </summary>
    public void StoreUserData(LoginResponseData data)
    {
        if (data == null)
        {
            Debug.LogError("[UserDataManager] Tentative de stockage de données null");
            return;
        }

        token = data.token;
        userId = data.user_id;
        email = data.email ?? "";
        isNaproMembership = data.is_napro_membership;
        isClMembership = data.is_cl_membership;
        isAnnieMembership = data.is_annie_membership;
        isUjsaMembership = data.is_ujsa_membership;
        emailValidated = data.email_validated;
        profilCompleted = data.profil_completed;
        aboutMe = data.about_me ?? "";

        if (data.company != null)
        {
            company = new CompanyData
            {
                id = data.company.id,
                name = data.company.name ?? ""
            };
        }

        // Afficher toutes les informations stockées dans la console
        LogUserData();
        
        // Sauvegarder dans PlayerPrefs pour la persistance
        SaveToPlayerPrefs();
        
        // Charger les données supplémentaires depuis le nouvel endpoint API
        LoadUserDataFromApi();
    }

    /// <summary>
    /// Affiche les informations utilisateur stockées dans la console
    /// </summary>
    public void LogUserData()
    {
        Debug.Log("═══════════════════════════════════════════════════════════");
        Debug.Log("[UserDataManager] 📋 INFORMATIONS UTILISATEUR STOCKÉES");
        Debug.Log("═══════════════════════════════════════════════════════════");
        
        if (IsLoggedIn())
        {
            Debug.Log($"✅ Utilisateur connecté");
            Debug.Log($"   User ID: {userId}");
            
            // Nouveau format
            if (!string.IsNullOrEmpty(firstName) || !string.IsNullOrEmpty(lastName))
            {
                Debug.Log($"   Prénom: {firstName}");
                Debug.Log($"   Nom: {lastName}");
                Debug.Log($"   Poste: {position}");
                Debug.Log($"   Entreprise: {companyName}");
                Debug.Log($"   Membership: {(hasMembership ? "Oui" : "Non")}");
                Debug.Log($"   Temps utilisé (API): {timeUsed}s");
            }
            
            Debug.Log($"   Email: {email}");
            Debug.Log($"   Token: {(string.IsNullOrEmpty(token) ? "Non défini" : token.Substring(0, Mathf.Min(50, token.Length)) + "...")}");
            Debug.Log($"   Temps de jeu local: {FormatPlayTime(playTime)}");
            
            // Ancien format (si disponible)
            if (emailValidated || profilCompleted || company != null)
            {
                Debug.Log($"   [Ancien format]");
                Debug.Log($"   Email validé: {(emailValidated ? "Oui" : "Non")}");
                Debug.Log($"   Profil complété: {(profilCompleted ? "Oui" : "Non")}");
                Debug.Log($"   À propos: {aboutMe}");
                Debug.Log($"   Memberships: NaPro={isNaproMembership}, CL={isClMembership}, Annie={isAnnieMembership}, UJSA={isUjsaMembership}");
                
                if (company != null)
                {
                    Debug.Log($"   Entreprise (ancien): {company.name} (ID: {company.id})");
                }
            }
        }
        else
        {
            Debug.Log("❌ Aucun utilisateur connecté");
        }
        
        Debug.Log("═══════════════════════════════════════════════════════════");
    }

    /// <summary>
    /// Retourne un résumé des informations utilisateur sous forme de string
    /// </summary>
    public string GetUserDataSummary()
    {
        if (!IsLoggedIn())
        {
            return "Aucun utilisateur connecté";
        }

        string summary = $"User ID: {userId}\n";
        summary += $"Email: {email}\n";
        summary += $"Email validé: {(emailValidated ? "Oui" : "Non")}\n";
        summary += $"Profil complété: {(profilCompleted ? "Oui" : "Non")}\n";
        summary += $"À propos: {aboutMe}\n";
        summary += $"Membreships - NaPro: {(isNaproMembership ? "Oui" : "Non")}, CL: {(isClMembership ? "Oui" : "Non")}, Annie: {(isAnnieMembership ? "Oui" : "Non")}, UJSA: {(isUjsaMembership ? "Oui" : "Non")}\n";
        
        if (company != null)
        {
            summary += $"Entreprise: {company.name} (ID: {company.id})";
        }
        else
        {
            summary += "Entreprise: Aucune";
        }

        return summary;
    }

    /// <summary>
    /// Vérifie si l'utilisateur est connecté
    /// </summary>
    public bool IsLoggedIn()
    {
        return !string.IsNullOrEmpty(token);
    }

    /// <summary>
    /// Déconnecte l'utilisateur (efface les données)
    /// </summary>
    public void Logout()
    {
        // Nouveau format
        token = null;
        userId = 0;
        firstName = "";
        lastName = "";
        email = "";
        position = "";
        companyName = "";
        hasMembership = false;
        timeUsed = 0;
        playTime = 0f;
        
        // Ancien format
        isNaproMembership = false;
        isClMembership = false;
        isAnnieMembership = false;
        isUjsaMembership = false;
        emailValidated = false;
        profilCompleted = false;
        aboutMe = "";
        company = null;

        // Effacer aussi les PlayerPrefs
        ClearPlayerPrefs();
        
        // Effacer les difficultés de quêtes
        questDifficulties.Clear();

        Debug.Log("[UserDataManager] Utilisateur déconnecté");
    }
    
    /// <summary>
    /// Définit le niveau de difficulté pour une quête spécifique
    /// </summary>
    public void SetQuestDifficulty(string questId, string difficulty)
    {
        if (string.IsNullOrEmpty(questId))
        {
            Debug.LogWarning("[UserDataManager] ⚠️ Tentative de définir une difficulté avec un questId vide");
            return;
        }
        
        questDifficulties[questId] = difficulty;
        SaveQuestDifficultiesToPlayerPrefs();
        Debug.Log($"[UserDataManager] 📊 Difficulté '{difficulty}' enregistrée pour la quête '{questId}'");
    }
    
    /// <summary>
    /// Récupère le niveau de difficulté pour une quête spécifique
    /// </summary>
    /// <param name="questId">ID de la quête</param>
    /// <param name="defaultDifficulty">Difficulté par défaut si aucune n'est enregistrée</param>
    /// <returns>Le niveau de difficulté enregistré ou la valeur par défaut</returns>
    public string GetQuestDifficulty(string questId, string defaultDifficulty = "Débutant")
    {
        if (string.IsNullOrEmpty(questId))
        {
            Debug.LogWarning("[UserDataManager] ⚠️ Tentative de récupérer une difficulté avec un questId vide");
            return defaultDifficulty;
        }
        
        if (questDifficulties.ContainsKey(questId))
        {
            string difficulty = questDifficulties[questId];
            Debug.Log($"[UserDataManager] 📊 Difficulté récupérée pour '{questId}': {difficulty}");
            return difficulty;
        }
        
        Debug.Log($"[UserDataManager] 📊 Aucune difficulté enregistrée pour '{questId}', utilisation de '{defaultDifficulty}'");
        return defaultDifficulty;
    }
    
    /// <summary>
    /// Sauvegarde les difficultés de quêtes dans PlayerPrefs
    /// </summary>
    private void SaveQuestDifficultiesToPlayerPrefs()
    {
        try
        {
            // Convertir le dictionnaire en JSON
            string json = JsonUtility.ToJson(new QuestDifficultiesData(questDifficulties));
            PlayerPrefs.SetString("UserData_QuestDifficulties", json);
            PlayerPrefs.Save();
        }
        catch (System.Exception e)
        {
            Debug.LogError($"[UserDataManager] ❌ Erreur lors de la sauvegarde des difficultés: {e.Message}");
        }
    }
    
    /// <summary>
    /// Charge les difficultés de quêtes depuis PlayerPrefs
    /// </summary>
    private void LoadQuestDifficultiesFromPlayerPrefs()
    {
        try
        {
            if (PlayerPrefs.HasKey("UserData_QuestDifficulties"))
            {
                string json = PlayerPrefs.GetString("UserData_QuestDifficulties");
                QuestDifficultiesData data = JsonUtility.FromJson<QuestDifficultiesData>(json);
                questDifficulties = data.ToDictionary();
                Debug.Log($"[UserDataManager] 📊 {questDifficulties.Count} difficulté(s) de quête(s) chargée(s)");
            }
        }
        catch (System.Exception e)
        {
            Debug.LogError($"[UserDataManager] ❌ Erreur lors du chargement des difficultés: {e.Message}");
            questDifficulties = new System.Collections.Generic.Dictionary<string, string>();
        }
    }
    
    /// <summary>
    /// Classe helper pour sérialiser le dictionnaire en JSON
    /// </summary>
    [System.Serializable]
    private class QuestDifficultiesData
    {
        public string[] keys;
        public string[] values;
        
        public QuestDifficultiesData(System.Collections.Generic.Dictionary<string, string> dict)
        {
            keys = new string[dict.Count];
            values = new string[dict.Count];
            int i = 0;
            foreach (var kvp in dict)
            {
                keys[i] = kvp.Key;
                values[i] = kvp.Value;
                i++;
            }
        }
        
        public System.Collections.Generic.Dictionary<string, string> ToDictionary()
        {
            var dict = new System.Collections.Generic.Dictionary<string, string>();
            for (int i = 0; i < keys.Length && i < values.Length; i++)
            {
                dict[keys[i]] = values[i];
            }
            return dict;
        }
    }
    
    /// <summary>
    /// Sauvegarde les données utilisateur dans PlayerPrefs
    /// </summary>
    private void SaveToPlayerPrefs()
    {
        try
        {
            // Nouveau format
            PlayerPrefs.SetString("UserData_Token", token ?? "");
            PlayerPrefs.SetInt("UserData_UserId", userId);
            PlayerPrefs.SetString("UserData_FirstName", firstName ?? "");
            PlayerPrefs.SetString("UserData_LastName", lastName ?? "");
            PlayerPrefs.SetString("UserData_Email", email ?? "");
            PlayerPrefs.SetString("UserData_Position", position ?? "");
            PlayerPrefs.SetString("UserData_CompanyName", companyName ?? "");
            PlayerPrefs.SetInt("UserData_HasMembership", hasMembership ? 1 : 0);
            PlayerPrefs.SetInt("UserData_TimeUsed", timeUsed);
            PlayerPrefs.SetFloat("UserData_PlayTime", playTime);
            
            // Ancien format (compatibilité)
            PlayerPrefs.SetInt("UserData_IsNaproMembership", isNaproMembership ? 1 : 0);
            PlayerPrefs.SetInt("UserData_IsClMembership", isClMembership ? 1 : 0);
            PlayerPrefs.SetInt("UserData_IsAnnieMembership", isAnnieMembership ? 1 : 0);
            PlayerPrefs.SetInt("UserData_IsUjsaMembership", isUjsaMembership ? 1 : 0);
            PlayerPrefs.SetInt("UserData_EmailValidated", emailValidated ? 1 : 0);
            PlayerPrefs.SetInt("UserData_ProfilCompleted", profilCompleted ? 1 : 0);
            PlayerPrefs.SetString("UserData_AboutMe", aboutMe ?? "");
            
            if (company != null)
            {
                string companyJson = JsonUtility.ToJson(company);
                PlayerPrefs.SetString("UserData_Company", companyJson);
            }
            else
            {
                PlayerPrefs.DeleteKey("UserData_Company");
            }
            
            PlayerPrefs.Save();
        }
        catch (Exception e)
        {
            Debug.LogError($"[UserDataManager] ❌ Erreur lors de la sauvegarde dans PlayerPrefs: {e.Message}");
        }
    }
    
    /// <summary>
    /// Charge les données utilisateur depuis PlayerPrefs
    /// </summary>
    private void LoadFromPlayerPrefs()
    {
        try
        {
            // Vérifier si des données existent
            if (!PlayerPrefs.HasKey("UserData_Token"))
            {
                Debug.Log("[UserDataManager] Aucune donnée sauvegardée dans PlayerPrefs");
                playTime = 0f;
                return;
            }
            
            // Nouveau format
            token = PlayerPrefs.GetString("UserData_Token", "");
            userId = PlayerPrefs.GetInt("UserData_UserId", 0);
            firstName = PlayerPrefs.GetString("UserData_FirstName", "");
            lastName = PlayerPrefs.GetString("UserData_LastName", "");
            email = PlayerPrefs.GetString("UserData_Email", "");
            position = PlayerPrefs.GetString("UserData_Position", "");
            companyName = PlayerPrefs.GetString("UserData_CompanyName", "");
            hasMembership = PlayerPrefs.GetInt("UserData_HasMembership", 0) == 1;
            timeUsed = PlayerPrefs.GetInt("UserData_TimeUsed", 0);
            playTime = PlayerPrefs.GetFloat("UserData_PlayTime", 0f);
            
            // Ancien format (compatibilité)
            isNaproMembership = PlayerPrefs.GetInt("UserData_IsNaproMembership", 0) == 1;
            isClMembership = PlayerPrefs.GetInt("UserData_IsClMembership", 0) == 1;
            isAnnieMembership = PlayerPrefs.GetInt("UserData_IsAnnieMembership", 0) == 1;
            isUjsaMembership = PlayerPrefs.GetInt("UserData_IsUjsaMembership", 0) == 1;
            emailValidated = PlayerPrefs.GetInt("UserData_EmailValidated", 0) == 1;
            profilCompleted = PlayerPrefs.GetInt("UserData_ProfilCompleted", 0) == 1;
            aboutMe = PlayerPrefs.GetString("UserData_AboutMe", "");
            
            // Charger la company depuis JSON (ancien format)
            if (PlayerPrefs.HasKey("UserData_Company"))
            {
                string companyJson = PlayerPrefs.GetString("UserData_Company", "");
                if (!string.IsNullOrEmpty(companyJson))
                {
                    company = JsonUtility.FromJson<CompanyData>(companyJson);
                }
            }
            
            // Vérifier que le token existe (sinon les données sont invalides)
            if (string.IsNullOrEmpty(token))
            {
                Debug.Log("[UserDataManager] Token vide dans PlayerPrefs, données invalides");
                ClearPlayerPrefs();
                return;
            }
            
            // Charger les difficultés de quêtes
            LoadQuestDifficultiesFromPlayerPrefs();
            
            LogUserData();
            
            // Rafraîchir les données depuis l'API si connecté (avec délai pour attendre GeneralConfigManager)
            StartCoroutine(LoadUserDataFromApiDelayed());
        }
        catch (Exception e)
        {
            Debug.LogError($"[UserDataManager] ❌ Erreur lors du chargement depuis PlayerPrefs: {e.Message}");
            ClearPlayerPrefs();
        }
    }
    
    /// <summary>
    /// Efface TOUS les PlayerPrefs (pas seulement les données utilisateur)
    /// </summary>
    private void ClearPlayerPrefs()
    {
        // Effacer TOUS les PlayerPrefs
        PlayerPrefs.DeleteAll();
        PlayerPrefs.Save();
        
        Debug.Log("[UserDataManager] ✅ TOUS les PlayerPrefs effacés");
    }

    void Awake()
    {
        if (_instance == null)
        {
            _instance = this;
            DontDestroyOnLoad(gameObject);
            
            // Charger les données sauvegardées au démarrage
            LoadFromPlayerPrefs();
            
            // Démarrer le chronomètre de temps de jeu
            StartCoroutine(PlayTimeCounter());
        }
        else if (_instance != this)
        {
            Destroy(gameObject);
        }
    }
    
    /// <summary>
    /// Coroutine qui incrémente le temps de jeu chaque seconde et sauvegarde régulièrement
    /// </summary>
    private IEnumerator PlayTimeCounter()
    {
        float lastSaveTime = Time.time;
        const float saveInterval = 5f; // Sauvegarder toutes les 5 secondes
        
        while (true)
        {
            yield return new WaitForSeconds(1f);
            
            // Incrémenter le temps de jeu
            playTime += 1f;
            
            // Sauvegarder régulièrement dans PlayerPrefs et envoyer via l'API
            if (Time.time - lastSaveTime >= saveInterval)
            {
                SaveToPlayerPrefs();
                StartCoroutine(SendPlayTimeToAPI());
                lastSaveTime = Time.time;
            }
        }
    }

    /// <summary>
    /// Envoie le temps de jeu utilisé via l'API
    /// </summary>
    private IEnumerator SendPlayTimeToAPI()
    {
        // Ne pas envoyer si l'utilisateur n'est pas connecté
        if (!IsLoggedIn() || string.IsNullOrEmpty(token))
        {
            yield break;
        }

        // Récupérer l'URL de base depuis general-config.json
        string apiBaseUrl = GeneralConfigManager.Instance?.GetApiBaseUrl() ?? "";
        
        if (string.IsNullOrEmpty(apiBaseUrl))
        {
            Debug.LogError("[UserDataManager] ❌ URL de base de l'API non définie dans general-config.json - Envoi du playtime annulé");
            yield break;
        }
        
        string url = $"{apiBaseUrl}/api/ujsa/user/free-time-used";

        FreeTimeUsedRequest requestData = new FreeTimeUsedRequest
        {
            time_used = Mathf.FloorToInt(playTime)
        };

        string jsonData = JsonUtility.ToJson(requestData);
        byte[] bodyRaw = Encoding.UTF8.GetBytes(jsonData);

        using (UnityWebRequest www = new UnityWebRequest(url, "POST"))
        {
            www.uploadHandler = new UploadHandlerRaw(bodyRaw);
            www.downloadHandler = new DownloadHandlerBuffer();
            www.SetRequestHeader("Content-Type", "application/json");
            www.SetRequestHeader("Authorization", $"Bearer {token}");

            yield return www.SendWebRequest();

            if (www.result != UnityWebRequest.Result.Success)
            {
                // Log seulement si ce n'est pas une erreur 404 (endpoint pas encore déployé)
                if (!www.error.Contains("404"))
                {
                    Debug.LogError($"[UserDataManager] ❌ Erreur lors de l'envoi du playtime: {www.error}");
                }
            }
        }
    }
    
    /// <summary>
    /// Charge les données utilisateur depuis l'API /api/ujsa/user
    /// </summary>
    public void LoadUserDataFromApi()
    {
        if (!IsLoggedIn())
        {
            Debug.LogWarning("[UserDataManager] Impossible de charger les données: utilisateur non connecté");
            return;
        }
        
        StartCoroutine(LoadUserDataFromApiCoroutine());
    }
    
    /// <summary>
    /// Charge les données utilisateur depuis l'API avec un délai pour attendre GeneralConfigManager
    /// </summary>
    private IEnumerator LoadUserDataFromApiDelayed()
    {
        // Attendre que GeneralConfigManager soit initialisé
        float timeout = 5f;
        float elapsed = 0f;
        
        while (GeneralConfigManager.Instance == null || string.IsNullOrEmpty(GeneralConfigManager.Instance.GetApiBaseUrl()))
        {
            elapsed += 0.1f;
            if (elapsed >= timeout)
            {
                Debug.LogWarning("[UserDataManager] ⚠️ Timeout en attendant GeneralConfigManager - rafraîchissement API annulé");
                yield break;
            }
            yield return new WaitForSeconds(0.1f);
        }
        
        Debug.Log("[UserDataManager] 🔄 Rafraîchissement des données depuis l'API...");
        yield return StartCoroutine(LoadUserDataFromApiCoroutine());
    }
    
    /// <summary>
    /// Coroutine pour charger les données utilisateur depuis l'API
    /// </summary>
    private IEnumerator LoadUserDataFromApiCoroutine()
    {
        string apiBaseUrl = GeneralConfigManager.Instance?.GetApiBaseUrl() ?? "";
        
        if (string.IsNullOrEmpty(apiBaseUrl))
        {
            Debug.LogError("[UserDataManager] ❌ URL de base de l'API non définie");
            yield break;
        }
        
        string url = $"{apiBaseUrl}/api/ujsa/user";
        
        using (UnityWebRequest www = UnityWebRequest.Get(url))
        {
            www.SetRequestHeader("Authorization", $"Bearer {token}");
            www.SetRequestHeader("Content-Type", "application/json");
            
            yield return www.SendWebRequest();
            
            if (www.result != UnityWebRequest.Result.Success)
            {
                Debug.LogError($"[UserDataManager] ❌ Erreur chargement données utilisateur: {www.error}");
                yield break;
            }
            
            string jsonResponse = www.downloadHandler.text;
            
            try
            {
                UserApiResponse response = JsonUtility.FromJson<UserApiResponse>(jsonResponse);
                
                if (response != null && response.status == "success" && response.data != null)
                {
                    // Stocker les nouvelles données
                    firstName = response.data.first_name ?? "";
                    lastName = response.data.last_name ?? "";
                    email = response.data.email ?? "";
                    position = response.data.position ?? "";
                    companyName = response.data.company ?? "";
                    hasMembership = response.data.has_membership;
                    timeUsed = response.data.time_used;
                    
                    // Mettre à jour aussi les anciennes propriétés pour compatibilité
                    isUjsaMembership = hasMembership;
                    
                    Debug.Log("═══════════════════════════════════════════════════════════");
                    Debug.Log("[UserDataManager] 📋 DONNÉES UTILISATEUR CHARGÉES DEPUIS API");
                    Debug.Log("═══════════════════════════════════════════════════════════");
                    Debug.Log($"   Prénom: {firstName}");
                    Debug.Log($"   Nom: {lastName}");
                    Debug.Log($"   Email: {email}");
                    Debug.Log($"   Poste: {position}");
                    Debug.Log($"   Entreprise: {companyName}");
                    Debug.Log($"   Membership: {(hasMembership ? "Oui" : "Non")}");
                    Debug.Log($"   Temps utilisé: {timeUsed}s");
                    Debug.Log("═══════════════════════════════════════════════════════════");
                    
                    // Sauvegarder
                    SaveToPlayerPrefs();
                }
                else
                {
                    Debug.LogWarning($"[UserDataManager] ⚠️ Réponse API invalide: {jsonResponse}");
                }
            }
            catch (Exception e)
            {
                Debug.LogError($"[UserDataManager] ❌ Erreur parsing JSON: {e.Message}");
                Debug.LogError($"[UserDataManager] JSON reçu: {jsonResponse}");
            }
        }
    }
    
    /// <summary>
    /// Retourne le nom complet de l'utilisateur
    /// </summary>
    public string GetFullName()
    {
        if (!string.IsNullOrEmpty(firstName) || !string.IsNullOrEmpty(lastName))
        {
            return $"{firstName} {lastName}".Trim();
        }
        return email;
    }
    
    /// <summary>
    /// Formate le temps de jeu en heures:minutes:secondes
    /// </summary>
    public string FormatPlayTime(float seconds)
    {
        int hours = Mathf.FloorToInt(seconds / 3600f);
        int minutes = Mathf.FloorToInt((seconds % 3600f) / 60f);
        int secs = Mathf.FloorToInt(seconds % 60f);
        
        if (hours > 0)
        {
            return $"{hours:D2}:{minutes:D2}:{secs:D2}";
        }
        else
        {
            return $"{minutes:D2}:{secs:D2}";
        }
    }
}

