using UnityEngine;

/// <summary>
/// S'assure que tous les Managers essentiels existent dans la scène
/// À placer sur un GameObject dans la scène Menu (et autres scènes de départ)
/// </summary>
public class ManagersInitializer : MonoBehaviour
{
    [Header("Configuration")]
    [SerializeField] private bool initializeOnAwake = true;
    [SerializeField] private bool createManagersIfMissing = true;
    
    [Header("Prefabs (optionnel)")]
    [SerializeField] private GameObject generalConfigManagerPrefab;
    [SerializeField] private GameObject levelManagerPrefab;
    
    void Awake()
    {
        if (initializeOnAwake)
        {
            InitializeManagers();
        }
    }
    
    void InitializeManagers()
    {
        DebugLogger.Log("═══ Initialisation des Managers ═══", "ManagersInitializer");
        
        // 1. GeneralConfigManager
        if (GeneralConfigManager.Instance == null)
        {
            if (createManagersIfMissing)
            {
                DebugLogger.LogWarning("GeneralConfigManager manquant - création automatique", "ManagersInitializer");
                CreateGeneralConfigManager();
            }
            else
            {
                DebugLogger.LogWarning("GeneralConfigManager manquant mais createManagersIfMissing=false - création ignorée", "ManagersInitializer");
            }
        }
        else
        {
            DebugLogger.LogInfo("GeneralConfigManager trouvé ✓", "ManagersInitializer");
        }
        
        // 2. LevelManager
        if (LevelManager.Instance == null)
        {
            if (createManagersIfMissing)
            {
                DebugLogger.LogWarning("LevelManager manquant - création automatique", "ManagersInitializer");
                CreateLevelManager();
            }
            else
            {
                DebugLogger.LogWarning("LevelManager manquant mais createManagersIfMissing=false - création ignorée", "ManagersInitializer");
            }
        }
        else
        {
            DebugLogger.LogInfo("LevelManager trouvé ✓", "ManagersInitializer");
        }
        
        // 3. PostMessageBridge (pour la communication iframe)
        // Note: Les singletons gèrent eux-mêmes la logique WebGL, on les initialise toujours
        if (createManagersIfMissing)
        {
            Debug.Log("[ManagersInitializer] Initialisation de PostMessageBridge...");
            // Accéder à l'instance pour forcer sa création
            var bridge = PostMessageBridge.Instance;
            Debug.Log("[ManagersInitializer] PostMessageBridge initialisé ✓");
        }
        
        // 4. IframeAuthManager (pour l'authentification via iframe)
        if (createManagersIfMissing)
        {
            Debug.Log("[ManagersInitializer] Initialisation de IframeAuthManager...");
            // Accéder à l'instance pour forcer sa création
            var iframeAuth = IframeAuthManager.Instance;
            Debug.Log("[ManagersInitializer] IframeAuthManager initialisé ✓");
        }
        
        DebugLogger.Log("═══ Managers initialisés ═══", "ManagersInitializer");
    }
    
    void CreateGeneralConfigManager()
    {
        GameObject go;
        
        if (generalConfigManagerPrefab != null)
        {
            go = Instantiate(generalConfigManagerPrefab);
            go.name = "GeneralConfigManager";
        }
        else
        {
            go = new GameObject("GeneralConfigManager");
            go.AddComponent<GeneralConfigManager>();
        }
        
        DontDestroyOnLoad(go);
        DebugLogger.LogInfo("GeneralConfigManager créé", "ManagersInitializer");
    }
    
    void CreateLevelManager()
    {
        GameObject go;
        
        if (levelManagerPrefab != null)
        {
            go = Instantiate(levelManagerPrefab);
            go.name = "LevelManager";
        }
        else
        {
            go = new GameObject("LevelManager");
            go.AddComponent<LevelManager>();
        }
        
        DontDestroyOnLoad(go);
        DebugLogger.LogInfo("LevelManager créé", "ManagersInitializer");
    }
    
    /// <summary>
    /// Méthode publique pour initialiser manuellement
    /// </summary>
    public void ManualInitialize()
    {
        InitializeManagers();
    }
}

