using UnityEngine;
using UnityEngine.UI;
using TMPro;
using System.Text;
using System.Collections;

/// <summary>
/// Onglet Debug qui affiche toutes les variables utilisateur stockées
/// </summary>
public class DebugTab : SettingsTab
{
    public Transform contentContainer;
    private TextMeshProUGUI debugText;
    private Button logoutButton;
    
    protected override void Awake()
    {
        base.Awake();
        tabName = "DEBUG";
        tabOrder = 4;
        Debug.Log($"[DebugTab] Awake - contentContainer: {(contentContainer != null ? contentContainer.name : "NULL")}");
    }
    
    protected override void OnEnable()
    {
        base.OnEnable();
        Debug.Log($"[DebugTab] OnEnable appelé - contentContainer: {(contentContainer != null ? contentContainer.name : "NULL")}");
        
        // Attendre une frame pour s'assurer que tout est bien initialisé
        StartCoroutine(UpdateDebugInfoDelayed());
    }
    
    IEnumerator UpdateDebugInfoDelayed()
    {
        yield return null; // Attendre une frame
        Debug.Log($"[DebugTab] UpdateDebugInfoDelayed - contentContainer après attente: {(contentContainer != null ? contentContainer.name : "NULL")}");
        UpdateDebugInfo();
    }
    
    void UpdateDebugInfo()
    {
        Debug.Log($"[DebugTab] UpdateDebugInfo - contentContainer: {(contentContainer != null ? contentContainer.name : "NULL")}");
        
        if (contentContainer == null)
        {
            Debug.LogError("[DebugTab] ❌ contentContainer est NULL ! Impossible de créer le texte de débogage.");
            return;
        }
        
        // Supprimer l'ancien texte s'il existe
        if (debugText != null && debugText.gameObject != null)
        {
            Debug.Log("[DebugTab] Destruction de l'ancien texte de débogage");
            Destroy(debugText.gameObject);
            debugText = null;
        }
        
        // Créer un nouveau texte - SIMPLE et VISIBLE
        GameObject textObj = new GameObject("DebugText");
        textObj.transform.SetParent(contentContainer, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = new Vector2(0, 0);
        textRect.anchorMax = new Vector2(1, 1);
        textRect.pivot = new Vector2(0.5f, 1); // Pivot en haut
        textRect.anchoredPosition = Vector2.zero;
        textRect.offsetMin = new Vector2(0, 70); // Laisser de l'espace en bas pour le bouton
        textRect.offsetMax = new Vector2(0, 0); // Commencer depuis le haut
        
        debugText = textObj.AddComponent<TextMeshProUGUI>();
        debugText.fontSize = 22;
        debugText.alignment = TextAlignmentOptions.TopLeft;
        debugText.color = new Color(0.1f, 0.1f, 0.1f, 1f); // Noir pour être bien visible
        debugText.textWrappingMode = TMPro.TextWrappingModes.Normal;
        debugText.raycastTarget = false;
        debugText.overflowMode = TMPro.TextOverflowModes.Overflow;
        
        Debug.Log("[DebugTab] TextMeshProUGUI créé, construction du texte...");
        
        // Construire le texte avec toutes les informations utilisateur
        StringBuilder sb = new StringBuilder();
        sb.AppendLine("===========================================================");
        sb.AppendLine("INFORMATIONS UTILISATEUR STOCKÉES");
        sb.AppendLine("===========================================================");
        sb.AppendLine();
        
        if (UserDataManager.Instance != null)
        {
            Debug.Log($"[DebugTab] UserDataManager.Instance trouvé, IsLoggedIn: {UserDataManager.Instance.IsLoggedIn()}");
            
            if (UserDataManager.Instance.IsLoggedIn())
            {
                sb.AppendLine("<b>UTILISATEUR CONNECTÉ</b>");
                sb.AppendLine();
                sb.AppendLine($"<b>User ID:</b> {UserDataManager.Instance.userId}");
                sb.AppendLine($"<b>Email:</b> {UserDataManager.Instance.email}");
                sb.AppendLine();
                
                string tokenDisplay = string.IsNullOrEmpty(UserDataManager.Instance.token) 
                    ? "Non défini" 
                    : UserDataManager.Instance.token.Substring(0, Mathf.Min(50, UserDataManager.Instance.token.Length)) + "...";
                sb.AppendLine($"<b>Token:</b> {tokenDisplay}");
                sb.AppendLine();
                
                sb.AppendLine($"<b>Email validé:</b> {(UserDataManager.Instance.emailValidated ? "Oui" : "Non")}");
                sb.AppendLine($"<b>Profil complété:</b> {(UserDataManager.Instance.profilCompleted ? "Oui" : "Non")}");
                sb.AppendLine($"<b>À propos:</b> {UserDataManager.Instance.aboutMe}");
                sb.AppendLine();
                
                sb.AppendLine("<b>Adhésions:</b>");
                sb.AppendLine($"  - NaPro: {(UserDataManager.Instance.isNaproMembership ? "Oui" : "Non")}");
                sb.AppendLine($"  - CL: {(UserDataManager.Instance.isClMembership ? "Oui" : "Non")}");
                sb.AppendLine($"  - Annie: {(UserDataManager.Instance.isAnnieMembership ? "Oui" : "Non")}");
                sb.AppendLine($"  - UJSA: {(UserDataManager.Instance.isUjsaMembership ? "Oui" : "Non")}");
                sb.AppendLine();
                
                if (UserDataManager.Instance.company != null)
                {
                    sb.AppendLine("<b>Entreprise:</b>");
                    sb.AppendLine($"  - ID: {UserDataManager.Instance.company.id}");
                    sb.AppendLine($"  - Nom: {UserDataManager.Instance.company.name}");
                }
                else
                {
                    sb.AppendLine("<b>Entreprise:</b> Aucune");
                }
                sb.AppendLine();
                sb.AppendLine($"<b>Temps de jeu:</b> {UserDataManager.Instance.FormatPlayTime(UserDataManager.Instance.playTime)}");
            }
            else
            {
                sb.AppendLine("<size=28><b>AUCUNE DONNÉE UTILISATEUR</b></size>");
                sb.AppendLine();
                sb.AppendLine("<i>Aucun utilisateur n'est actuellement connecté.</i>");
                sb.AppendLine();
                sb.AppendLine("Le panneau d'identification s'affichera automatiquement");
                sb.AppendLine("au démarrage si vous n'êtes pas connecté.");
                sb.AppendLine();
                sb.AppendLine("<color=#666666>Les informations de connexion apparaîtront ici après identification.</color>");
                sb.AppendLine();
                sb.AppendLine($"<b>Temps de jeu:</b> {UserDataManager.Instance.FormatPlayTime(UserDataManager.Instance.playTime)}");
            }
        }
        else
        {
            Debug.LogWarning("[DebugTab] UserDataManager.Instance est NULL !");
            sb.AppendLine("<size=28><b>ERREUR SYSTÈME</b></size>");
            sb.AppendLine();
            sb.AppendLine("<color=red>UserDataManager non initialisé</color>");
            sb.AppendLine();
            sb.AppendLine("Veuillez redémarrer le jeu.");
        }
        
        sb.AppendLine();
        sb.AppendLine("===========================================================");
        
        string finalText = sb.ToString();
        debugText.text = finalText;
        
        Debug.Log($"[DebugTab] ✅ Texte assigné : {finalText.Length} caractères");
        Debug.Log($"[DebugTab] Contenu :\n{finalText}");
        
        // Créer ou mettre à jour le bouton de déconnexion
        CreateOrUpdateLogoutButton();
    }
    
    void CreateOrUpdateLogoutButton()
    {
        bool isLoggedIn = UserDataManager.Instance != null && UserDataManager.Instance.IsLoggedIn();
        
        // Si le bouton existe déjà, le mettre à jour
        if (logoutButton != null)
        {
            logoutButton.interactable = isLoggedIn;
            logoutButton.gameObject.SetActive(true);
            return;
        }
        
        // Créer le bouton de déconnexion
        GameObject buttonObj = new GameObject("LogoutButton");
        buttonObj.transform.SetParent(contentContainer, false);
        
        RectTransform buttonRect = buttonObj.AddComponent<RectTransform>();
        buttonRect.anchorMin = new Vector2(0.5f, 0);
        buttonRect.anchorMax = new Vector2(0.5f, 0);
        buttonRect.pivot = new Vector2(0.5f, 0);
        buttonRect.sizeDelta = new Vector2(300, 50);
        buttonRect.anchoredPosition = new Vector2(0, 20);
        
        Image buttonImage = buttonObj.AddComponent<Image>();
        buttonImage.color = new Color(0.8f, 0.2f, 0.2f, 1f); // Rouge
        
        logoutButton = buttonObj.AddComponent<Button>();
        logoutButton.interactable = isLoggedIn;
        
        // Couleurs du bouton
        ColorBlock colors = logoutButton.colors;
        colors.normalColor = new Color(0.8f, 0.2f, 0.2f, 1f);
        colors.highlightedColor = new Color(0.9f, 0.3f, 0.3f, 1f);
        colors.pressedColor = new Color(0.7f, 0.1f, 0.1f, 1f);
        colors.disabledColor = new Color(0.5f, 0.5f, 0.5f, 0.5f);
        logoutButton.colors = colors;
        
        // Texte du bouton
        GameObject textObj = new GameObject("Text");
        textObj.transform.SetParent(buttonObj.transform, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.offsetMin = Vector2.zero;
        textRect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI buttonText = textObj.AddComponent<TextMeshProUGUI>();
        buttonText.text = "SE DÉCONNECTER";
        buttonText.fontSize = 18;
        buttonText.alignment = TextAlignmentOptions.Center;
        buttonText.color = Color.white;
        buttonText.raycastTarget = false;
        
        // Action du bouton
        logoutButton.onClick.AddListener(OnLogoutClicked);
        
        Debug.Log($"[DebugTab] Bouton de déconnexion créé - Actif: {isLoggedIn}");
    }
    
    void OnLogoutClicked()
    {
        Debug.Log("[DebugTab] 🚪 Déconnexion demandée");
        
        if (UserDataManager.Instance != null)
        {
            // Déconnecter l'utilisateur (efface les PlayerPrefs)
            UserDataManager.Instance.Logout();
            
            Debug.Log("[DebugTab] ✅ Utilisateur déconnecté, redémarrage du jeu...");
            
            // Redémarrer le jeu en rechargeant la scène principale
            StartCoroutine(RestartGame());
        }
    }
    
    System.Collections.IEnumerator RestartGame()
    {
        // Attendre une frame pour que les logs soient visibles
        yield return null;
        
        // Fermer le panneau des paramètres
        if (SettingsManager.Instance != null)
        {
            SettingsManager.Instance.CloseSettings();
        }
        
        // Détecter la scène actuelle pour charger la bonne scène après déconnexion
        string currentSceneName = UnityEngine.SceneManagement.SceneManager.GetActiveScene().name;
        string targetScene = "main"; // Par défaut
        
        // Si on est dans la scène "menu", rester dans "menu"
        if (currentSceneName.ToLowerInvariant() == "menu")
        {
            targetScene = "menu";
            Debug.Log("[DebugTab] Scène actuelle: menu, redirection vers menu");
        }
        else
        {
            targetScene = "main";
            Debug.Log("[DebugTab] Scène actuelle: " + currentSceneName + ", redirection vers main");
        }
        
        // Charger la scène appropriée
        UnityEngine.SceneManagement.SceneManager.LoadScene(targetScene);
        
        Debug.Log($"[DebugTab] ✅ Jeu redémarré - Scène chargée: {targetScene}");
    }
}
