using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Video;
using UnityEngine.Networking;
using UnityEngine.EventSystems;
using TMPro;

/// <summary>
/// Construit dynamiquement le panneau de prévisualisation de quête
/// </summary>
public class QuestPreviewPanelBuilder : MonoBehaviour
{
    public static QuestPreviewPanelBuilder Instance { get; private set; }
    
    private Canvas parentCanvas;
    private GameObject panelRoot;
    private QuestPreviewPanelConfig config;
    private TMP_FontAsset titleFont;
    private TMP_FontAsset textFont;
    private Sprite closeButtonSprite;
    private Sprite questIconSprite;
    
    // Liste pour tracker les textures créées dynamiquement et les nettoyer
    private List<Texture2D> dynamicTextures = new List<Texture2D>();
    
    // Données de la quête actuelle
    private string currentQuestId;
    private int currentQuestNumericId; // ID numérique pour les appels API
    private string currentQuestTitle;
    private string currentQuestDescription;
    private string currentQuestDuration;
    private string currentVideoUrl;
    private string[] currentFormations;
    private List<Training> currentTrainings; // Données complètes avec URLs
    private List<QuestUserProgress> currentUserProgress; // Progression utilisateur par difficulté
    private Action onStartQuest;
    private Action onSkipQuest;
    
    // Références UI
    private List<Button> difficultyButtons = new List<Button>();
    private int selectedDifficultyIndex = -1; // -1 = aucune sélection
    private int inProgressDifficultyIndex = -1; // Index de la difficulté en cours (completion > 0 et < 100)
    private Button startButton;
    private Image startButtonImage;
    private Sprite startButtonEnabledSprite;
    private Sprite startButtonDisabledSprite;
    private TextMeshProUGUI startButtonText;
    
    void Awake()
    {
        if (Instance == null)
        {
            Instance = this;
        }
        else
        {
            Destroy(gameObject);
        }
    }
    
    public void SetParentCanvas(Canvas canvas)
    {
        parentCanvas = canvas;
        Debug.Log($"[QuestPreviewPanel] Canvas parent défini: {canvas.name}");
    }
    
    /// <summary>
    /// Affiche le panel avec les données d'une quête depuis l'API
    /// </summary>
    public void ShowQuestPreviewWithQuestData(string mapId, Quest questData, Action onStart, Action onSkip)
    {
        currentQuestId = mapId;
        currentQuestNumericId = questData.id; // Stocker l'ID numérique pour les appels API
        currentQuestTitle = questData.title;
        currentQuestDescription = questData.description;
        currentQuestDuration = questData.duration;
        currentVideoUrl = questData.trailer_url;
        currentTrainings = questData.trainings; // Stocker les trainings complets avec URLs
        currentUserProgress = questData.user; // Stocker la progression utilisateur
        
        // Convertir les trainings en tableau de string (pour compatibilité)
        if (questData.trainings != null && questData.trainings.Count > 0)
        {
            currentFormations = new string[questData.trainings.Count];
            for (int i = 0; i < questData.trainings.Count; i++)
            {
                currentFormations[i] = questData.trainings[i].title;
            }
        }
        else
        {
            currentFormations = new string[0];
        }
        
        onStartQuest = onStart;
        onSkipQuest = onSkip;
        
        // Utiliser sprite_url ou une icône par défaut
        string iconUrl = !string.IsNullOrEmpty(questData.sprite_url) ? questData.sprite_url : "UI_quest_icon_dda.png";
        
        // Log de la progression utilisateur
        if (currentUserProgress != null && currentUserProgress.Count > 0)
        {
            Debug.Log($"[QuestPreviewPanel] Progression utilisateur pour la quête:");
            foreach (var progress in currentUserProgress)
            {
                Debug.Log($"  - {progress.difficulty}: completion={progress.completion}%, score={progress.score}");
            }
        }
        else
        {
            Debug.Log($"[QuestPreviewPanel] Pas de données de progression utilisateur");
        }
        
        Debug.Log($"[QuestPreviewPanel] Affichage de la quête - Titre: {currentQuestTitle}, Durée: {currentQuestDuration}, Formations: {currentFormations.Length}");
        
        StartCoroutine(BuildPanelCoroutine(iconUrl));
    }
    
    /// <summary>
    /// Affiche le panel de prévisualisation pour une quête (ancienne méthode pour compatibilité)
    /// </summary>
    public void ShowQuestPreview(string questId, string title, string description, string duration, 
        string videoUrl, string[] formations, string iconUrl, Action onStart, Action onSkip)
    {
        currentQuestId = questId;
        currentQuestTitle = title;
        currentQuestDescription = description;
        currentQuestDuration = duration;
        currentVideoUrl = videoUrl;
        currentFormations = formations;
        onStartQuest = onStart;
        onSkipQuest = onSkip;
        
        StartCoroutine(BuildPanelCoroutine(iconUrl));
    }
    
    /// <summary>
    /// Vérifie si le panel est actuellement ouvert
    /// </summary>
    public bool IsPanelOpen()
    {
        return panelRoot != null;
    }
    
    /// <summary>
    /// Ferme le panel
    /// </summary>
    public void ClosePanel()
    {
        // Nettoyer toutes les textures dynamiques pour éviter les fuites mémoire
        foreach (Texture2D tex in dynamicTextures)
        {
            if (tex != null)
            {
                Destroy(tex);
            }
        }
        dynamicTextures.Clear();
        
        if (panelRoot != null)
        {
            Destroy(panelRoot);
            panelRoot = null;
        }
        
        Debug.Log("[QuestPreviewPanel] Panel fermé et ressources nettoyées");
    }
    
    /// <summary>
    /// Nettoie les ressources quand l'objet est détruit
    /// </summary>
    void OnDestroy()
    {
        // Nettoyer les textures si elles n'ont pas été nettoyées
        foreach (Texture2D tex in dynamicTextures)
        {
            if (tex != null)
            {
                Destroy(tex);
            }
        }
        dynamicTextures.Clear();
    }
    
    IEnumerator BuildPanelCoroutine(string iconUrl)
    {
        // Charger la configuration
        if (GeneralConfigManager.Instance == null)
        {
            Debug.LogError("[QuestPreviewPanel] ❌ GeneralConfigManager.Instance est null!");
            yield break;
        }
        
        var generalConfig = GeneralConfigManager.Instance.GetConfig();
        if (generalConfig == null)
        {
            Debug.LogError("[QuestPreviewPanel] ❌ GeneralConfig est null!");
            yield break;
        }
        
        config = generalConfig.questPreviewPanel;
        if (config == null)
        {
            Debug.LogError("[QuestPreviewPanel] ❌ questPreviewPanel n'est pas défini dans GeneralConfig!");
            Debug.LogError("[QuestPreviewPanel] ❌ Vérifiez que le fichier general-config.json contient bien la section 'questPreviewPanel'");
            yield break;
        }
        
        Debug.Log("[QuestPreviewPanel] ✅ Configuration chargée avec succès");
        
        // Charger les ressources en parallèle
        yield return StartCoroutine(LoadResources(iconUrl));
        
        // Trouver le canvas si non défini
        if (parentCanvas == null)
        {
            parentCanvas = FindFirstObjectByType<Canvas>();
            if (parentCanvas == null)
            {
                Debug.LogError("[QuestPreviewPanel] Aucun Canvas trouvé!");
                yield break;
            }
        }
        
        // Détruire l'ancien panel s'il existe
        if (panelRoot != null)
        {
            Destroy(panelRoot);
        }
        
        // Créer le panel
        CreatePanel();
        
        Debug.Log("[QuestPreviewPanel] Panel créé avec succès");
    }
    
    IEnumerator LoadResources(string iconUrl)
    {
        var generalConfig = GeneralConfigManager.Instance?.GetConfig();
        string uiPath = generalConfig?.assetsPaths?.uiPath ?? "";
        
        // Charger les polices (comme LoginPopup : essayer plusieurs chemins)
        string titleFontName = config.header?.title?.fontName ?? "Anton-Regular SDF";
        string textFontName = config.description?.fontName ?? "Lato-Regular SDF";
        
        Debug.Log($"[QuestPreviewPanel] Tentative chargement polices - Title: '{titleFontName}', Text: '{textFontName}'");
        
        // Essayer plusieurs chemins pour la police du titre (comme LoginPopup)
        titleFont = Resources.Load<TMP_FontAsset>($"Fonts/{titleFontName}");
        if (titleFont == null)
        {
            titleFont = Resources.Load<TMP_FontAsset>($"fonts/{titleFontName}");
        }
        if (titleFont == null)
        {
            titleFont = Resources.Load<TMP_FontAsset>(titleFontName);
        }
        
        // Essayer plusieurs chemins pour la police du texte
        textFont = Resources.Load<TMP_FontAsset>($"Fonts/{textFontName}");
        if (textFont == null)
        {
            textFont = Resources.Load<TMP_FontAsset>($"fonts/{textFontName}");
        }
        if (textFont == null)
        {
            textFont = Resources.Load<TMP_FontAsset>(textFontName);
        }
        
        if (titleFont != null)
            Debug.Log($"[QuestPreviewPanel] ✅ Police titre chargée: {titleFont.name}");
        else
            Debug.LogWarning($"[QuestPreviewPanel] ❌ Police titre '{titleFontName}' non trouvée - Essayé: Fonts/{titleFontName}, fonts/{titleFontName}, {titleFontName}");
            
        if (textFont != null)
            Debug.Log($"[QuestPreviewPanel] ✅ Police texte chargée: {textFont.name}");
        else
            Debug.LogWarning($"[QuestPreviewPanel] ❌ Police texte '{textFontName}' non trouvée");
        
        // Charger le sprite du bouton fermer
        string closeButtonUrl = uiPath + (config.header?.closeButton?.imageUrl ?? "UI_fermer.png");
        yield return StartCoroutine(LoadSprite(closeButtonUrl, sprite => closeButtonSprite = sprite));
        
        // Charger l'icône de la quête
        if (!string.IsNullOrEmpty(iconUrl))
        {
            string questIconUrl = iconUrl.StartsWith("http") ? iconUrl : uiPath + iconUrl;
            yield return StartCoroutine(LoadSprite(questIconUrl, sprite => questIconSprite = sprite));
        }
    }
    
    IEnumerator LoadSprite(string url, Action<Sprite> onLoaded)
    {
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                Sprite sprite = Sprite.Create(texture, new Rect(0, 0, texture.width, texture.height), 
                    new Vector2(0.5f, 0.5f), 100f);
                onLoaded?.Invoke(sprite);
            }
            else
            {
                Debug.LogWarning($"[QuestPreviewPanel] Échec du chargement: {url}");
            }
        }
    }
    
    IEnumerator LoadFormationArrowSprite(Image targetImage)
    {
        var generalConfig = GeneralConfigManager.Instance?.GetConfig();
        string uiPath = generalConfig?.assetsPaths?.uiPath ?? "https://ujsa.studioplc.fr/datas/UI/";
        string url = uiPath + "fleche_formation.png";
        
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                Sprite sprite = Sprite.Create(texture, new Rect(0, 0, texture.width, texture.height), 
                    new Vector2(0.5f, 0.5f), 100f);
                if (targetImage != null)
                {
                    targetImage.sprite = sprite;
                }
            }
            else
            {
                Debug.LogWarning($"[QuestPreviewPanel] ⚠️ Impossible de charger fleche_formation.png: {url}");
            }
        }
    }
    
    void CreatePanel()
    {
        // Créer un Canvas overlay dédié pour garantir le centrage
        panelRoot = new GameObject("QuestPreviewPanel");
        
        // Ajouter un Canvas en mode Overlay (comme LoginPopup)
        Canvas overlayCanvas = panelRoot.AddComponent<Canvas>();
        overlayCanvas.renderMode = RenderMode.ScreenSpaceOverlay;
        overlayCanvas.sortingOrder = 1000;  // Très au-dessus de tout (main canvas est à ~10, settings à ~100)
        
        // CanvasScaler identique à LoginPopup
        UnityEngine.UI.CanvasScaler scaler = panelRoot.AddComponent<UnityEngine.UI.CanvasScaler>();
        scaler.uiScaleMode = UnityEngine.UI.CanvasScaler.ScaleMode.ScaleWithScreenSize;
        scaler.referenceResolution = new Vector2(1920, 1080);
        scaler.matchWidthOrHeight = 0f;
        
        // GraphicRaycaster pour détecter les clics
        UnityEngine.UI.GraphicRaycaster raycaster = panelRoot.AddComponent<UnityEngine.UI.GraphicRaycaster>();
        raycaster.blockingObjects = UnityEngine.UI.GraphicRaycaster.BlockingObjects.All;
        
        Debug.Log($"[QuestPreviewPanel] Canvas overlay créé (sortingOrder={overlayCanvas.sortingOrder}, raycaster activé)");
        
        // Fond sombre (plein écran)
        CreateDarkBackground();
        
        // Container du panel (centré)
        GameObject panelContainer = CreatePanelContainer();
        
        // Header avec titre et bouton fermer
        CreateHeader(panelContainer.transform);
        
        // Description
        CreateDescription(panelContainer.transform);
        
        // Contenu principal (vidéo + infos)
        CreateMainContent(panelContainer.transform);
        
        // Lien d'aide (désactivé)
        // CreateHelpLink(panelContainer.transform);
        
        // Boutons
        CreateButtons(panelContainer.transform);
    }
    
    void CreateDarkBackground()
    {
        GameObject darkBg = new GameObject("DarkBackground");
        darkBg.transform.SetParent(panelRoot.transform, false);
        
        RectTransform rect = darkBg.AddComponent<RectTransform>();
        rect.anchorMin = Vector2.zero;
        rect.anchorMax = Vector2.one;
        rect.offsetMin = Vector2.zero;
        rect.offsetMax = Vector2.zero;
        
        Image image = darkBg.AddComponent<Image>();
        image.color = HexToColor(config.darkBackgroundColor);
        image.raycastTarget = true; // Bloque TOUS les clics sans action - ne ferme PAS le panel
        
        // Ajouter un EventTrigger pour capturer explicitement les clics et les empêcher de passer
        UnityEngine.EventSystems.EventTrigger eventTrigger = darkBg.AddComponent<UnityEngine.EventSystems.EventTrigger>();
        
        // Capturer les clics (PointerClick)
        UnityEngine.EventSystems.EventTrigger.Entry clickEntry = new UnityEngine.EventSystems.EventTrigger.Entry();
        clickEntry.eventID = UnityEngine.EventSystems.EventTriggerType.PointerClick;
        clickEntry.callback.AddListener((data) => {
            Debug.Log("[QuestPreviewPanel] Clic sur le fond sombre capturé et bloqué");
            // Ne rien faire - juste bloquer le clic
        });
        eventTrigger.triggers.Add(clickEntry);
        
        // Capturer aussi les PointerDown pour être encore plus sûr
        UnityEngine.EventSystems.EventTrigger.Entry downEntry = new UnityEngine.EventSystems.EventTrigger.Entry();
        downEntry.eventID = UnityEngine.EventSystems.EventTriggerType.PointerDown;
        downEntry.callback.AddListener((data) => {
            Debug.Log("[QuestPreviewPanel] PointerDown sur le fond sombre capturé et bloqué");
            // Ne rien faire - juste bloquer le clic
        });
        eventTrigger.triggers.Add(downEntry);
        
        Debug.Log("[QuestPreviewPanel] Fond sombre créé avec EventTrigger - bloque les clics, ne ferme PAS le panel");
    }
    
    GameObject CreatePanelContainer()
    {
        // Structure EXACTEMENT comme SettingsPanelBuilder
        // Créer un conteneur pour tout (ombre + panneau)
        GameObject outerContainer = new GameObject("PanelOuterContainer");
        outerContainer.transform.SetParent(panelRoot.transform, false);
        
        RectTransform outerRect = outerContainer.AddComponent<RectTransform>();
        outerRect.anchorMin = new Vector2(0.5f, 0.5f);
        outerRect.anchorMax = new Vector2(0.5f, 0.5f);
        outerRect.pivot = new Vector2(0.5f, 0.5f);
        outerRect.sizeDelta = new Vector2(config.panelSize.x, config.panelSize.y);
        outerRect.anchoredPosition = Vector2.zero;
        
        // Créer l'ombre comme un élément séparé DERRIÈRE le panneau
        if (config.shadow?.enabled ?? true)
        {
            GameObject shadowObj = new GameObject("PanelShadow");
            shadowObj.transform.SetParent(outerContainer.transform, false);
            
            RectTransform shadowRect = shadowObj.AddComponent<RectTransform>();
            shadowRect.anchorMin = Vector2.zero;
            shadowRect.anchorMax = Vector2.one;
            shadowRect.sizeDelta = Vector2.zero;
            // Décaler l'ombre selon la config
            shadowRect.anchoredPosition = new Vector2(config.shadow.offsetX, -config.shadow.offsetY);
            
            Image shadowImage = shadowObj.AddComponent<Image>();
            Color shadowColor = HexToColor(config.shadow.color);
            shadowImage.sprite = CreateRoundedColorSprite((int)config.panelSize.x, (int)config.panelSize.y, 
                config.cornerRadius, shadowColor);
            shadowImage.color = Color.white;
            shadowImage.type = Image.Type.Simple;
            shadowImage.raycastTarget = false;
            
            // L'ombre doit être en premier (derrière le panneau)
            shadowObj.transform.SetAsFirstSibling();
        }
        
        // Créer le panneau principal
        GameObject container = new GameObject("PanelContainer");
        container.transform.SetParent(outerContainer.transform, false);
        
        RectTransform rect = container.AddComponent<RectTransform>();
        
        // Remplir tout le conteneur parent (EXACTEMENT comme SettingsPanelBuilder)
        rect.anchorMin = Vector2.zero;
        rect.anchorMax = Vector2.one;
        rect.sizeDelta = Vector2.zero;
        rect.offsetMin = Vector2.zero;
        rect.offsetMax = Vector2.zero;
        rect.localScale = Vector3.one;
        rect.localRotation = Quaternion.identity;
        
        Image image = container.AddComponent<Image>();
        
        // Créer un sprite arrondi avec la couleur de fond (taille exacte comme SettingsPanelBuilder)
        image.sprite = CreateRoundedColorSprite((int)config.panelSize.x, (int)config.panelSize.y, 
            config.cornerRadius, HexToColor(config.backgroundColor));
        image.color = Color.white;
        image.type = Image.Type.Simple;
        image.raycastTarget = true;
        
        Debug.Log($"[QuestPreviewPanel] Panel créé: {config.panelSize.x}x{config.panelSize.y}, radius={config.cornerRadius}");
        
        return container;
    }
    
    void CreateHeader(Transform parent)
    {
        var headerConfig = config.header;
        
        GameObject header = new GameObject("Header");
        header.transform.SetParent(parent, false);
        
        RectTransform headerRect = header.AddComponent<RectTransform>();
        headerRect.anchorMin = new Vector2(0, 1);
        headerRect.anchorMax = new Vector2(1, 1);
        headerRect.pivot = new Vector2(0.5f, 1);
        headerRect.sizeDelta = new Vector2(0, headerConfig.height);
        headerRect.anchoredPosition = Vector2.zero;
        
        Image headerImage = header.AddComponent<Image>();
        
        // Créer un sprite arrondi uniquement en haut pour le header (EXACTEMENT comme SettingsPanelBuilder)
        headerImage.sprite = CreateRoundedTopSprite((int)config.panelSize.x, (int)headerConfig.height, 
            headerConfig.cornerRadius, HexToColor(headerConfig.backgroundColor));
        headerImage.color = Color.white;
        headerImage.type = Image.Type.Simple;
        headerImage.raycastTarget = true; // Bloquer les clics
        
        // Icône de la quête
        if (questIconSprite != null)
        {
            GameObject icon = new GameObject("QuestIcon");
            icon.transform.SetParent(header.transform, false);
            
            RectTransform iconRect = icon.AddComponent<RectTransform>();
            iconRect.anchorMin = new Vector2(0, 0.5f);
            iconRect.anchorMax = new Vector2(0, 0.5f);
            iconRect.pivot = new Vector2(0, 0.5f);
            iconRect.sizeDelta = new Vector2(headerConfig.iconSize, headerConfig.iconSize);
            iconRect.anchoredPosition = new Vector2(headerConfig.iconMarginLeft, 0);
            
            Image iconImage = icon.AddComponent<Image>();
            iconImage.sprite = questIconSprite;
            iconImage.preserveAspect = true;
        }
        
        // Titre - CENTRÉ
        GameObject titleObj = new GameObject("Title");
        titleObj.transform.SetParent(header.transform, false);
        
        RectTransform titleRect = titleObj.AddComponent<RectTransform>();
        // Centrer le titre dans le header (en laissant de l'espace pour l'icône et le bouton fermer)
        titleRect.anchorMin = new Vector2(0, 0);
        titleRect.anchorMax = new Vector2(1, 1);
        titleRect.offsetMin = Vector2.zero;
        titleRect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI titleText = titleObj.AddComponent<TextMeshProUGUI>();
        titleText.text = currentQuestTitle?.ToUpper() ?? ""; // En majuscules
        titleText.fontSize = headerConfig.title.fontSize;
        
        Color titleColor = HexToColor(headerConfig.title.textColor);
        titleText.color = titleColor;
        titleText.alignment = TextAlignmentOptions.Center; // CENTRÉ au lieu de MidlineLeft
        
        Debug.Log($"[QuestPreviewPanel] Config titre - fontSize: {headerConfig.title.fontSize}, textColor: {headerConfig.title.textColor} → RGBA({titleColor.r:F3}, {titleColor.g:F3}, {titleColor.b:F3}, {titleColor.a:F3}), fontName: {headerConfig.title.fontName}");
        
        // Style de police selon fontWeight (comme dans LoginPopup)
        string fontWeight = headerConfig.title.fontWeight ?? "normal";
        if (fontWeight.ToLower() == "bold")
            titleText.fontStyle = FontStyles.Bold;
        else
            titleText.fontStyle = FontStyles.Normal;
            
        if (titleFont != null)
        {
            titleText.font = titleFont;
            Debug.Log($"[QuestPreviewPanel] ✅ Police titre appliquée: {titleFont.name}");
        }
        else
        {
            Debug.LogWarning($"[QuestPreviewPanel] ⚠️ Police titre non chargée, utilisation de la police par défaut");
        }
        
        // Bouton fermer
        CreateCloseButton(header.transform);
    }
    
    void CreateCloseButton(Transform parent)
    {
        var closeConfig = config.header.closeButton;
        
        GameObject closeBtn = new GameObject("CloseButton");
        closeBtn.transform.SetParent(parent, false);
        
        RectTransform rect = closeBtn.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(1, 0.5f);
        rect.anchorMax = new Vector2(1, 0.5f);
        rect.pivot = new Vector2(1, 0.5f);
        rect.sizeDelta = new Vector2(closeConfig.size, closeConfig.size);
        rect.anchoredPosition = new Vector2(-closeConfig.marginRight, 0);
        
        Image image = closeBtn.AddComponent<Image>();
        if (closeButtonSprite != null)
        {
            image.sprite = closeButtonSprite;
            image.color = Color.white;
        }
        else
        {
            image.color = HexToColor("#64477f");
        }
        image.raycastTarget = true;
        
        Button button = closeBtn.AddComponent<Button>();
        button.targetGraphic = image;
        button.onClick.AddListener(ClosePanel);
        
        ColorBlock colors = button.colors;
        colors.normalColor = Color.white;
        colors.highlightedColor = new Color(0.9f, 0.9f, 0.9f, 1f);
        colors.pressedColor = new Color(0.8f, 0.8f, 0.8f, 1f);
        button.colors = colors;
    }
    
    void CreateDescription(Transform parent)
    {
        var descConfig = config.description;
        
        GameObject descObj = new GameObject("Description");
        descObj.transform.SetParent(parent, false);
        
        RectTransform rect = descObj.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(0, 1);
        rect.anchorMax = new Vector2(1, 1);
        rect.pivot = new Vector2(0.5f, 1);
        rect.anchoredPosition = new Vector2(0, -descConfig.positionY);
        rect.sizeDelta = new Vector2(-descConfig.marginHorizontal * 2, descConfig.height);
        
        TextMeshProUGUI text = descObj.AddComponent<TextMeshProUGUI>();
        text.text = currentQuestDescription;
        text.fontSize = descConfig.fontSize;
        text.color = HexToColor(descConfig.textColor);
        text.alignment = TextAlignmentOptions.Center;
        if (textFont != null) text.font = textFont;
        
        Debug.Log($"[QuestPreviewPanel] Description créée - Position Y: {descConfig.positionY}, Hauteur: {descConfig.height}");
    }
    
    void CreateMainContent(Transform parent)
    {
        // Container sans LayoutGroup - positionnement absolu
        GameObject content = new GameObject("MainContent");
        content.transform.SetParent(parent, false);
        
        RectTransform contentRect = content.AddComponent<RectTransform>();
        contentRect.anchorMin = Vector2.zero;
        contentRect.anchorMax = Vector2.one;
        contentRect.offsetMin = Vector2.zero;
        contentRect.offsetMax = Vector2.zero;
        
        // Créer chaque élément avec son positionnement absolu
        CreateDurationDifficultyCard(content.transform);
        CreateFormationsCard(content.transform);
        CreateVideoColumn(content.transform);
    }
    
    // Supprimée - utilise maintenant le positionnement absolu
    
    void CreateDurationDifficultyCard(Transform parent)
    {
        var cardConfig = config.durationDifficultyCard;
        var cardsConfig = config.infoCards;
        
        // Créer la carte avec positionnement absolu
        GameObject card = new GameObject("DurationDifficultyCard");
        card.transform.SetParent(parent, false);
        
        RectTransform cardRect = card.AddComponent<RectTransform>();
        cardRect.anchorMin = new Vector2(0, 1); // Ancrer en haut à gauche
        cardRect.anchorMax = new Vector2(0, 1);
        cardRect.pivot = new Vector2(0, 1);
        cardRect.anchoredPosition = new Vector2(cardConfig.x, -cardConfig.y);
        cardRect.sizeDelta = new Vector2(cardConfig.width, cardConfig.height);
        
        // Image de fond
        Image bgImage = card.AddComponent<Image>();
        bgImage.sprite = CreateRoundedColorSprite((int)cardConfig.width, (int)cardConfig.height, 
            cardConfig.cornerRadius, HexToColor(cardConfig.backgroundColor));
        bgImage.type = Image.Type.Simple;
        bgImage.raycastTarget = false;
        
        // Ombre
        Shadow shadow = card.AddComponent<Shadow>();
        shadow.effectColor = HexToColor(cardConfig.shadowColor);
        shadow.effectDistance = new Vector2(0, -cardConfig.shadowOffsetY);
        
        Debug.Log($"[QuestPreviewPanel] Carte Durée/Difficulté créée - Position: ({cardConfig.x}, {cardConfig.y}), Taille: {cardConfig.width}x{cardConfig.height}");
        
        // Positionnement absolu - pas de layout automatique
        
        // Durée - position depuis le JSON
        GameObject durationSection = new GameObject("DurationSection");
        durationSection.transform.SetParent(card.transform, false);
        
        RectTransform durationRect = durationSection.AddComponent<RectTransform>();
        durationRect.anchorMin = new Vector2(0, 1);
        durationRect.anchorMax = new Vector2(0, 1);
        durationRect.pivot = new Vector2(0, 1);
        durationRect.anchoredPosition = new Vector2(cardsConfig.duration.x, -cardsConfig.duration.y);
        
        VerticalLayoutGroup durationLayout = durationSection.AddComponent<VerticalLayoutGroup>();
        durationLayout.spacing = cardsConfig.duration.labelMarginBottom;
        durationLayout.childAlignment = TextAnchor.UpperCenter;
        durationLayout.childControlWidth = false;
        durationLayout.childControlHeight = false;
        
        ContentSizeFitter durationFitter = durationSection.AddComponent<ContentSizeFitter>();
        durationFitter.horizontalFit = ContentSizeFitter.FitMode.PreferredSize;
        durationFitter.verticalFit = ContentSizeFitter.FitMode.PreferredSize;
        
        CreateCardLabel(durationSection.transform, cardsConfig.duration.label);
        CreateCardValue(durationSection.transform, currentQuestDuration ?? cardsConfig.duration.defaultValue);
        
        // Difficulté - position depuis le JSON
        GameObject difficultySection = new GameObject("DifficultySection");
        difficultySection.transform.SetParent(card.transform, false);
        
        RectTransform difficultyRect = difficultySection.AddComponent<RectTransform>();
        difficultyRect.anchorMin = new Vector2(0, 1);
        difficultyRect.anchorMax = new Vector2(0, 1);
        difficultyRect.pivot = new Vector2(0, 1);
        difficultyRect.anchoredPosition = new Vector2(cardsConfig.difficulty.x, -cardsConfig.difficulty.y);
        
        VerticalLayoutGroup difficultyLayout = difficultySection.AddComponent<VerticalLayoutGroup>();
        difficultyLayout.spacing = cardsConfig.difficulty.labelMarginBottom;
        difficultyLayout.childAlignment = TextAnchor.UpperCenter;
        difficultyLayout.childControlWidth = false;
        difficultyLayout.childControlHeight = false;
        
        ContentSizeFitter difficultyFitter = difficultySection.AddComponent<ContentSizeFitter>();
        difficultyFitter.horizontalFit = ContentSizeFitter.FitMode.PreferredSize;
        difficultyFitter.verticalFit = ContentSizeFitter.FitMode.PreferredSize;
        
        CreateCardLabel(difficultySection.transform, cardsConfig.difficulty.label);
        CreateDifficultySelector(difficultySection.transform);
        
        Debug.Log($"[QuestPreviewPanel] Durée position: ({cardsConfig.duration.x}, {cardsConfig.duration.y}), Difficulté position: ({cardsConfig.difficulty.x}, {cardsConfig.difficulty.y})");
    }
    
    void CreateDifficultySelector(Transform parent)
    {
        var diffConfig = config.infoCards.difficulty;
        difficultyButtons.Clear();
        selectedDifficultyIndex = -1;
        
        // Container pour les boutons
        GameObject selectorObj = new GameObject("DifficultySelector");
        selectorObj.transform.SetParent(parent, false);
        
        RectTransform selectorRect = selectorObj.AddComponent<RectTransform>();
        selectorRect.anchorMin = new Vector2(0.5f, 0.5f);
        selectorRect.anchorMax = new Vector2(0.5f, 0.5f);
        selectorRect.pivot = new Vector2(0.5f, 0.5f);
        
        // Layout horizontal pour les boutons - centré
        HorizontalLayoutGroup layout = selectorObj.AddComponent<HorizontalLayoutGroup>();
        layout.spacing = 8;
        layout.childAlignment = TextAnchor.MiddleCenter;
        layout.childControlWidth = false;
        layout.childControlHeight = false;
        layout.childForceExpandWidth = false;
        layout.childForceExpandHeight = false;
        
        // ContentSizeFitter pour que le container s'adapte à ses enfants
        ContentSizeFitter fitter = selectorObj.AddComponent<ContentSizeFitter>();
        fitter.horizontalFit = ContentSizeFitter.FitMode.PreferredSize;
        fitter.verticalFit = ContentSizeFitter.FitMode.PreferredSize;
        
        // Couleurs
        Color normalBgColor = HexToColor("#f5ece5");
        Color selectedBgColor = HexToColor("#a95bfb"); // Violet
        Color normalTextColor = HexToColor("#64477f");
        Color selectedTextColor = Color.white;
        
        // Créer un bouton pour chaque option
        for (int i = 0; i < diffConfig.options.Length; i++)
        {
            string optionText = diffConfig.options[i];
            int buttonIndex = i;
            
            GameObject buttonObj = new GameObject($"DifficultyButton_{i}");
            buttonObj.transform.SetParent(selectorObj.transform, false);
            
            RectTransform buttonRect = buttonObj.AddComponent<RectTransform>();
            buttonRect.sizeDelta = new Vector2(140, 45);
            
            // Image de fond
            Image buttonBg = buttonObj.AddComponent<Image>();
            buttonBg.sprite = CreateRoundedColorSprite(140, 45, 22, Color.white);
            buttonBg.color = normalBgColor;
            buttonBg.type = Image.Type.Simple;
            buttonBg.raycastTarget = true;
            
            // LayoutElement pour le dimensionnement
            LayoutElement layoutElement = buttonObj.AddComponent<LayoutElement>();
            layoutElement.preferredWidth = 140;
            layoutElement.preferredHeight = 45;
            
            // Bouton
            Button button = buttonObj.AddComponent<Button>();
            button.targetGraphic = buttonBg;
            button.transition = Selectable.Transition.None; // On gère manuellement
            
            // Texte
            GameObject textObj = new GameObject("Text");
            textObj.transform.SetParent(buttonObj.transform, false);
            
            RectTransform textRect = textObj.AddComponent<RectTransform>();
            textRect.anchorMin = Vector2.zero;
            textRect.anchorMax = Vector2.one;
            textRect.offsetMin = Vector2.zero;
            textRect.offsetMax = Vector2.zero;
            
            TextMeshProUGUI buttonText = textObj.AddComponent<TextMeshProUGUI>();
            buttonText.text = optionText;
            buttonText.fontSize = diffConfig.buttonFontSize;
            buttonText.color = normalTextColor;
            buttonText.alignment = TextAlignmentOptions.Center;
            buttonText.fontStyle = FontStyles.Normal;
            if (textFont != null) buttonText.font = textFont;
            
            // Ajouter le listener
            button.onClick.AddListener(() => {
                SelectDifficulty(buttonIndex);
            });
            
            difficultyButtons.Add(button);
        }
        
        // Déterminer quelle difficulté sélectionner automatiquement
        int autoSelectIndex = DetermineAutoSelectDifficulty(diffConfig.options);
        
        if (autoSelectIndex >= 0)
        {
            SelectDifficulty(autoSelectIndex);
            Debug.Log($"[QuestPreviewPanel] 📊 Difficulté auto-sélectionnée: {diffConfig.options[autoSelectIndex]} (index: {autoSelectIndex})");
        }
        else
        {
            Debug.Log($"[QuestPreviewPanel] 📊 Aucune difficulté sélectionnée (toutes les completions sont à 0 ou pas de données)");
        }
        
        Debug.Log($"[QuestPreviewPanel] Sélecteur de difficulté créé avec {diffConfig.options.Length} boutons");
        
        // Mettre à jour l'état du bouton démarrer
        UpdateStartButtonState();
    }
    
    /// <summary>
    /// Détermine quelle difficulté sélectionner automatiquement basé sur la progression utilisateur
    /// </summary>
    /// <returns>Index de la difficulté à sélectionner, ou -1 si aucune</returns>
    private int DetermineAutoSelectDifficulty(string[] difficultyOptions)
    {
        // Réinitialiser l'index de difficulté en cours
        inProgressDifficultyIndex = -1;
        
        // Si pas de données de progression utilisateur, ne rien sélectionner
        if (currentUserProgress == null || currentUserProgress.Count == 0)
        {
            Debug.Log($"[QuestPreviewPanel] Pas de données de progression utilisateur");
            return -1;
        }
        
        // Vérifier si toutes les completions sont à 0
        bool allZero = true;
        int inProgressIndex = -1;
        
        foreach (var progress in currentUserProgress)
        {
            if (progress.completion > 0)
            {
                allZero = false;
                
                // Si completion > 0 et < 100, c'est la difficulté en cours
                if (progress.completion < 100)
                {
                    // Trouver l'index correspondant dans les options
                    for (int i = 0; i < difficultyOptions.Length; i++)
                    {
                        if (difficultyOptions[i] == progress.difficulty)
                        {
                            inProgressIndex = i;
                            inProgressDifficultyIndex = i; // Stocker dans la variable de classe
                            Debug.Log($"[QuestPreviewPanel] Difficulté en cours trouvée: {progress.difficulty} (completion={progress.completion}%, index={i})");
                            break;
                        }
                    }
                }
            }
        }
        
        // Si toutes les completions sont à 0, ne rien sélectionner
        if (allZero)
        {
            Debug.Log($"[QuestPreviewPanel] Toutes les completions sont à 0 - aucune sélection");
            return -1;
        }
        
        // Si une difficulté est en cours (completion > 0 et < 100), la sélectionner
        if (inProgressIndex >= 0)
        {
            return inProgressIndex;
        }
        
        // Sinon (toutes les difficultés commencées sont à 100%), 
        // utiliser la difficulté sauvegardée localement ou ne rien sélectionner
        string savedDifficulty = UserDataManager.Instance?.GetQuestDifficulty(currentQuestId, null);
        if (!string.IsNullOrEmpty(savedDifficulty))
        {
            int savedIndex = System.Array.IndexOf(difficultyOptions, savedDifficulty);
            if (savedIndex >= 0)
            {
                Debug.Log($"[QuestPreviewPanel] Utilisation de la difficulté sauvegardée: {savedDifficulty}");
                return savedIndex;
            }
        }
        
        return -1;
    }
    
    void SelectDifficulty(int index)
    {
        var diffConfig = config.infoCards.difficulty;
        
        // Couleurs
        Color normalBgColor = HexToColor("#f5ece5");
        Color selectedBgColor = HexToColor("#a95bfb"); // Violet
        Color normalTextColor = HexToColor("#64477f");
        Color selectedTextColor = Color.white;
        
        // Mettre à jour l'apparence de tous les boutons
        for (int i = 0; i < difficultyButtons.Count; i++)
        {
            Button btn = difficultyButtons[i];
            Image bgImage = btn.GetComponent<Image>();
            TextMeshProUGUI btnText = btn.GetComponentInChildren<TextMeshProUGUI>();
            
            if (i == index)
            {
                // Sélectionné
                bgImage.color = selectedBgColor;
                if (btnText != null) btnText.color = selectedTextColor;
            }
            else
            {
                // Non sélectionné
                bgImage.color = normalBgColor;
                if (btnText != null) btnText.color = normalTextColor;
            }
        }
        
        selectedDifficultyIndex = index;
        
        // Sauvegarder la sélection
        if (index >= 0 && index < diffConfig.options.Length)
        {
            string difficulty = diffConfig.options[index];
            UserDataManager.Instance.SetQuestDifficulty(currentQuestId, difficulty);
            Debug.Log($"[QuestPreviewPanel] 💾 Difficulté '{difficulty}' sélectionnée et sauvegardée");
        }
        
        // Mettre à jour l'état du bouton démarrer
        UpdateStartButtonState();
    }
    
    void CreateFormationsCard(Transform parent)
    {
        var cardConfig = config.formationsCard;
        var cardsConfig = config.infoCards;
        var formationsConfig = cardsConfig.formations;
        
        // Créer la carte avec positionnement absolu
        GameObject card = new GameObject("FormationsCard");
        card.transform.SetParent(parent, false);
        
        RectTransform cardRect = card.AddComponent<RectTransform>();
        cardRect.anchorMin = new Vector2(0, 1); // Ancrer en haut à gauche
        cardRect.anchorMax = new Vector2(0, 1);
        cardRect.pivot = new Vector2(0, 1);
        cardRect.anchoredPosition = new Vector2(cardConfig.x, -cardConfig.y);
        cardRect.sizeDelta = new Vector2(cardConfig.width, cardConfig.height);
        
        // Image de fond
        Image bgImage = card.AddComponent<Image>();
        bgImage.sprite = CreateRoundedColorSprite((int)cardConfig.width, (int)cardConfig.height, 
            cardConfig.cornerRadius, HexToColor(cardConfig.backgroundColor));
        bgImage.type = Image.Type.Simple;
        bgImage.raycastTarget = false;
        
        // Ombre
        Shadow shadow = card.AddComponent<Shadow>();
        shadow.effectColor = HexToColor(cardConfig.shadowColor);
        shadow.effectDistance = new Vector2(0, -cardConfig.shadowOffsetY);
        
        Debug.Log($"[QuestPreviewPanel] Carte Formations créée - Position: ({cardConfig.x}, {cardConfig.y}), Taille: {cardConfig.width}x{cardConfig.height}");
        
        // Pas de VerticalLayoutGroup sur la carte - positionnement manuel
        
        // Titre avec marge configurable
        GameObject labelObj = new GameObject("Label");
        labelObj.transform.SetParent(card.transform, false);
        
        RectTransform labelRect = labelObj.AddComponent<RectTransform>();
        labelRect.anchorMin = new Vector2(0, 1);
        labelRect.anchorMax = new Vector2(1, 1);
        labelRect.pivot = new Vector2(0.5f, 1);
        labelRect.anchoredPosition = new Vector2(0, -cardConfig.padding);
        labelRect.sizeDelta = new Vector2(-cardConfig.padding * 2, 30);
        
        TextMeshProUGUI labelText = labelObj.AddComponent<TextMeshProUGUI>();
        labelText.text = formationsConfig.label;
        labelText.fontSize = config.infoCards.cardTitle.fontSize;
        labelText.color = HexToColor(config.infoCards.cardTitle.textColor);
        labelText.alignment = TextAlignmentOptions.Center;
        labelText.fontStyle = FontStyles.Normal;
        if (titleFont != null) labelText.font = titleFont;
        
        // Container pour les items de formation
        GameObject itemsContainer = new GameObject("ItemsContainer");
        itemsContainer.transform.SetParent(card.transform, false);
        
        RectTransform containerRect = itemsContainer.AddComponent<RectTransform>();
        containerRect.anchorMin = new Vector2(0, 1);
        containerRect.anchorMax = new Vector2(1, 1);
        containerRect.pivot = new Vector2(0.5f, 1);
        float topOffset = cardConfig.padding + 30 + formationsConfig.labelMarginBottom; // padding + hauteur label + marge
        containerRect.anchoredPosition = new Vector2(0, -topOffset);
        containerRect.sizeDelta = new Vector2(-cardConfig.padding * 2, cardConfig.height - topOffset - cardConfig.padding);
        
        VerticalLayoutGroup layout = itemsContainer.AddComponent<VerticalLayoutGroup>();
        layout.spacing = formationsConfig.itemSpacing;
        layout.padding = new RectOffset(0, 0, 0, 0);
        layout.childAlignment = TextAnchor.UpperLeft;
        layout.childControlWidth = true;
        layout.childControlHeight = false;
        layout.childForceExpandWidth = true;
        layout.childForceExpandHeight = false;
        
        Debug.Log($"[QuestPreviewPanel] Formations: labelMarginBottom={formationsConfig.labelMarginBottom}px, itemSpacing={formationsConfig.itemSpacing}px");
        
        // Liste des formations (utiliser currentTrainings si disponible, sinon fallback)
        if (currentTrainings != null && currentTrainings.Count > 0)
        {
            foreach (Training training in currentTrainings)
            {
                CreateFormationItem(itemsContainer.transform, training.title, training.url);
            }
        }
        else if (currentFormations != null && currentFormations.Length > 0)
        {
            foreach (string formation in currentFormations)
            {
                CreateFormationItem(itemsContainer.transform, formation, "");
            }
        }
        else if (formationsConfig.items != null)
        {
            foreach (string formation in formationsConfig.items)
            {
                CreateFormationItem(itemsContainer.transform, formation, "");
            }
        }
    }
    
    void CreateFormationItem(Transform parent, string text, string url)
    {
        var formationsConfig = config.infoCards.formations;
        
        GameObject item = new GameObject("FormationItem");
        item.transform.SetParent(parent, false);
        
        RectTransform itemRect = item.AddComponent<RectTransform>();
        itemRect.sizeDelta = new Vector2(0, formationsConfig.itemHeight);
        
        // Ajouter LayoutElement avec hauteur fixe
        LayoutElement itemLayoutElement = item.AddComponent<LayoutElement>();
        itemLayoutElement.preferredHeight = formationsConfig.itemHeight;
        itemLayoutElement.minHeight = formationsConfig.itemHeight;
        itemLayoutElement.flexibleHeight = 0;
        
        HorizontalLayoutGroup layout = item.AddComponent<HorizontalLayoutGroup>();
        layout.spacing = 5;
        layout.padding = new RectOffset(0, 0, 0, 0);
        layout.childAlignment = TextAnchor.MiddleLeft;
        layout.childControlWidth = true;
        layout.childControlHeight = true;
        layout.childForceExpandWidth = false;
        
        // Bullet/Arrow - Image PNG (40x40)
        GameObject bullet = new GameObject("Bullet");
        bullet.transform.SetParent(item.transform, false);
        
        RectTransform bulletRect = bullet.AddComponent<RectTransform>();
        bulletRect.sizeDelta = new Vector2(40, 40);
        
        // Ajouter LayoutElement pour fixer la taille de la puce (FIXE 40x40)
        LayoutElement bulletLayout = bullet.AddComponent<LayoutElement>();
        bulletLayout.minWidth = 40;
        bulletLayout.preferredWidth = 40;
        bulletLayout.minHeight = 40;
        bulletLayout.preferredHeight = 40;
        bulletLayout.flexibleWidth = 0;   // Ne pas étirer horizontalement
        bulletLayout.flexibleHeight = 0;  // Ne pas étirer verticalement
        
        Image bulletImage = bullet.AddComponent<Image>();
        bulletImage.preserveAspect = true;
        
        // Charger l'image de la flèche
        StartCoroutine(LoadFormationArrowSprite(bulletImage));
        
        // Texte (cliquable si URL fournie) - prend toute la largeur restante
        GameObject textObj = new GameObject("Text");
        textObj.transform.SetParent(item.transform, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        
        // Ajouter LayoutElement pour que le texte prenne toute la largeur disponible
        LayoutElement textLayout = textObj.AddComponent<LayoutElement>();
        textLayout.flexibleWidth = 1;  // Prendre toute la largeur disponible
        // Pas de hauteur fixe - s'adapte au contenu
        
        TextMeshProUGUI textComp = textObj.AddComponent<TextMeshProUGUI>();
        textComp.text = text;
        
        // Utiliser la config des formations pour le style
        textComp.fontSize = formationsConfig.fontSize;
        
        // Couleur et style selon si c'est un lien ou non
        bool isLink = !string.IsNullOrEmpty(url);
        textComp.color = HexToColor(isLink ? formationsConfig.linkColor : formationsConfig.textColor);
        
        textComp.alignment = TextAlignmentOptions.TopLeft;  // Alignement en haut à gauche
        textComp.textWrappingMode = TextWrappingModes.Normal;  // Activer le retour à la ligne
        textComp.overflowMode = TextOverflowModes.Overflow;  // Permettre au texte de s'étendre verticalement
        textComp.margin = new Vector4(0, 0, 0, 0);  // Pas de marge
        textComp.enableAutoSizing = false;  // Pas de redimensionnement automatique
        
        // Souligner si c'est un lien et que l'option est activée
        if (isLink && formationsConfig.underlineLinks)
        {
            textComp.fontStyle = FontStyles.Underline;
        }
        
        if (textFont != null) textComp.font = textFont;
        
        // Rendre cliquable si URL fournie
        if (isLink)
        {
            // Ajouter un bouton pour rendre le texte cliquable
            Button button = textObj.AddComponent<Button>();
            button.transition = Selectable.Transition.ColorTint;
            
            ColorBlock colors = button.colors;
            colors.normalColor = Color.white;
            colors.highlightedColor = new Color(0.95f, 0.9f, 1f, 1f); // Légèrement teinté au survol
            colors.pressedColor = new Color(0.85f, 0.75f, 0.95f, 1f);
            button.colors = colors;
            
            button.onClick.AddListener(() => {
                Debug.Log($"[QuestPreviewPanel] Ouverture de la formation: {text} → {url}");
                Application.OpenURL(url);
            });
            
            // Changer le style du texte pour indiquer qu'il est cliquable
            textComp.fontStyle = FontStyles.Underline;
        }
    }
    
    // OBSOLÈTE - Utilise maintenant le positionnement absolu dans CreateDurationDifficultyCard et CreateFormationsCard
    
    void CreateCardLabel(Transform parent, string text)
    {
        GameObject labelObj = new GameObject("Label");
        labelObj.transform.SetParent(parent, false);
        
        RectTransform rect = labelObj.AddComponent<RectTransform>();
        rect.sizeDelta = new Vector2(320, 30);  // Augmenté pour tenir "NIVEAU DE DIFFICULTÉ" sur une ligne
        
        // Ajouter LayoutElement pour contrôler la hauteur exacte
        LayoutElement layoutElement = labelObj.AddComponent<LayoutElement>();
        layoutElement.preferredHeight = 30;  // Hauteur exacte sans expansion
        layoutElement.minHeight = 30;
        
        TextMeshProUGUI labelText = labelObj.AddComponent<TextMeshProUGUI>();
        labelText.text = text;
        labelText.fontSize = config.infoCards.cardTitle.fontSize;
        labelText.color = HexToColor(config.infoCards.cardTitle.textColor);
        labelText.alignment = TextAlignmentOptions.Center;
        labelText.fontStyle = FontStyles.Normal;
        labelText.textWrappingMode = TextWrappingModes.NoWrap;  // Pas de retour à la ligne
        labelText.overflowMode = TextOverflowModes.Overflow;  // Permettre le débordement si nécessaire
        labelText.margin = new Vector4(0, 0, 0, 0);  // Pas de marge
        if (titleFont != null) labelText.font = titleFont;
    }
    
    void CreateCardValue(Transform parent, string text)
    {
        GameObject valueObj = new GameObject("Value");
        valueObj.transform.SetParent(parent, false);
        
        RectTransform rect = valueObj.AddComponent<RectTransform>();
        rect.sizeDelta = new Vector2(200, 30);
        
        TextMeshProUGUI valueText = valueObj.AddComponent<TextMeshProUGUI>();
        valueText.text = text;
        valueText.fontSize = config.infoCards.cardText.fontSize + 4;
        valueText.color = HexToColor(config.infoCards.cardText.textColor);
        valueText.alignment = TextAlignmentOptions.Center;
        if (textFont != null) valueText.font = textFont;
    }
    
    void CreateVideoColumn(Transform parent)
    {
        var videoConfig = config.videoPlayer;
        
        GameObject column = new GameObject("VideoColumn");
        column.transform.SetParent(parent, false);
        
        // Positionnement absolu
        RectTransform columnRect = column.AddComponent<RectTransform>();
        columnRect.anchorMin = new Vector2(0, 1); // Ancrer en haut à gauche
        columnRect.anchorMax = new Vector2(0, 1);
        columnRect.pivot = new Vector2(0, 1);
        columnRect.anchoredPosition = new Vector2(videoConfig.x, -videoConfig.y);
        columnRect.sizeDelta = new Vector2(videoConfig.width, videoConfig.height);
        
        Debug.Log($"[QuestPreviewPanel] Vidéo créée - Position: ({videoConfig.x}, {videoConfig.y}), Taille: {videoConfig.width}x{videoConfig.height}, Radius: {videoConfig.cornerRadius}");
        
        Image bgImage = column.AddComponent<Image>();
        bgImage.sprite = CreateRoundedColorSprite((int)videoConfig.width, (int)videoConfig.height, 
            videoConfig.cornerRadius, new Color(0.2f, 0.2f, 0.3f));
        bgImage.type = Image.Type.Simple;
        bgImage.raycastTarget = true; // Bloquer les clics
        
        // Masque pour les coins arrondis de la vidéo
        Mask mask = column.AddComponent<Mask>();
        mask.showMaskGraphic = true;
        
        // Video Player - utiliser la vidéo par défaut si aucune vidéo n'est fournie
        string videoUrl = !string.IsNullOrEmpty(currentVideoUrl) 
            ? currentVideoUrl 
            : "https://ujsa.studioplc.fr/datas/videos/Q00_intro.mp4";
        
        Debug.Log($"[QuestPreviewPanel] 🎬 URL vidéo: {videoUrl} (défaut: {string.IsNullOrEmpty(currentVideoUrl)})");
        
        VideoPlayer videoPlayer = column.AddComponent<VideoPlayer>();
        videoPlayer.source = VideoSource.Url;
        videoPlayer.url = videoUrl;
        videoPlayer.isLooping = true;
        videoPlayer.playOnAwake = false; // On va démarrer manuellement après la préparation
        videoPlayer.renderMode = VideoRenderMode.RenderTexture;
        videoPlayer.audioOutputMode = VideoAudioOutputMode.None; // Désactiver le son
        videoPlayer.skipOnDrop = true;
        videoPlayer.waitForFirstFrame = true;
        
        // RawImage pour afficher la vidéo
        GameObject videoDisplay = new GameObject("VideoDisplay");
        videoDisplay.transform.SetParent(column.transform, false);
        
        RectTransform videoRect = videoDisplay.AddComponent<RectTransform>();
        videoRect.anchorMin = Vector2.zero;
        videoRect.anchorMax = Vector2.one;
        videoRect.offsetMin = Vector2.zero;
        videoRect.offsetMax = Vector2.zero;
        
        RawImage rawImage = videoDisplay.AddComponent<RawImage>();
        rawImage.color = Color.white;
        
        // Créer une RenderTexture pour la vidéo
        RenderTexture renderTexture = new RenderTexture(1280, 720, 0);
        renderTexture.Create();
        videoPlayer.targetTexture = renderTexture;
        rawImage.texture = renderTexture;
        
        Debug.Log($"[QuestPreviewPanel] 🎬 VideoPlayer configuré - RenderTexture: {renderTexture.width}x{renderTexture.height}");
        
        // Gestion des événements
        videoPlayer.prepareCompleted += (source) => 
        {
            Debug.Log($"[QuestPreviewPanel] ✅ Vidéo préparée, démarrage de la lecture...");
            source.Play();
        };
        
        videoPlayer.started += (source) =>
        {
            Debug.Log($"[QuestPreviewPanel] ▶️ Vidéo démarrée!");
        };
        
        videoPlayer.errorReceived += (source, message) =>
        {
            Debug.LogError($"[QuestPreviewPanel] ❌ Erreur vidéo: {message}");
        };
        
        videoPlayer.loopPointReached += (source) =>
        {
            Debug.Log($"[QuestPreviewPanel] 🔄 Boucle vidéo atteinte");
        };
        
        // Démarrer la préparation
        Debug.Log($"[QuestPreviewPanel] 🎬 Démarrage de la préparation de la vidéo...");
        videoPlayer.Prepare();
    }
    
    void CreateHelpLink(Transform parent)
    {
        var helpConfig = config.helpLink;
        
        GameObject helpObj = new GameObject("HelpLink");
        helpObj.transform.SetParent(parent, false);
        
        // Positionnement absolu
        RectTransform rect = helpObj.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(0, 1); // Ancrer en haut à gauche
        rect.anchorMax = new Vector2(0, 1);
        rect.pivot = new Vector2(0.5f, 0.5f); // Pivot au centre pour centrer le contenu
        rect.anchoredPosition = new Vector2(helpConfig.x + helpConfig.width / 2, -helpConfig.y - helpConfig.height / 2);
        rect.sizeDelta = new Vector2(helpConfig.width, helpConfig.height);
        
        Debug.Log($"[QuestPreviewPanel] Lien d'aide créé - Position: ({helpConfig.x}, {helpConfig.y}), Taille: {helpConfig.width}x{helpConfig.height}");
        
        HorizontalLayoutGroup layout = helpObj.AddComponent<HorizontalLayoutGroup>();
        layout.spacing = 8;
        layout.childAlignment = TextAnchor.MiddleCenter;
        layout.childControlWidth = false;
        layout.childControlHeight = false;
        
        // Icône ?
        GameObject iconObj = new GameObject("Icon");
        iconObj.transform.SetParent(helpObj.transform, false);
        
        RectTransform iconRect = iconObj.AddComponent<RectTransform>();
        iconRect.sizeDelta = new Vector2(helpConfig.iconSize, helpConfig.iconSize);
        
        TextMeshProUGUI iconText = iconObj.AddComponent<TextMeshProUGUI>();
        iconText.text = "?";
        iconText.fontSize = helpConfig.fontSize;
        iconText.color = HexToColor(helpConfig.textColor);
        iconText.alignment = TextAlignmentOptions.Center;
        iconText.fontStyle = FontStyles.Bold;
        
        // Cercle autour du ?
        Outline iconOutline = iconObj.AddComponent<Outline>();
        iconOutline.effectColor = HexToColor(helpConfig.textColor);
        iconOutline.effectDistance = new Vector2(1, -1);
        
        // Texte
        GameObject textObj = new GameObject("Text");
        textObj.transform.SetParent(helpObj.transform, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.sizeDelta = new Vector2(150, 25);
        
        TextMeshProUGUI text = textObj.AddComponent<TextMeshProUGUI>();
        text.text = helpConfig.text;
        text.fontSize = helpConfig.fontSize;
        text.color = HexToColor(helpConfig.textColor);
        text.alignment = TextAlignmentOptions.MidlineLeft;
        text.fontStyle = FontStyles.Bold;
        
        // Bouton
        Button button = helpObj.AddComponent<Button>();
        button.transition = Selectable.Transition.ColorTint;
        button.onClick.AddListener(() => {
            Debug.Log("[QuestPreviewPanel] Aide demandée");
            // TODO: Ouvrir l'aide
        });
    }
    
    void CreateButtons(Transform parent)
    {
        var buttonsConfig = config.buttons;
        
        // Pas de container - positionnement direct des boutons
        
        // Bouton "Passer cette quête"
        CreateActionButton(parent, buttonsConfig.skipQuest, false, () => {
            ClosePanel();
            onSkipQuest?.Invoke();
        });
        
        // Bouton "Commencer la quête"
        var generalConfig = GeneralConfigManager.Instance?.GetConfig();
        var enabledStyle = generalConfig?.buttonStyles?.GetStyle(buttonsConfig.startQuest.style);
        var disabledStyle = generalConfig?.buttonStyles?.GetStyle(buttonsConfig.startQuest.disabledStyle);
        
        // Créer les sprites avec les dimensions du bouton
        if (enabledStyle != null)
        {
            Color startColor = HexToColor(enabledStyle.gradient?.startColor ?? "#b87aff");
            Color endColor = HexToColor(enabledStyle.gradient?.endColor ?? "#7b4fbf");
            Color borderColor = HexToColor(enabledStyle.borderColor ?? "#f5ece5");
            startButtonEnabledSprite = CreateGradientSpriteWithBorder((int)buttonsConfig.startQuest.width, (int)buttonsConfig.startQuest.height, 
                enabledStyle.borderRadius, startColor, endColor, borderColor, enabledStyle.borderWidth);
        }
        
        if (disabledStyle != null)
        {
            Color startColor = HexToColor(disabledStyle.gradient?.startColor ?? "#cccccc");
            Color endColor = HexToColor(disabledStyle.gradient?.endColor ?? "#999999");
            Color borderColor = HexToColor(disabledStyle.borderColor ?? "#f5ece5");
            startButtonDisabledSprite = CreateGradientSpriteWithBorder((int)buttonsConfig.startQuest.width, (int)buttonsConfig.startQuest.height, 
                disabledStyle.borderRadius, startColor, endColor, borderColor, disabledStyle.borderWidth);
        }
        
        startButton = CreateActionButton(parent, buttonsConfig.startQuest, true, () => {
            if (selectedDifficultyIndex >= 0) // Un niveau a été sélectionné
            {
                // Vérifier si l'utilisateur change de difficulté alors qu'une autre est en cours
                if (inProgressDifficultyIndex >= 0 && selectedDifficultyIndex != inProgressDifficultyIndex)
                {
                    // Appeler l'endpoint de reset avant de lancer la quête
                    StartCoroutine(ResetCurrentProgressAndStartQuest());
                }
                else
                {
                    ClosePanel();
                    onStartQuest?.Invoke();
                }
            }
        });
        
        startButtonImage = startButton.GetComponent<Image>();
        startButtonText = startButton.GetComponentInChildren<TextMeshProUGUI>();
        UpdateStartButtonState();
    }
    
    /// <summary>
    /// Appelle l'endpoint de reset de la progression courante puis lance la quête
    /// </summary>
    private IEnumerator ResetCurrentProgressAndStartQuest()
    {
        // Construire l'URL de reset
        string resetUrl = GeneralConfigManager.Instance?.GetQuestResetCurrentApiUrl(currentQuestNumericId);
        
        if (string.IsNullOrEmpty(resetUrl))
        {
            Debug.LogWarning($"[QuestPreviewPanel] ⚠️ URL de reset invalide, lancement direct de la quête");
            ClosePanel();
            onStartQuest?.Invoke();
            yield break;
        }
        
        Debug.Log($"[QuestPreviewPanel] 🔄 Changement de difficulté détecté (de index {inProgressDifficultyIndex} vers {selectedDifficultyIndex})");
        Debug.Log($"[QuestPreviewPanel] 📡 Appel POST vers: {resetUrl}");
        
        // Désactiver le bouton pendant l'appel
        if (startButton != null)
        {
            startButton.interactable = false;
        }
        
        using (UnityWebRequest request = new UnityWebRequest(resetUrl, "POST"))
        {
            request.downloadHandler = new DownloadHandlerBuffer();
            request.SetRequestHeader("Content-Type", "application/json");
            
            // Ajouter le token d'authentification si disponible
            string authToken = UserDataManager.Instance?.token;
            if (!string.IsNullOrEmpty(authToken))
            {
                request.SetRequestHeader("Authorization", $"Bearer {authToken}");
            }
            
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Debug.Log($"[QuestPreviewPanel] ✅ Reset de la progression réussi: {request.downloadHandler.text}");
            }
            else
            {
                Debug.LogWarning($"[QuestPreviewPanel] ⚠️ Erreur lors du reset: {request.error} - Lancement de la quête quand même");
            }
        }
        
        // Réactiver le bouton (au cas où on reste sur le panel)
        if (startButton != null)
        {
            startButton.interactable = true;
        }
        
        // Lancer la quête
        ClosePanel();
        onStartQuest?.Invoke();
    }
    
    Button CreateActionButton(Transform parent, QuestPreviewButtonConfig buttonConfig, bool isStartButton, Action onClick)
    {
        var generalConfig = GeneralConfigManager.Instance?.GetConfig();
        var buttonStyle = generalConfig?.buttonStyles?.GetStyle(buttonConfig.style);
        var buttonsConfig = config.buttons;
        
        GameObject buttonObj = new GameObject(isStartButton ? "StartButton" : "SkipButton");
        buttonObj.transform.SetParent(parent, false);
        
        // Positionnement absolu
        RectTransform rect = buttonObj.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(0, 1); // Ancrer en haut à gauche
        rect.anchorMax = new Vector2(0, 1);
        rect.pivot = new Vector2(0, 1);
        rect.anchoredPosition = new Vector2(buttonConfig.x, -buttonsConfig.y);
        rect.sizeDelta = new Vector2(buttonConfig.width, buttonConfig.height);
        
        Debug.Log($"[QuestPreviewPanel] Bouton créé: {buttonConfig.label} - Position: ({buttonConfig.x}, {buttonsConfig.y}), Taille: {buttonConfig.width}x{buttonConfig.height}");
        
        Image image = buttonObj.AddComponent<Image>();
        
        // Utiliser le style du bouton avec les dimensions du config
        if (buttonStyle != null)
        {
            Color startColor = HexToColor(buttonStyle.gradient?.startColor ?? "#b87aff");
            Color endColor = HexToColor(buttonStyle.gradient?.endColor ?? "#7b4fbf");
            Color borderColor = HexToColor(buttonStyle.borderColor ?? "#f5ece5");
            
            image.sprite = CreateGradientSpriteWithBorder((int)buttonConfig.width, (int)buttonConfig.height, 
                buttonStyle.borderRadius, startColor, endColor, borderColor, buttonStyle.borderWidth);
            image.color = Color.white;
        }
        else
        {
            image.sprite = CreateRoundedColorSprite((int)buttonConfig.width, (int)buttonConfig.height, 25, HexToColor("#a95bfb"));
            image.color = Color.white;
        }
        image.type = Image.Type.Simple;
        image.raycastTarget = true; // S'assurer que le bouton bloque les clics
        
        Button button = buttonObj.AddComponent<Button>();
        button.targetGraphic = image;
        button.onClick.AddListener(() => onClick?.Invoke());
        
        // Texte du bouton
        GameObject textObj = new GameObject("Text");
        textObj.transform.SetParent(buttonObj.transform, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.offsetMin = Vector2.zero;
        textRect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI text = textObj.AddComponent<TextMeshProUGUI>();
        text.text = buttonConfig.label;
        text.fontSize = buttonStyle?.text?.fontSize ?? 22;
        text.color = HexToColor(buttonStyle?.text?.color ?? "#FFFFFF");
        text.alignment = TextAlignmentOptions.Center;
        text.fontStyle = FontStyles.Normal;
        if (titleFont != null) text.font = titleFont;
        
        return button;
    }
    
    void UpdateStartButtonState()
    {
        if (startButton == null || startButtonImage == null) return;
        
        bool isEnabled = selectedDifficultyIndex >= 0;
        startButton.interactable = isEnabled;
        
        // Changer le sprite selon l'état (enabled vs disabled)
        if (isEnabled && startButtonEnabledSprite != null)
        {
            startButtonImage.sprite = startButtonEnabledSprite;
            startButtonImage.color = Color.white;
        }
        else if (!isEnabled && startButtonDisabledSprite != null)
        {
            startButtonImage.sprite = startButtonDisabledSprite;
            startButtonImage.color = Color.white;
        }
        
        Debug.Log($"[QuestPreviewPanel] Bouton état: {(isEnabled ? "ENABLED (validationPurple)" : "DISABLED (secondaryAction)")}");
    }
    
    // ========== Utilitaires ==========
    
    // Méthode copiée de SettingsPanelBuilder - avec anti-aliasing propre
    Sprite CreateRoundedColorSprite(int width, int height, float radius, Color fillColor)
    {
        Texture2D texture = new Texture2D(width, height);
        Color[] pixels = new Color[width * height];

        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                float alpha = 1f;

                // Coin supérieur gauche
                if (x < radius && y > height - radius)
                {
                    float dx = radius - x;
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius)
                        alpha = 0f;
                    else
                        alpha = 1f - Mathf.Clamp01((distance - radius + 1) / 1);
                }
                // Coin supérieur droit
                else if (x > width - radius && y > height - radius)
                {
                    float dx = x - (width - radius);
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius)
                        alpha = 0f;
                    else
                        alpha = 1f - Mathf.Clamp01((distance - radius + 1) / 1);
                }
                // Coin inférieur gauche
                else if (x < radius && y < radius)
                {
                    float dx = radius - x;
                    float dy = radius - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius)
                        alpha = 0f;
                    else
                        alpha = 1f - Mathf.Clamp01((distance - radius + 1) / 1);
                }
                // Coin inférieur droit
                else if (x > width - radius && y < radius)
                {
                    float dx = x - (width - radius);
                    float dy = radius - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius)
                        alpha = 0f;
                    else
                        alpha = 1f - Mathf.Clamp01((distance - radius + 1) / 1);
                }

                pixels[y * width + x] = new Color(fillColor.r, fillColor.g, fillColor.b, fillColor.a * alpha);
            }
        }

        texture.SetPixels(pixels);
        texture.Apply();
        
        dynamicTextures.Add(texture);

        return Sprite.Create(texture, new Rect(0, 0, width, height), new Vector2(0.5f, 0.5f), 100f);
    }
    
    // Méthode copiée de SettingsPanelBuilder - coins arrondis uniquement en HAUT
    Sprite CreateRoundedTopSprite(int width, int height, float radius, Color fillColor)
    {
        Texture2D texture = new Texture2D(width, height);
        Color[] pixels = new Color[width * height];

        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                float alpha = 1f;

                // Seulement les coins du HAUT sont arrondis
                // Coin supérieur gauche
                if (x < radius && y > height - radius)
                {
                    float dx = radius - x;
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius)
                        alpha = 0f;
                    else
                        alpha = 1f - Mathf.Clamp01((distance - radius + 1) / 1);
                }
                // Coin supérieur droit
                else if (x > width - radius && y > height - radius)
                {
                    float dx = x - (width - radius);
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius)
                        alpha = 0f;
                    else
                        alpha = 1f - Mathf.Clamp01((distance - radius + 1) / 1);
                }
                // Coins inférieurs : pas arrondis (alpha reste à 1)

                pixels[y * width + x] = new Color(fillColor.r, fillColor.g, fillColor.b, fillColor.a * alpha);
            }
        }

        texture.SetPixels(pixels);
        texture.Apply();
        
        dynamicTextures.Add(texture);

        return Sprite.Create(texture, new Rect(0, 0, width, height), new Vector2(0.5f, 0.5f), 100f);
    }
    
    Sprite CreateGradientSpriteWithBorder(int width, int height, float radius, Color startColor, Color endColor, Color borderColor, float borderWidth)
    {
        Texture2D texture = new Texture2D(width, height);
        Color[] pixels = new Color[width * height];

        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                float alpha = 1f;
                bool isBorder = false;
                
                // Vérifier si on est dans la bordure
                if (x < borderWidth || x >= width - borderWidth || y < borderWidth || y >= height - borderWidth)
                {
                    isBorder = true;
                }

                // Coins arrondis
                float distanceFromCorner = float.MaxValue;
                
                // Coin supérieur gauche
                if (x < radius && y > height - radius)
                {
                    float dx = radius - x;
                    float dy = (height - radius) - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }
                // Coin supérieur droit
                else if (x > width - radius && y > height - radius)
                {
                    float dx = x - (width - radius);
                    float dy = (height - radius) - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }
                // Coin inférieur gauche
                else if (x < radius && y < radius)
                {
                    float dx = radius - x;
                    float dy = radius - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }
                // Coin inférieur droit
                else if (x > width - radius && y < radius)
                {
                    float dx = x - (width - radius);
                    float dy = radius - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }

                Color pixelColor;
                if (isBorder && alpha > 0)
                {
                    pixelColor = borderColor;
                }
                else
                {
                    // Dégradé vertical
                    float t = (float)y / height;
                    pixelColor = Color.Lerp(endColor, startColor, t);
                }
                
                pixels[y * width + x] = new Color(pixelColor.r, pixelColor.g, pixelColor.b, pixelColor.a * alpha);
            }
        }

        texture.SetPixels(pixels);
        texture.Apply();
        
        // Tracker la texture pour la nettoyer plus tard
        dynamicTextures.Add(texture);

        return Sprite.Create(texture, new Rect(0, 0, width, height), new Vector2(0.5f, 0.5f), 100f);
    }
    
    Color HexToColor(string hex)
    {
        if (string.IsNullOrEmpty(hex)) return Color.white;
        
        hex = hex.TrimStart('#');
        
        if (hex.Length == 6)
        {
            hex = hex + "FF";
        }
        
        if (hex.Length == 8)
        {
            byte r = byte.Parse(hex.Substring(0, 2), System.Globalization.NumberStyles.HexNumber);
            byte g = byte.Parse(hex.Substring(2, 2), System.Globalization.NumberStyles.HexNumber);
            byte b = byte.Parse(hex.Substring(4, 2), System.Globalization.NumberStyles.HexNumber);
            byte a = byte.Parse(hex.Substring(6, 2), System.Globalization.NumberStyles.HexNumber);
            
            return new Color(r / 255f, g / 255f, b / 255f, a / 255f);
        }
        
        return Color.white;
    }
}

