using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Networking;
using TMPro;
using System;
using System.Collections;
using System.Collections.Generic;

/// <summary>
/// Onglet QUÊTES dans le panneau des paramètres
/// Affiche la liste des quêtes à gauche et les détails à droite (personnages, badges, formations)
/// </summary>
public class QuetesTab : SettingsTab
{
    private QuetesTabConfig config;
    private GeneralConfig generalConfig;
    
    // Références aux blocs
    private GameObject questListBlock;
    private GameObject detailsBlock;
    
    // Contenu des détails
    private GameObject detailsHeader;
    private GameObject detailsContent;
    
    // Onglets des détails
    private List<Button> detailTabButtons = new List<Button>();
    private int activeDetailTab = 0;
    
    // Données
    private List<Quest> questsData = new List<Quest>();
    private Quest selectedQuest;
    private Dictionary<int, List<QuestCharacter>> questCharacters = new Dictionary<int, List<QuestCharacter>>();
    
    // Références aux items de quête pour la sélection
    private Dictionary<int, GameObject> questItemsMap = new Dictionary<int, GameObject>();
    
    // Mosaïque de personnages
    private List<QuestCharacter> currentCharacters = new List<QuestCharacter>();
    private int selectedCharacterIndex = 0;
    private int visibleStartIndex = 0; // Index du premier personnage visible
    private List<GameObject> avatarButtons = new List<GameObject>();
    private GameObject avatarsContainer;
    private GameObject characterDetailPanel;
    private TextMeshProUGUI characterNameText;
    private TextMeshProUGUI characterDescriptionText;
    private TextMeshProUGUI characterLongDescriptionText;
    
    // Sprites et polices
    private TMP_FontAsset titleFont;
    private TMP_FontAsset textFont;
    private List<Texture2D> dynamicTextures = new List<Texture2D>();
    
    protected override void Awake()
    {
        base.Awake();
        
        Debug.Log("[QuetesTab] Awake() appelé");
        
        // Charger la configuration
        if (GeneralConfigManager.Instance != null)
        {
            generalConfig = GeneralConfigManager.Instance.GetConfig();
            config = generalConfig?.quetesTabConfig;
            
            if (config != null)
            {
                Debug.Log("[QuetesTab] ✅ Configuration chargée");
                
                // Charger les polices
                LoadFonts();
                
                // Démarrer la construction de l'interface
                StartCoroutine(BuildInterface());
            }
            else
            {
                Debug.LogError("[QuetesTab] ❌ Configuration quetesTabConfig non trouvée!");
            }
        }
        else
        {
            Debug.LogError("[QuetesTab] ❌ GeneralConfigManager.Instance est null!");
        }
    }
    
    private bool isInitialized = false;
    
    protected override void OnEnable()
    {
        base.OnEnable();
        
        Debug.Log("[QuetesTab] OnEnable() - Onglet QUÊTES activé");
        
        // Si l'interface n'est pas encore construite, attendre
        if (questListBlock == null)
        {
            Debug.Log("[QuetesTab] ⚠️ Interface pas encore construite, attente...");
            StartCoroutine(WaitForInterfaceAndPopulate());
            return;
        }
        
        // Ne recharger que si pas encore initialisé
        if (!isInitialized)
        {
            Debug.Log("[QuetesTab] 🔄 Premier chargement des données...");
            LoadQuestsData();
            PopulateQuestList();
            isInitialized = true;
        }
        else
        {
            Debug.Log("[QuetesTab] ✅ Déjà initialisé, pas de rechargement");
        }
    }
    
    IEnumerator WaitForInterfaceAndPopulate()
    {
        // Attendre que l'interface soit construite
        int maxWait = 50; // 5 secondes max
        int waited = 0;
        
        while (questListBlock == null && waited < maxWait)
        {
            yield return new WaitForSeconds(0.1f);
            waited++;
        }
        
        if (questListBlock != null && !isInitialized)
        {
            Debug.Log($"[QuetesTab] ✅ Interface prête après {waited * 0.1f}s");
            LoadQuestsData();
            PopulateQuestList();
            isInitialized = true;
        }
        else if (questListBlock == null)
        {
            Debug.LogError($"[QuetesTab] ❌ Interface toujours pas prête après {maxWait * 0.1f}s!");
        }
    }
    
    void OnDestroy()
    {
        // Nettoyer les textures dynamiques
        foreach (Texture2D tex in dynamicTextures)
        {
            if (tex != null)
            {
                Destroy(tex);
            }
        }
        dynamicTextures.Clear();
    }
    
    void LoadFonts()
    {
        string titleFontName = config?.fonts?.titleFontName ?? "Anton-Regular SDF";
        string textFontName = config?.fonts?.textFontName ?? "Lato-Regular SDF";
        
        titleFont = Resources.Load<TMP_FontAsset>($"Fonts/{titleFontName}");
        if (titleFont == null)
            titleFont = Resources.Load<TMP_FontAsset>(titleFontName);
            
        textFont = Resources.Load<TMP_FontAsset>($"Fonts/{textFontName}");
        if (textFont == null)
            textFont = Resources.Load<TMP_FontAsset>(textFontName);
            
        if (titleFont != null)
            Debug.Log($"[QuetesTab] ✅ Police titre chargée: {titleFont.name}");
        if (textFont != null)
            Debug.Log($"[QuetesTab] ✅ Police texte chargée: {textFont.name}");
    }
    
    void LoadQuestsData()
    {
        Debug.Log("[QuetesTab] 📥 LoadQuestsData() appelé");
        
        // Vider la liste existante pour recharger
        questsData.Clear();
        Debug.Log("[QuetesTab] 🗑️ Liste des quêtes vidée");
        
        // Charger les données des quêtes depuis l'API
        // Essayer de trouver MainSceneManager dans la scène
        MainSceneManager mainSceneManager = FindFirstObjectByType<MainSceneManager>();
        
        Debug.Log($"[QuetesTab] MainSceneManager trouvé: {mainSceneManager != null}");
        
        if (mainSceneManager != null)
        {
            Debug.Log($"[QuetesTab] MainSceneManager: {mainSceneManager.name}");
            
            // Accéder aux données via réflexion car sceneConfig est privé
            var sceneConfigField = mainSceneManager.GetType().GetField("sceneConfig", 
                System.Reflection.BindingFlags.NonPublic | System.Reflection.BindingFlags.Instance);
            
            Debug.Log($"[QuetesTab] sceneConfigField trouvé: {sceneConfigField != null}");
            
            if (sceneConfigField != null)
            {
                MainSceneConfig sceneConfig = sceneConfigField.GetValue(mainSceneManager) as MainSceneConfig;
                Debug.Log($"[QuetesTab] sceneConfig récupéré: {sceneConfig != null}");
                
                if (sceneConfig != null)
                {
                    Debug.Log($"[QuetesTab] sceneConfig.quests: {sceneConfig.quests != null}");
                    Debug.Log($"[QuetesTab] sceneConfig.quests.Count: {sceneConfig.quests?.Count ?? 0}");
                }
                
                if (sceneConfig != null && sceneConfig.quests != null && sceneConfig.quests.Count > 0)
                {
                    // Filtrer uniquement les quêtes où has_access == true
                    questsData = new List<Quest>();
                    foreach (Quest q in sceneConfig.quests)
                    {
                        if (q.has_access)
                        {
                            questsData.Add(q);
                            Debug.Log($"[QuetesTab] 📋 Quête {q.id}: {q.title} (has_access={q.has_access}, status={q.status ?? "null"})");
                            Debug.Log($"[QuetesTab]    Description: {q.description}");
                            Debug.Log($"[QuetesTab]    Durée: {q.duration}");
                            Debug.Log($"[QuetesTab]    Formations: {q.trainings?.Count ?? 0}");
                        }
                        else
                        {
                            Debug.Log($"[QuetesTab] ⚠️ Quête {q.id} ignorée (has_access=false)");
                        }
                    }
                    Debug.Log($"[QuetesTab] ✅ {questsData.Count} quêtes avec accès chargées depuis MainSceneManager");
                }
                else
                {
                    Debug.LogWarning($"[QuetesTab] ⚠️ sceneConfig trouvé mais pas de quêtes (quests = {sceneConfig?.quests?.Count ?? 0})");
                }
            }
            else
            {
                Debug.LogWarning("[QuetesTab] ⚠️ Impossible d'accéder au champ sceneConfig via réflexion");
            }
        }
        else
        {
            Debug.LogWarning("[QuetesTab] ⚠️ MainSceneManager non trouvé dans la scène (probablement dans MapManager)");
        }
        
        // Fallback: créer quelques quêtes d'exemple si aucune donnée trouvée
        if (questsData.Count == 0)
        {
            Debug.LogWarning("[QuetesTab] ⚠️ Aucune donnée de l'API, création du fallback...");
            CreateFallbackQuests();
            Debug.Log($"[QuetesTab] ✅ {questsData.Count} quêtes de fallback créées");
        }
        
        Debug.Log($"[QuetesTab] 📊 Total final: {questsData.Count} quêtes");
        
        // Charger les personnages des quêtes
        StartCoroutine(LoadQuestCharacters());
    }
    
    void CreateFallbackQuests()
    {
        Debug.Log("[QuetesTab] 🔨 Création de quêtes de fallback...");
        
        // Créer des quêtes d'exemple pour le fallback avec des titres réalistes
        string[] titles = new string[]
        {
            "QUÊTE 1 : LES DENTS DE L'AMER",
            "QUÊTE 2 : LA MARCHE DES ZOMBIES",
            "QUÊTE 3 : JURASSIC VILLE",
            "QUÊTE 4 : LE CLAN DES CIGARES",
            "QUÊTE 5 : BULLES SUR LA VILLE"
        };
        
        for (int i = 0; i < titles.Length; i++)
        {
            var quest = new Quest
            {
                id = i + 1,
                title = titles[i],
                description = $"Les rideaux de fer tombent sur la ville... Découvrez pourquoi les habitants du quartier sont sur les dents ! C'est une aventure passionnante qui vous attend dans cette quête {i + 1}.",
                duration = "2 H 00",
                trainings = new List<Training>
                {
                    new Training { title = "Comprendre le régime d'assurance maladie", url = "" },
                    new Training { title = "Comprendre l'assurance complémentaire santé", url = "" }
                }
            };
            
            questsData.Add(quest);
            Debug.Log($"[QuetesTab] ✅ Quête fallback créée: {quest.title}");
        }
        
        Debug.Log($"[QuetesTab] ✅ {questsData.Count} quêtes de fallback créées au total");
    }
    
    IEnumerator LoadQuestCharacters()
    {
        // Charger les personnages depuis persos.json
        string persosJsonPath = Application.streamingAssetsPath + "/persos.json";
        
        if (System.IO.File.Exists(persosJsonPath))
        {
            string json = System.IO.File.ReadAllText(persosJsonPath);
            CharactersData charactersData = JsonUtility.FromJson<CharactersData>(json);
            
            if (charactersData != null && charactersData.personnages != null)
            {
                Debug.Log($"[QuetesTab] ✅ {charactersData.personnages.Count} personnages chargés depuis persos.json");
                
                // Pour le moment, assigner tous les personnages à toutes les quêtes
                // TODO: Implémenter une logique pour assigner les personnages par quête
                foreach (Quest quest in questsData)
                {
                    questCharacters[quest.id] = charactersData.personnages;
                }
            }
        }
        else
        {
            Debug.LogWarning($"[QuetesTab] ⚠️ Fichier persos.json non trouvé à {persosJsonPath}");
        }
        
        yield return null;
    }
    
    IEnumerator BuildInterface()
    {
        Debug.Log("[QuetesTab] Construction de l'interface...");
        
        if (config == null)
        {
            Debug.LogError("[QuetesTab] ❌ Configuration null, impossible de construire l'interface");
            yield break;
        }
        
        // Créer le bloc de liste des quêtes (à gauche)
        CreateQuestListBlock();
        
        // Créer le bloc de détails (à droite)
        CreateDetailsBlock();
        
        Debug.Log("[QuetesTab] ✅ Interface construite");
        
        yield return null;
        
        // Charger les données des quêtes une seule fois
        Debug.Log("[QuetesTab] 🎯 Chargement initial des données...");
        LoadQuestsData();
        
        yield return new WaitForSeconds(0.2f);
        
        // Remplir la liste des quêtes (utilise fallback si pas de données API)
        Debug.Log("[QuetesTab] 📋 Remplissage de la liste...");
        PopulateQuestList();
        
        isInitialized = true;
        Debug.Log("[QuetesTab] ✅ Construction de l'interface terminée");
    }
    
    void CreateQuestListBlock()
    {
        var listConfig = config.questListBlock;
        
        questListBlock = new GameObject("QuestListBlock");
        questListBlock.transform.SetParent(transform, false);
        questListBlock.transform.SetAsLastSibling();
        
        RectTransform rect = questListBlock.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(0, 1);
        rect.anchorMax = new Vector2(0, 1);
        rect.pivot = new Vector2(0, 1);
        rect.anchoredPosition = new Vector2(listConfig.x, -listConfig.y);
        rect.sizeDelta = new Vector2(listConfig.width, listConfig.height);
        
        // Image de fond avec coins arrondis (sprite blanc + couleur via Image.color)
        Image bgImage = questListBlock.AddComponent<Image>();
        bgImage.sprite = CreateRoundedSprite((int)listConfig.width, (int)listConfig.height,
            listConfig.cornerRadius, Color.white);
        bgImage.color = HexToColor(listConfig.backgroundColor);
        bgImage.type = Image.Type.Simple;
        bgImage.raycastTarget = false;
        
        if (listConfig.shadowEnabled)
        {
            Shadow shadow = questListBlock.AddComponent<Shadow>();
            shadow.effectColor = HexToColor(listConfig.shadowColor);
            shadow.effectDistance = new Vector2(0, -listConfig.shadowOffsetY);
        }
        
        // ScrollRect pour la liste - APPROCHE SIMPLIFIÉE SANS LAYOUTGROUP
        GameObject viewport = new GameObject("Viewport");
        viewport.transform.SetParent(questListBlock.transform, false);
        
        RectTransform viewportRect = viewport.AddComponent<RectTransform>();
        viewportRect.anchorMin = Vector2.zero;
        viewportRect.anchorMax = Vector2.one;
        viewportRect.offsetMin = new Vector2(listConfig.padding, listConfig.padding);
        viewportRect.offsetMax = new Vector2(-listConfig.padding, -listConfig.padding);
        
        // IMPORTANT: RectMask2D au lieu de Mask pour un meilleur clipping
        RectMask2D rectMask = viewport.AddComponent<RectMask2D>();
        
        // Content container - SANS LayoutGroup, positionnement manuel
        GameObject content = new GameObject("Content");
        content.transform.SetParent(viewport.transform, false);
        
        RectTransform contentRect = content.AddComponent<RectTransform>();
        // Ancré en haut à gauche, taille calculée manuellement
        contentRect.anchorMin = new Vector2(0, 1);
        contentRect.anchorMax = new Vector2(0, 1);
        contentRect.pivot = new Vector2(0, 1);
        contentRect.anchoredPosition = Vector2.zero;
        // La taille sera mise à jour dans PopulateQuestList
        float contentWidth = listConfig.width - listConfig.padding * 2;
        contentRect.sizeDelta = new Vector2(contentWidth, 100); // Hauteur temporaire
        
        ScrollRect scrollRect = questListBlock.AddComponent<ScrollRect>();
        scrollRect.content = contentRect;
        scrollRect.viewport = viewportRect;
        scrollRect.horizontal = false;
        scrollRect.vertical = true;
        scrollRect.movementType = ScrollRect.MovementType.Clamped;
        scrollRect.scrollSensitivity = 20f;
        
        Debug.Log($"[QuetesTab] ✅ Bloc liste créé (SANS LayoutGroup) - Position: ({listConfig.x}, {listConfig.y}), Taille: {listConfig.width}x{listConfig.height}");
    }
    
    void CreateDetailsBlock()
    {
        var detailsConfig = config.detailsBlock;
        
        detailsBlock = new GameObject("DetailsBlock");
        detailsBlock.transform.SetParent(transform, false);
        
        RectTransform rect = detailsBlock.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(0, 1);
        rect.anchorMax = new Vector2(0, 1);
        rect.pivot = new Vector2(0, 1);
        rect.anchoredPosition = new Vector2(detailsConfig.x, -detailsConfig.y);
        rect.sizeDelta = new Vector2(detailsConfig.width, detailsConfig.height);
        
        // Image de fond avec coins arrondis (sprite blanc + couleur via Image.color)
        Image bgImage = detailsBlock.AddComponent<Image>();
        bgImage.sprite = CreateRoundedSprite((int)detailsConfig.width, (int)detailsConfig.height,
            detailsConfig.cornerRadius, Color.white);
        bgImage.color = HexToColor(detailsConfig.backgroundColor);
        bgImage.type = Image.Type.Simple;
        bgImage.raycastTarget = false;
        
        // Ombre
        if (detailsConfig.shadowEnabled)
        {
            Shadow shadow = detailsBlock.AddComponent<Shadow>();
            shadow.effectColor = HexToColor(detailsConfig.shadowColor);
            shadow.effectDistance = new Vector2(0, -detailsConfig.shadowOffsetY);
        }
        
        // Header avec les onglets (PERSONNAGES, BADGES, FORMATIONS)
        CreateDetailsHeader();
        
        // Zone de contenu
        CreateDetailsContent();
        
        Debug.Log($"[QuetesTab] ✅ Bloc détails créé - Position: ({detailsConfig.x}, {detailsConfig.y}), Taille: {detailsConfig.width}x{detailsConfig.height}");
    }
    
    void CreateDetailsHeader()
    {
        var headerConfig = config.detailsHeader;
        
        detailsHeader = new GameObject("DetailsHeader");
        detailsHeader.transform.SetParent(detailsBlock.transform, false);
        
        RectTransform rect = detailsHeader.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(0, 1);
        rect.anchorMax = new Vector2(1, 1);
        rect.pivot = new Vector2(0.5f, 1);
        rect.sizeDelta = new Vector2(0, headerConfig.height);
        rect.anchoredPosition = Vector2.zero;
        
        // Container pour les onglets
        GameObject tabsContainer = new GameObject("TabsContainer");
        tabsContainer.transform.SetParent(detailsHeader.transform, false);
        
        RectTransform tabsRect = tabsContainer.AddComponent<RectTransform>();
        tabsRect.anchorMin = new Vector2(0, 0);
        tabsRect.anchorMax = new Vector2(1, 1);
        tabsRect.offsetMin = new Vector2(headerConfig.paddingHorizontal, headerConfig.paddingVertical);
        tabsRect.offsetMax = new Vector2(-headerConfig.paddingHorizontal, -headerConfig.paddingVertical);
        
        HorizontalLayoutGroup layout = tabsContainer.AddComponent<HorizontalLayoutGroup>();
        layout.spacing = headerConfig.tabSpacing;
        layout.childAlignment = TextAnchor.MiddleCenter;
        layout.childControlWidth = false;
        layout.childControlHeight = false;
        layout.childForceExpandWidth = false;
        layout.childForceExpandHeight = false;
        
        // Créer les 3 boutons d'onglets
        CreateDetailTabButton(tabsContainer.transform, "PERSONNAGES", 0);
        CreateDetailTabButton(tabsContainer.transform, "BADGES", 1);
        CreateDetailTabButton(tabsContainer.transform, "FORMATIONS", 2);
        
        Debug.Log("[QuetesTab] ✅ Header des détails créé avec 3 onglets");
    }
    
    void CreateDetailTabButton(Transform parent, string label, int index)
    {
        var tabConfig = config.detailsHeader.tabStyle;
        
        GameObject tabButton = new GameObject($"TabButton_{label}");
        tabButton.transform.SetParent(parent, false);
        
        RectTransform rect = tabButton.AddComponent<RectTransform>();
        rect.sizeDelta = new Vector2(tabConfig.width, tabConfig.height);
        
        LayoutElement layoutElement = tabButton.AddComponent<LayoutElement>();
        layoutElement.preferredWidth = tabConfig.width;
        layoutElement.preferredHeight = tabConfig.height;
        
        Image image = tabButton.AddComponent<Image>();
        image.sprite = CreateRoundedSprite((int)tabConfig.width, (int)tabConfig.height, 
            tabConfig.cornerRadius, Color.white);
        image.type = Image.Type.Simple;
        image.color = Color.clear; // Transparent par défaut
        image.raycastTarget = true;
        
        Button button = tabButton.AddComponent<Button>();
        button.targetGraphic = image;
        button.transition = Selectable.Transition.None;
        
        button.onClick.AddListener(() => {
            Debug.Log($"[QuetesTab] Clic sur l'onglet {label} (index {index})");
            ShowDetailTab(index);
        });
        
        // Texte
        GameObject textObj = new GameObject("Text");
        textObj.transform.SetParent(tabButton.transform, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.offsetMin = Vector2.zero;
        textRect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI text = textObj.AddComponent<TextMeshProUGUI>();
        text.text = label;
        text.fontSize = 32;
        // Couleur initiale : premier onglet sélectionné, les autres non
        text.color = (index == 0) ? HexToColor("#64477D") : HexToColor("#ac99b1");
        text.alignment = TextAlignmentOptions.Center;
        text.fontStyle = FontStyles.Normal;
        if (titleFont != null) text.font = titleFont;
        
        detailTabButtons.Add(button);
        
        Debug.Log($"[QuetesTab] Bouton d'onglet créé: {label}");
    }
    
    void CreateDetailsContent()
    {
        var contentConfig = config.detailsContent;
        var headerConfig = config.detailsHeader;
        
        // Le detailsContent est directement dans detailsBlock, SOUS le header
        detailsContent = new GameObject("DetailsContent");
        detailsContent.transform.SetParent(detailsBlock.transform, false);
        
        RectTransform rect = detailsContent.AddComponent<RectTransform>();
        rect.anchorMin = new Vector2(0, 0);
        rect.anchorMax = new Vector2(1, 1);
        // Positionné sous le header (onglets) avec du padding
        rect.offsetMin = new Vector2(contentConfig.padding, contentConfig.padding);
        rect.offsetMax = new Vector2(-contentConfig.padding, -(headerConfig.height + 10));
        
        // Pas de cadre par défaut - il sera créé dynamiquement quand une quête est sélectionnée
        // Afficher le message par défaut
        ShowNoQuestSelectedMessage();
        
        Debug.Log("[QuetesTab] ✅ Zone de contenu des détails créée (sans cadre par défaut)");
    }
    
    void ShowNoQuestSelectedMessage()
    {
        // Nettoyer le contenu existant
        foreach (Transform child in detailsContent.transform)
        {
            Destroy(child.gameObject);
        }
        
        // Texte centré "Sélectionnez une quête"
        GameObject messageObj = new GameObject("NoQuestMessage");
        messageObj.transform.SetParent(detailsContent.transform, false);
        
        RectTransform rect = messageObj.AddComponent<RectTransform>();
        rect.anchorMin = Vector2.zero;
        rect.anchorMax = Vector2.one;
        rect.offsetMin = Vector2.zero;
        rect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI text = messageObj.AddComponent<TextMeshProUGUI>();
        text.text = "Sélectionnez une quête";
        text.fontSize = 28;
        text.color = HexToColor("#8a7a9a");
        text.alignment = TextAlignmentOptions.Center;
        text.fontStyle = FontStyles.Italic;
        if (textFont != null) text.font = textFont;
    }
    
    void PopulateQuestList()
    {
        Debug.Log($"[QUESTLIST_DEBUG] 🔄 PopulateQuestList() appelé - {questsData.Count} quêtes à afficher");
        
        // Vider le dictionnaire des références
        questItemsMap.Clear();
        
        if (questListBlock == null)
        {
            Debug.LogError("[QuetesTab] ❌ questListBlock est null!");
            return;
        }
        
        Transform viewport = questListBlock.transform.Find("Viewport");
        if (viewport == null)
        {
            Debug.LogError("[QuetesTab] ❌ Viewport non trouvé!");
            return;
        }
        
        Transform content = viewport.Find("Content");
        if (content == null)
        {
            Debug.LogError("[QuetesTab] ❌ Content non trouvé!");
            return;
        }
        
        // Nettoyer le contenu existant
        while (content.childCount > 0)
        {
            DestroyImmediate(content.GetChild(0).gameObject);
        }
        
        if (questsData.Count == 0)
        {
            Debug.LogWarning("[QuetesTab] ⚠️ Aucune quête à afficher");
            return;
        }
        
        var listConfig = config.questListBlock;
        var itemConfig = listConfig.itemStyle;
        float itemWidth = listConfig.width - listConfig.padding * 2;
        float itemHeight = itemConfig.height;
        float spacing = listConfig.itemSpacing;
        
        // Calculer la hauteur totale du content
        float totalHeight = questsData.Count * itemHeight + (questsData.Count - 1) * spacing;
        
        RectTransform contentRect = content.GetComponent<RectTransform>();
        contentRect.sizeDelta = new Vector2(itemWidth, totalHeight);
        
        Debug.Log($"[QUESTLIST_DEBUG] 📐 Content size définie à: {itemWidth}x{totalHeight}");
        
        // Créer chaque item avec positionnement manuel
        float currentY = 0;
        for (int i = 0; i < questsData.Count; i++)
        {
            Quest quest = questsData[i];
            if (quest != null)
            {
                CreateQuestListItemManual(content, quest, itemWidth, itemHeight, currentY);
                currentY -= itemHeight + spacing;
            }
        }
        
        Debug.Log($"[QUESTLIST_DEBUG] ✅ {questsData.Count} quêtes créées manuellement");
        
        // Vérifier le premier item
        if (content.childCount > 0)
        {
            RectTransform firstItem = content.GetChild(0).GetComponent<RectTransform>();
            Debug.Log($"[QUESTLIST_DEBUG] 📐 Premier item - pos: {firstItem.anchoredPosition}, size: {firstItem.sizeDelta}");
        }
        
        // Sélectionner la première quête avec un petit délai pour s'assurer que tout est prêt
        if (questsData.Count > 0)
        {
            StartCoroutine(SelectFirstQuestWithDelay());
        }
    }
    
    IEnumerator SelectFirstQuestWithDelay()
    {
        // Attendre que l'interface soit complètement prête
        yield return new WaitForEndOfFrame();
        yield return new WaitForSeconds(0.1f);
        
        if (questsData != null && questsData.Count > 0)
        {
            Debug.Log("[QuetesTab] 🎯 Sélection automatique de la première quête");
            SelectQuest(questsData[0]);
        }
    }
    
    void CreateQuestListItemManual(Transform parent, Quest quest, float width, float height, float yPos)
    {
        var itemConfig = config.questListBlock.itemStyle;
        var lockedConfig = itemConfig.lockedStyle;
        
        // Déterminer si la quête est verrouillée
        bool isLocked = quest.status == "locked";
        
        GameObject item = new GameObject($"QuestItem_{quest.id}");
        item.transform.SetParent(parent, false);
        
        RectTransform itemRect = item.AddComponent<RectTransform>();
        // Positionnement manuel - ancré en haut à gauche du parent
        itemRect.anchorMin = new Vector2(0, 1);
        itemRect.anchorMax = new Vector2(0, 1);
        itemRect.pivot = new Vector2(0, 1);
        itemRect.anchoredPosition = new Vector2(0, yPos);
        itemRect.sizeDelta = new Vector2(width, height);
        
        Debug.Log($"[QUESTLIST_DEBUG] 🔨 Item '{quest.title}' créé - pos: (0, {yPos}), size: ({width}, {height}), locked: {isLocked}");
        
        // Fond avec coins arrondis - gris si locked
        Image bgImage = item.AddComponent<Image>();
        bgImage.sprite = CreateRoundedSprite((int)width, (int)height, itemConfig.cornerRadius, Color.white);
        
        string bgColor = isLocked && lockedConfig != null 
            ? lockedConfig.backgroundColor 
            : itemConfig.normalBackgroundColor;
        bgImage.color = HexToColor(bgColor);
        bgImage.type = Image.Type.Simple;
        bgImage.raycastTarget = true;
        
        // Bouton - désactivé si locked
        if (!isLocked)
        {
            Button button = item.AddComponent<Button>();
            button.targetGraphic = bgImage;
            button.transition = Selectable.Transition.None; // On gère manuellement les couleurs
            
            button.onClick.AddListener(() => {
                Debug.Log($"[QuetesTab] 🖱️ CLIC sur quête: {quest.title} (ID: {quest.id})");
                SelectQuest(quest);
            });
        }
        else
        {
            Debug.Log($"[QuetesTab] 🔒 Quête {quest.id} verrouillée - pas de bouton");
        }
        
        // Zone de texte qui remplit l'item avec padding
        float padding = itemConfig.paddingHorizontal;
        float paddingV = itemConfig.paddingVertical;
        float textWidth = width - padding * 2;
        
        // Couleurs du titre et description selon l'état
        string titleColorHex = isLocked && lockedConfig != null 
            ? lockedConfig.titleColor 
            : itemConfig.titleColor;
        string descColorHex = isLocked && lockedConfig != null 
            ? lockedConfig.descriptionColor 
            : itemConfig.descriptionColor;
        
        // Calculer la largeur du titre pour laisser de la place au cadenas si locked
        float lockIconSpace = 0;
        if (isLocked && lockedConfig != null)
        {
            lockIconSpace = lockedConfig.lockIconSize + lockedConfig.lockIconMarginLeft + 10;
        }
        
        // Titre en haut - centré horizontalement
        GameObject titleObj = new GameObject("Title");
        titleObj.transform.SetParent(item.transform, false);
        
        RectTransform titleRect = titleObj.AddComponent<RectTransform>();
        titleRect.anchorMin = new Vector2(0, 1);
        titleRect.anchorMax = new Vector2(0, 1);
        titleRect.pivot = new Vector2(0, 1);
        titleRect.anchoredPosition = new Vector2(padding, -paddingV);
        titleRect.sizeDelta = new Vector2(textWidth - lockIconSpace, 40); // Hauteur 40px pour le titre
        
        TextMeshProUGUI titleText = titleObj.AddComponent<TextMeshProUGUI>();
        titleText.text = (quest.title ?? "Titre manquant").ToUpper(); // En majuscules
        titleText.fontSize = itemConfig.titleFontSize;
        Color titleColorValue = HexToColor(titleColorHex);
        titleText.color = titleColorValue;
        titleText.fontStyle = FontStyles.Bold;
        titleText.alignment = TextAlignmentOptions.Center; // Texte centré dans la zone
        titleText.raycastTarget = false;
        titleText.overflowMode = TextOverflowModes.Truncate;
        // Police Anton pour le titre
        if (titleFont != null) titleText.font = titleFont;
        
        Debug.Log($"[QUESTLIST_DEBUG] 📝 Titre: '{titleText.text}' - couleur: {titleColorValue}, fontSize: {itemConfig.titleFontSize}, font: {(titleText.font != null ? titleText.font.name : "DEFAULT")}");
        
        // Ajouter l'icône cadenas si locked
        if (isLocked && lockedConfig != null)
        {
            GameObject lockObj = new GameObject("LockIcon");
            lockObj.transform.SetParent(item.transform, false);
            
            RectTransform lockRect = lockObj.AddComponent<RectTransform>();
            lockRect.anchorMin = new Vector2(1, 1);
            lockRect.anchorMax = new Vector2(1, 1);
            lockRect.pivot = new Vector2(1, 0.5f);
            // Positionné à droite du titre, aligné verticalement avec lui
            lockRect.anchoredPosition = new Vector2(-padding, -(paddingV + 20)); // 20 = moitié de 40px hauteur titre
            lockRect.sizeDelta = new Vector2(lockedConfig.lockIconSize, lockedConfig.lockIconSize);
            
            // D'abord essayer de charger l'image
            string lockIconUrl = GeneralConfigManager.Instance?.GetUIUrl(lockedConfig.lockIconUrl) ?? "";
            Debug.Log($"[QuetesTab] 🔒 URL cadenas construite: '{lockIconUrl}' (depuis '{lockedConfig.lockIconUrl}')");
            
            if (!string.IsNullOrEmpty(lockIconUrl))
            {
                Image lockImage = lockObj.AddComponent<Image>();
                lockImage.preserveAspect = true;
                lockImage.raycastTarget = false;
                lockImage.color = Color.clear; // Transparent jusqu'au chargement du sprite
                lockImage.type = Image.Type.Simple;
                
                // Lancer la coroutine sur GeneralConfigManager qui reste actif
                // (évite l'interruption quand l'onglet est désactivé)
                if (GeneralConfigManager.Instance != null)
                {
                    GeneralConfigManager.Instance.StartCoroutine(LoadLockIconSprite(lockIconUrl, lockImage, lockObj));
                }
                else
                {
                    StartCoroutine(LoadLockIconSprite(lockIconUrl, lockImage, lockObj));
                }
            }
            else
            {
                // Fallback: utiliser un texte cadenas
                Debug.LogWarning($"[QuetesTab] ⚠️ URL cadenas vide, utilisation du texte emoji");
                CreateLockTextFallback(lockObj, lockedConfig.lockIconSize);
            }
            
            Debug.Log($"[QuetesTab] 🔒 Icône cadenas ajoutée pour quête {quest.id}");
        }
        
        // Description sous le titre
        GameObject descObj = new GameObject("Description");
        descObj.transform.SetParent(item.transform, false);
        
        RectTransform descRect = descObj.AddComponent<RectTransform>();
        descRect.anchorMin = new Vector2(0, 1);
        descRect.anchorMax = new Vector2(0, 1);
        descRect.pivot = new Vector2(0, 1);
        descRect.anchoredPosition = new Vector2(padding, -(paddingV + 45)); // Sous le titre (40px + 5px spacing)
        descRect.sizeDelta = new Vector2(textWidth, height - paddingV * 2 - 50);
        
        TextMeshProUGUI descText = descObj.AddComponent<TextMeshProUGUI>();
        descText.text = quest.description ?? "Aucune description disponible";
        descText.fontSize = itemConfig.descriptionFontSize;
        descText.color = HexToColor(descColorHex);
        descText.alignment = TextAlignmentOptions.TopLeft;
        descText.overflowMode = TextOverflowModes.Truncate;
        descText.textWrappingMode = TMPro.TextWrappingModes.Normal;
        descText.maxVisibleLines = 4;
        descText.raycastTarget = false;
        if (textFont != null) descText.font = textFont;
        
        // Stocker la référence pour la sélection (même si locked, pour la mise à jour d'apparence)
        questItemsMap[quest.id] = item;
        
        // Marquer comme locked pour UpdateQuestItemsAppearance
        if (isLocked)
        {
            item.AddComponent<LockedQuestMarker>();
        }
        
        Debug.Log($"[QUESTLIST_DEBUG] ✅ Item complet créé pour: {quest.title}");
    }
    
    IEnumerator LoadLockIconSprite(string url, Image targetImage, GameObject lockObj)
    {
        Debug.Log($"[QuetesTab] 🔒 Tentative de chargement du cadenas: {url}");
        
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                Debug.Log($"[QuetesTab] 🔒 Texture chargée: {texture.width}x{texture.height}, format: {texture.format}");
                
                dynamicTextures.Add(texture);
                Sprite sprite = Sprite.Create(texture, new Rect(0, 0, texture.width, texture.height), 
                    new Vector2(0.5f, 0.5f), 100f);
                    
                if (targetImage != null)
                {
                    targetImage.sprite = sprite;
                    targetImage.color = Color.white; // Important : couleur blanche pour afficher le sprite correctement
                    targetImage.type = Image.Type.Simple;
                    Debug.Log($"[QuetesTab] ✅ Icône cadenas chargée et assignée avec succès");
                }
            }
            else
            {
                Debug.LogError($"[QuetesTab] ❌ Impossible de charger l'icône cadenas: {url}");
                Debug.LogError($"[QuetesTab] ❌ Erreur: {request.error}, Code: {request.responseCode}");
                
                // En cas d'échec, supprimer l'image et utiliser un texte emoji cadenas
                if (lockObj != null && targetImage != null)
                {
                    Destroy(targetImage);
                    float iconSize = config?.questListBlock?.itemStyle?.lockedStyle?.lockIconSize ?? 24;
                    CreateLockTextFallback(lockObj, iconSize);
                }
            }
        }
    }
    
    void CreateLockTextFallback(GameObject parent, float size)
    {
        // Créer un texte avec le symbole cadenas Unicode
        TextMeshProUGUI lockText = parent.AddComponent<TextMeshProUGUI>();
        lockText.text = "🔒"; // Emoji cadenas
        lockText.fontSize = size * 0.8f;
        lockText.color = HexToColor("#808080");
        lockText.alignment = TextAlignmentOptions.Center;
        lockText.raycastTarget = false;
        
        Debug.Log("[QuetesTab] ✅ Fallback texte cadenas créé");
    }
    
    void SelectQuest(Quest quest)
    {
        Debug.Log($"[QuetesTab] 🎯 SelectQuest() appelé pour: {quest?.title ?? "NULL"} (ID: {quest?.id ?? -1})");
        
        if (quest == null)
        {
            Debug.LogError("[QuetesTab] ❌ Quest est NULL!");
            return;
        }
        
        selectedQuest = quest;
        Debug.Log($"[QuetesTab] ✅ selectedQuest défini: {selectedQuest.title}");
        
        // Mettre à jour l'apparence de tous les items de quête
        UpdateQuestItemsAppearance(quest.id);
        
        if (detailsContent == null)
        {
            Debug.LogError("[QuetesTab] ❌ detailsContent est NULL! Impossible d'afficher les détails.");
            return;
        }
        
        Debug.Log($"[QuetesTab] ✅ detailsContent existe: {detailsContent.name}");
        
        // Appeler l'API pour récupérer les détails de la quête (personnages, badges, etc.)
        Debug.Log("[QuetesTab] 🚀 Démarrage de LoadQuestDetailsFromApi...");
        StartCoroutine(LoadQuestDetailsFromApi(quest.id));
        
        // Afficher immédiatement l'onglet PERSONNAGES par défaut
        activeDetailTab = 0;
        UpdateDetailTabButtons(0);
    }
    
    void UpdateQuestItemsAppearance(int selectedQuestId)
    {
        var itemConfig = config.questListBlock.itemStyle;
        var lockedConfig = itemConfig.lockedStyle;
        
        foreach (var kvp in questItemsMap)
        {
            int questId = kvp.Key;
            GameObject item = kvp.Value;
            
            if (item == null) continue;
            
            // Vérifier si c'est une quête verrouillée
            bool isLocked = item.GetComponent<LockedQuestMarker>() != null;
            
            // Ne pas changer l'apparence des quêtes verrouillées
            if (isLocked)
            {
                continue;
            }
            
            bool isSelected = (questId == selectedQuestId);
            
            // Mettre à jour la couleur de fond
            Image bgImage = item.GetComponent<Image>();
            if (bgImage != null)
            {
                bgImage.color = isSelected 
                    ? HexToColor(itemConfig.selectedBackgroundColor) 
                    : HexToColor(itemConfig.normalBackgroundColor);
            }
            
            // Mettre à jour la couleur du titre
            Transform titleTransform = item.transform.Find("Title");
            if (titleTransform != null)
            {
                TextMeshProUGUI titleText = titleTransform.GetComponent<TextMeshProUGUI>();
                if (titleText != null)
                {
                    titleText.color = isSelected ? Color.white : HexToColor(itemConfig.titleColor);
                }
            }
            
            // Mettre à jour la couleur de la description
            Transform descTransform = item.transform.Find("Description");
            if (descTransform != null)
            {
                TextMeshProUGUI descText = descTransform.GetComponent<TextMeshProUGUI>();
                if (descText != null)
                {
                    descText.color = isSelected ? Color.white : HexToColor(itemConfig.descriptionColor);
                }
            }
        }
    }
    
    IEnumerator LoadQuestDetailsFromApi(int questId)
    {
        Debug.Log($"[QuetesTab] 📡 Chargement des détails de la quête {questId}...");
        
        // Afficher un message de chargement
        ShowLoadingMessage("Chargement des détails...");
        
        // Construire l'URL de l'API
        string apiUrl = GeneralConfigManager.Instance?.GetQuestConfigApiUrl(questId);
        
        if (string.IsNullOrEmpty(apiUrl))
        {
            Debug.LogError("[QuetesTab] ❌ Impossible de construire l'URL de l'API");
            ShowDetailTab(activeDetailTab);
            yield break;
        }
        
        // Récupérer le token d'authentification
        string token = UserDataManager.Instance?.token;
        
        if (string.IsNullOrEmpty(token))
        {
            Debug.LogWarning("[QuetesTab] ⚠️ Pas de token d'authentification");
            ShowDetailTab(activeDetailTab);
            yield break;
        }
        
        Debug.Log($"[QuetesTab] 🌐 Appel API: {apiUrl}");
        
        using (UnityWebRequest www = UnityWebRequest.Get(apiUrl))
        {
            www.SetRequestHeader("Authorization", $"Bearer {token}");
            yield return www.SendWebRequest();
            
            if (www.result != UnityWebRequest.Result.Success)
            {
                Debug.LogError($"[QuetesTab] ❌ Erreur API: {www.error}");
                Debug.LogError($"[QuetesTab] Code: {www.responseCode}");
                ShowDetailTab(activeDetailTab);
                yield break;
            }
            
            string jsonResponse = www.downloadHandler.text;
            Debug.Log($"[QuetesTab] ✅ Réponse reçue ({jsonResponse.Length} caractères)");
            Debug.Log($"[QuetesTab] 🔍 Début réponse: {jsonResponse.Substring(0, Mathf.Min(500, jsonResponse.Length))}");
            
            // Parser la réponse
            try
            {
                // Log la réponse brute pour debug
                Debug.Log($"[QuetesTab] 📄 Réponse JSON brute (premiers 1000 caractères):\n{jsonResponse.Substring(0, Mathf.Min(1000, jsonResponse.Length))}");
                
                QuestDetailApiResponse response = JsonUtility.FromJson<QuestDetailApiResponse>(jsonResponse);
                
                if (response != null && response.status == "success" && response.data != null)
                {
                    Debug.Log($"[QuetesTab] ✅ Détails de la quête chargés");
                    
                    // Stocker les personnages
                    if (response.data.characters != null && response.data.characters.Count > 0)
                    {
                        questCharacters[questId] = response.data.characters;
                        Debug.Log($"[QuetesTab] 👥 {response.data.characters.Count} personnages chargés");
                        
                        foreach (var character in response.data.characters)
                        {
                            // Log toutes les propriétés pour debug
                            Debug.Log($"[QuetesTab]   - Champs bruts: nom='{character.nom}', name='{character.name}', desc_courte='{character.description_courte}', short_desc='{character.short_description}'");
                            Debug.Log($"[QuetesTab]   - Valeurs finales: GetName()='{character.GetName()}', GetDescription()='{character.GetDescription()}', GetAvatarUrl()='{character.GetAvatarUrl()}'");
                        }
                    }
                    else
                    {
                        Debug.Log("[QuetesTab] ⚠️ Aucun personnage dans la réponse API");
                    }
                    
                    // Stocker les badges si présents
                    if (response.data.badges != null && response.data.badges.Count > 0)
                    {
                        Debug.Log($"[QuetesTab] 🏅 {response.data.badges.Count} badges chargés");
                    }
                    
                    // Stocker les formations dans selectedQuest
                    if (response.data.trainings != null && response.data.trainings.Count > 0)
                    {
                        selectedQuest.trainings = response.data.trainings;
                        Debug.Log($"[QuetesTab] 📚 {response.data.trainings.Count} formations chargées");
                        foreach (var training in response.data.trainings)
                        {
                            Debug.Log($"[QuetesTab]   - Formation: '{training.title}' → {training.url}");
                        }
                    }
                    else
                    {
                        Debug.Log("[QuetesTab] ⚠️ Aucune formation dans la réponse API");
                    }
                }
                else
                {
                    Debug.LogWarning($"[QuetesTab] ⚠️ Réponse API invalide ou sans données");
                }
            }
            catch (Exception e)
            {
                Debug.LogError($"[QuetesTab] ❌ Erreur parsing JSON: {e.Message}");
            }
        }
        
        // Afficher l'onglet actif avec les nouvelles données
        ShowDetailTab(activeDetailTab);
    }
    
    void ShowLoadingMessage(string message)
    {
        Debug.Log($"[QuetesTab] 📢 ShowLoadingMessage() appelé avec: '{message}'");
        
        if (detailsContent == null)
        {
            Debug.LogError("[QuetesTab] ❌ detailsContent est NULL dans ShowLoadingMessage!");
            return;
        }
        
        // Nettoyer le contenu existant
        foreach (Transform child in detailsContent.transform)
        {
            Destroy(child.gameObject);
        }
        
        GameObject loadingObj = new GameObject("LoadingMessage");
        loadingObj.transform.SetParent(detailsContent.transform, false);
        Debug.Log("[QuetesTab] ✅ LoadingMessage créé");
        
        RectTransform rect = loadingObj.AddComponent<RectTransform>();
        rect.anchorMin = Vector2.zero;
        rect.anchorMax = Vector2.one;
        rect.offsetMin = Vector2.zero;
        rect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI text = loadingObj.AddComponent<TextMeshProUGUI>();
        text.text = message;
        text.fontSize = 24;
        text.color = HexToColor("#a95bfb");
        text.alignment = TextAlignmentOptions.Center;
        text.fontStyle = FontStyles.Italic;
        if (textFont != null) text.font = textFont;
    }
    
    void ShowDetailTab(int index)
    {
        activeDetailTab = index;
        
        // Mettre à jour les couleurs des boutons d'onglets
        UpdateDetailTabButtons(index);
        
        // Afficher le contenu correspondant
        switch (index)
        {
            case 0: // PERSONNAGES
                ShowPersonnagesContent();
                break;
            case 1: // BADGES
                ShowBadgesContent();
                break;
            case 2: // FORMATIONS
                ShowFormationsContent();
                break;
        }
    }
    
    void UpdateDetailTabButtons(int selectedIndex)
    {
        for (int i = 0; i < detailTabButtons.Count; i++)
        {
            Button button = detailTabButtons[i];
            Image image = button.GetComponent<Image>();
            TextMeshProUGUI text = button.GetComponentInChildren<TextMeshProUGUI>();
            
            // Pas de fond (gélule) - toujours transparent
            image.color = Color.clear;
            
            if (i == selectedIndex)
            {
                // Onglet sélectionné : texte en violet foncé
                if (text != null)
                    text.color = HexToColor("#64477D");
            }
            else
            {
                // Onglet non sélectionné : texte en gris violet clair
                if (text != null)
                    text.color = HexToColor("#ac99b1");
            }
        }
    }
    
    void ShowPersonnagesContent()
    {
        Debug.Log("[QuetesTab] ========== ShowPersonnagesContent() MOSAIQUE ==========");
        
        // Nettoyer le contenu existant
        foreach (Transform child in detailsContent.transform)
        {
            Destroy(child.gameObject);
        }
        avatarButtons.Clear();
        
        if (selectedQuest == null)
        {
            Debug.Log("[QuetesTab] ⚠️ selectedQuest est NULL");
            ShowNoQuestSelectedMessage();
            return;
        }
        
        
        // Récupérer les personnages de la quête
        currentCharacters = GetQuestCharacters(selectedQuest.id);
        
        if (currentCharacters == null || currentCharacters.Count == 0)
        {
            Debug.Log("[QuetesTab] ⚠️ Aucun personnage dans la liste");
            CreateEmptyMessage("Aucun personnage pour cette quête");
            return;
        }
        
        Debug.Log($"[QuetesTab] 👥 {currentCharacters.Count} personnages à afficher en mosaïque");
        
        var mosaicConfig = config.characterMosaic;
        var detailConfig = config.characterDetail;
        float padding = mosaicConfig.padding;
        float avatarSize = mosaicConfig.avatarSize;
        float spacing = mosaicConfig.avatarSpacing;
        float arrowSize = mosaicConfig.arrowSize;
        
        // Container principal
        GameObject container = new GameObject("PersonnagesContainer");
        container.transform.SetParent(detailsContent.transform, false);
        
        RectTransform containerRect = container.AddComponent<RectTransform>();
        containerRect.anchorMin = Vector2.zero;
        containerRect.anchorMax = Vector2.one;
        containerRect.offsetMin = Vector2.zero;
        containerRect.offsetMax = Vector2.zero;
        
        // Zone de mosaïque avec flèches
        GameObject mosaicZone = new GameObject("MosaicZone");
        mosaicZone.transform.SetParent(container.transform, false);
        
        RectTransform mosaicRect = mosaicZone.AddComponent<RectTransform>();
        mosaicRect.anchorMin = new Vector2(0, 1);
        mosaicRect.anchorMax = new Vector2(1, 1);
        mosaicRect.pivot = new Vector2(0.5f, 1);
        mosaicRect.anchoredPosition = new Vector2(0, -padding);
        float mosaicHeight = avatarSize + padding;
        mosaicRect.sizeDelta = new Vector2(0, mosaicHeight);
        
        // Flèche gauche
        CreateNavigationArrow(mosaicZone.transform, true, arrowSize, mosaicConfig.arrowColor);
        
        // Conteneur des avatars (centré) - seulement 3 visibles
        avatarsContainer = new GameObject("AvatarsContainer");
        avatarsContainer.transform.SetParent(mosaicZone.transform, false);
        
        RectTransform avatarsRect = avatarsContainer.AddComponent<RectTransform>();
        avatarsRect.anchorMin = new Vector2(0.5f, 0.5f);
        avatarsRect.anchorMax = new Vector2(0.5f, 0.5f);
        avatarsRect.pivot = new Vector2(0.5f, 0.5f);
        
        // Calculer la largeur totale pour 3 avatars max
        int maxVisible = mosaicConfig.columns; // 3 par défaut
        float totalAvatarsWidth = maxVisible * avatarSize + (maxVisible - 1) * spacing;
        avatarsRect.sizeDelta = new Vector2(totalAvatarsWidth, avatarSize);
        
        // Initialiser l'index de départ
        visibleStartIndex = 0;
        
        // Créer les 3 avatars visibles
        RefreshVisibleAvatars();
        
        // Flèche droite
        CreateNavigationArrow(mosaicZone.transform, false, arrowSize, mosaicConfig.arrowColor);
        
        // Zone de détail du personnage
        CreateCharacterDetailPanel(container.transform, mosaicHeight + padding, detailConfig);
        
        // Sélectionner le premier personnage par défaut
        selectedCharacterIndex = 0;
        UpdateSelectedCharacter();
        
        Debug.Log("[QuetesTab] ✅ Mosaïque créée");
    }
    
    void CreateNavigationArrow(Transform parent, bool isLeft, float size, string colorHex)
    {
        GameObject arrow = new GameObject(isLeft ? "LeftArrow" : "RightArrow");
        arrow.transform.SetParent(parent, false);
        
        RectTransform arrowRect = arrow.AddComponent<RectTransform>();
        arrowRect.anchorMin = new Vector2(isLeft ? 0 : 1, 0.5f);
        arrowRect.anchorMax = new Vector2(isLeft ? 0 : 1, 0.5f);
        arrowRect.pivot = new Vector2(0.5f, 0.5f);
        arrowRect.anchoredPosition = new Vector2(isLeft ? size / 2 + 10 : -size / 2 - 10, 0);
        arrowRect.sizeDelta = new Vector2(size, size);
        
        // Image de la flèche
        Image arrowImage = arrow.AddComponent<Image>();
        arrowImage.color = Color.white;
        arrowImage.raycastTarget = true;
        
        // Charger l'image depuis l'URL
        string imageName = isLeft ? "nav_gauche.png" : "nav_droite.png";
        string imageUrl = GeneralConfigManager.Instance?.GetUIUrl(imageName) ?? "";
        if (!string.IsNullOrEmpty(imageUrl))
        {
            StartCoroutine(LoadArrowSprite(imageUrl, arrowImage));
        }
        
        // Bouton
        Button button = arrow.AddComponent<Button>();
        button.transition = Selectable.Transition.ColorTint;
        button.targetGraphic = arrowImage;
        
        int direction = isLeft ? -1 : 1;
        button.onClick.AddListener(() => NavigateCharacter(direction));
    }
    
    IEnumerator LoadArrowSprite(string url, Image targetImage)
    {
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                dynamicTextures.Add(texture); // Garder la référence pour le nettoyage
                Sprite sprite = Sprite.Create(texture, new Rect(0, 0, texture.width, texture.height), 
                    new Vector2(0.5f, 0.5f), 100f);
                if (targetImage != null)
                {
                    targetImage.sprite = sprite;
                }
            }
            else
            {
                Debug.LogWarning($"[QuetesTab] ⚠️ Impossible de charger l'image flèche: {url}");
            }
        }
    }
    
    IEnumerator LoadFormationArrowSprite(Image targetImage)
    {
        string uiPath = generalConfig?.assetsPaths?.uiPath ?? "https://ujsa.studioplc.fr/datas/UI/";
        string url = uiPath + "fleche_formation.png";
        
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                dynamicTextures.Add(texture);
                Sprite sprite = Sprite.Create(texture, new Rect(0, 0, texture.width, texture.height), 
                    new Vector2(0.5f, 0.5f), 100f);
                if (targetImage != null)
                {
                    targetImage.sprite = sprite;
                }
            }
            else
            {
                Debug.LogWarning($"[QuetesTab] ⚠️ Impossible de charger fleche_formation.png: {url}");
            }
        }
    }
    
    void CreateAvatarButton(Transform parent, QuestCharacter character, int index, float xPos, float size, QuetesTabCharacterMosaicConfig mosaicConfig)
    {
        // Cadre carré avec fond et coins arrondis
        GameObject frameObj = new GameObject($"AvatarFrame_{index}");
        frameObj.transform.SetParent(parent, false);
        
        RectTransform frameRect = frameObj.AddComponent<RectTransform>();
        frameRect.anchorMin = new Vector2(0.5f, 0.5f);
        frameRect.anchorMax = new Vector2(0.5f, 0.5f);
        frameRect.pivot = new Vector2(0.5f, 0.5f);
        frameRect.anchoredPosition = new Vector2(xPos, 0);
        frameRect.sizeDelta = new Vector2(size, size);
        
        // Fond du cadre avec coins arrondis (couleur fixe, pas de changement à la sélection)
        Image frameImage = frameObj.AddComponent<Image>();
        frameImage.sprite = CreateRoundedSprite((int)size, (int)size, (int)mosaicConfig.avatarCornerRadius, Color.white);
        frameImage.color = HexToColor(mosaicConfig.avatarBorderColor);
        frameImage.type = Image.Type.Simple;
        frameImage.raycastTarget = true;
        
        // Bouton
        Button button = frameObj.AddComponent<Button>();
        button.targetGraphic = frameImage;
        button.transition = Selectable.Transition.None;
        
        int capturedIndex = index;
        button.onClick.AddListener(() => SelectCharacter(capturedIndex));
        
        // Image de l'avatar qui remplit tout le cadre (sans bordure intérieure)
        GameObject avatarObj = new GameObject("AvatarImage");
        avatarObj.transform.SetParent(frameObj.transform, false);
        
        RectTransform avatarRect = avatarObj.AddComponent<RectTransform>();
        // L'avatar remplit tout le cadre
        avatarRect.anchorMin = Vector2.zero;
        avatarRect.anchorMax = Vector2.one;
        avatarRect.pivot = new Vector2(0.5f, 0.5f);
        avatarRect.offsetMin = Vector2.zero;
        avatarRect.offsetMax = Vector2.zero;
        
        Image avatarImage = avatarObj.AddComponent<Image>();
        avatarImage.sprite = CreateRoundedSprite((int)size, (int)size, (int)mosaicConfig.avatarCornerRadius, Color.white);
        avatarImage.color = HexToColor("#e8ddd5"); // Couleur par défaut avant chargement
        avatarImage.type = Image.Type.Simple;
        avatarImage.preserveAspect = false; // Remplir tout l'espace
        avatarImage.raycastTarget = false;
        
        // Charger l'avatar
        string avatarUrl = character.GetAvatarUrl();
        if (!string.IsNullOrEmpty(avatarUrl))
        {
            string fullUrl = GetAvatarUrl(avatarUrl);
            StartCoroutine(LoadAvatarSprite(fullUrl, avatarImage));
        }
        
        // Bordure pointillée (cachée par défaut)
        CreateDashedBorder(frameObj.transform, size, mosaicConfig.avatarCornerRadius);
        
        avatarButtons.Add(frameObj);
    }
    
    void CreateDashedBorder(Transform parent, float size, float cornerRadius)
    {
        // Créer un cadre de sélection propre (comme le cadre des avatars mais juste le contour)
        GameObject borderContainer = new GameObject("DashedBorder");
        borderContainer.transform.SetParent(parent, false);
        
        RectTransform containerRect = borderContainer.AddComponent<RectTransform>();
        containerRect.anchorMin = Vector2.zero;
        containerRect.anchorMax = Vector2.one;
        containerRect.offsetMin = new Vector2(-4, -4);
        containerRect.offsetMax = new Vector2(4, 4);
        
        // Utiliser la même méthode que pour les cadres existants
        Image borderImage = borderContainer.AddComponent<Image>();
        int borderSize = (int)(size + 8);
        borderImage.sprite = CreateRoundedBorderOnlySprite(borderSize, borderSize, (int)cornerRadius + 2, 3);
        borderImage.color = HexToColor("#a23cff");
        borderImage.type = Image.Type.Simple;
        borderImage.raycastTarget = false;
        
        borderContainer.SetActive(false);
    }
    
    Sprite CreateRoundedBorderOnlySprite(int width, int height, int radius, int borderWidth)
    {
        Texture2D texture = new Texture2D(width, height, TextureFormat.RGBA32, false);
        Color[] pixels = new Color[width * height];
        
        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                // Vérifier si le pixel est dans le rectangle arrondi extérieur
                bool inOuter = IsInsideRoundedRect(x, y, width, height, radius);
                // Vérifier si le pixel est dans le rectangle arrondi intérieur
                bool inInner = IsInsideRoundedRect(x - borderWidth, y - borderWidth, 
                    width - borderWidth * 2, height - borderWidth * 2, 
                    Mathf.Max(0, radius - borderWidth));
                
                // Le pixel fait partie de la bordure s'il est dans l'extérieur mais pas dans l'intérieur
                if (inOuter && !inInner)
                {
                    pixels[y * width + x] = Color.white;
                }
                else
                {
                    pixels[y * width + x] = Color.clear;
                }
            }
        }
        
        texture.SetPixels(pixels);
        texture.Apply();
        
        return Sprite.Create(texture, new Rect(0, 0, width, height), new Vector2(0.5f, 0.5f), 100f);
    }
    
    bool IsInsideRoundedRect(int x, int y, int width, int height, int radius)
    {
        if (x < 0 || y < 0 || x >= width || y >= height) return false;
        
        // Vérifier les coins
        // Coin bas-gauche
        if (x < radius && y < radius)
        {
            float dx = radius - x;
            float dy = radius - y;
            return (dx * dx + dy * dy) <= radius * radius;
        }
        // Coin bas-droite
        if (x >= width - radius && y < radius)
        {
            float dx = x - (width - radius - 1);
            float dy = radius - y;
            return (dx * dx + dy * dy) <= radius * radius;
        }
        // Coin haut-gauche
        if (x < radius && y >= height - radius)
        {
            float dx = radius - x;
            float dy = y - (height - radius - 1);
            return (dx * dx + dy * dy) <= radius * radius;
        }
        // Coin haut-droite
        if (x >= width - radius && y >= height - radius)
        {
            float dx = x - (width - radius - 1);
            float dy = y - (height - radius - 1);
            return (dx * dx + dy * dy) <= radius * radius;
        }
        
        // Dans la zone centrale ou les côtés
        return true;
    }
    
    void CreateCharacterDetailPanel(Transform parent, float topOffset, QuetesTabCharacterDetailConfig detailConfig)
    {
        characterDetailPanel = new GameObject("CharacterDetailPanel");
        characterDetailPanel.transform.SetParent(parent, false);
        
        RectTransform panelRect = characterDetailPanel.AddComponent<RectTransform>();
        panelRect.anchorMin = new Vector2(0, 1);
        panelRect.anchorMax = new Vector2(1, 1);
        panelRect.pivot = new Vector2(0.5f, 1);
        panelRect.anchoredPosition = new Vector2(0, -(topOffset + detailConfig.marginTop));
        panelRect.sizeDelta = new Vector2(-detailConfig.padding * 2, detailConfig.height);
        
        // Pas de fond pour le panel principal (transparent)
        
        // Charger les polices
        TMP_FontAsset latoBoldFont = Resources.Load<TMP_FontAsset>("Fonts/Lato-Bold SDF");
        TMP_FontAsset latoRegularFont = Resources.Load<TMP_FontAsset>("Fonts/Lato-Regular SDF");
        
        // Description courte (en haut, sans cadre)
        GameObject descObj = new GameObject("Description");
        descObj.transform.SetParent(characterDetailPanel.transform, false);
        
        RectTransform descRect = descObj.AddComponent<RectTransform>();
        descRect.anchorMin = new Vector2(0, 1);
        descRect.anchorMax = new Vector2(1, 1);
        descRect.pivot = new Vector2(0.5f, 1);
        descRect.anchoredPosition = new Vector2(0, 0);
        descRect.sizeDelta = new Vector2(-detailConfig.padding * 2, 35);
        
        characterDescriptionText = descObj.AddComponent<TextMeshProUGUI>();
        characterDescriptionText.text = "";
        characterDescriptionText.fontSize = 24;
        characterDescriptionText.color = HexToColor("#64477f");
        characterDescriptionText.alignment = TextAlignmentOptions.Center;
        characterDescriptionText.textWrappingMode = TextWrappingModes.Normal;
        characterDescriptionText.overflowMode = TextOverflowModes.Truncate;
        characterDescriptionText.fontStyle = FontStyles.Normal; // Pas d'italique
        characterDescriptionText.raycastTarget = false;
        if (latoBoldFont != null) characterDescriptionText.font = latoBoldFont;
        
        // Description longue (dans un cadre aux coins arrondis)
        float longDescTop = 45; // Après la description courte
        
        GameObject longDescContainer = new GameObject("LongDescriptionContainer");
        longDescContainer.transform.SetParent(characterDetailPanel.transform, false);
        
        RectTransform longDescContainerRect = longDescContainer.AddComponent<RectTransform>();
        longDescContainerRect.anchorMin = new Vector2(0, 0);
        longDescContainerRect.anchorMax = new Vector2(1, 1);
        longDescContainerRect.pivot = new Vector2(0.5f, 0.5f);
        longDescContainerRect.offsetMin = new Vector2(0, 0);
        longDescContainerRect.offsetMax = new Vector2(0, -longDescTop);
        
        // Fond avec coins arrondis pour la description longue
        Image longDescBg = longDescContainer.AddComponent<Image>();
        longDescBg.sprite = CreateRoundedSprite(600, 100, 12, Color.white);
        longDescBg.color = HexToColor(detailConfig.backgroundColor);
        longDescBg.type = Image.Type.Sliced;
        
        // Texte de la description longue
        GameObject longDescObj = new GameObject("LongDescriptionText");
        longDescObj.transform.SetParent(longDescContainer.transform, false);
        
        RectTransform longDescRect = longDescObj.AddComponent<RectTransform>();
        longDescRect.anchorMin = Vector2.zero;
        longDescRect.anchorMax = Vector2.one;
        longDescRect.offsetMin = new Vector2(15, 10);
        longDescRect.offsetMax = new Vector2(-15, -10);
        
        characterLongDescriptionText = longDescObj.AddComponent<TextMeshProUGUI>();
        characterLongDescriptionText.text = "";
        characterLongDescriptionText.fontSize = 18;
        characterLongDescriptionText.color = HexToColor("#64477f");
        characterLongDescriptionText.alignment = TextAlignmentOptions.Center;
        characterLongDescriptionText.textWrappingMode = TextWrappingModes.Normal;
        characterLongDescriptionText.overflowMode = TextOverflowModes.Ellipsis;
        characterLongDescriptionText.fontStyle = FontStyles.Normal;
        characterLongDescriptionText.raycastTarget = false;
        if (latoRegularFont != null) characterLongDescriptionText.font = latoRegularFont;
        
        // On n'utilise plus characterNameText, mais on garde la référence pour éviter les erreurs
        characterNameText = null;
    }
    
    void RefreshVisibleAvatars()
    {
        if (avatarsContainer == null || currentCharacters == null) return;
        
        var mosaicConfig = config.characterMosaic;
        float avatarSize = mosaicConfig.avatarSize;
        float spacing = mosaicConfig.avatarSpacing;
        int maxVisible = mosaicConfig.columns;
        
        // Supprimer les anciens avatars
        foreach (var avatar in avatarButtons)
        {
            if (avatar != null) Destroy(avatar);
        }
        avatarButtons.Clear();
        
        // Calculer les positions
        float totalWidth = maxVisible * avatarSize + (maxVisible - 1) * spacing;
        float startX = -totalWidth / 2 + avatarSize / 2;
        
        // Créer les avatars visibles (max 3)
        for (int i = 0; i < maxVisible; i++)
        {
            int characterIndex = visibleStartIndex + i;
            if (characterIndex >= currentCharacters.Count) break;
            
            float xPos = startX + i * (avatarSize + spacing);
            CreateAvatarButton(avatarsContainer.transform, currentCharacters[characterIndex], characterIndex, xPos, avatarSize, mosaicConfig);
        }
        
        Debug.Log($"[QuetesTab] 🔄 Avatars rafraîchis: visibleStartIndex={visibleStartIndex}, total={currentCharacters.Count}");
    }
    
    void NavigateCharacter(int direction)
    {
        if (currentCharacters == null || currentCharacters.Count == 0) return;
        
        var mosaicConfig = config.characterMosaic;
        int maxVisible = mosaicConfig.columns;
        
        // Déplacer la fenêtre visible
        visibleStartIndex += direction;
        
        // Bornes
        if (visibleStartIndex < 0)
            visibleStartIndex = 0;
        else if (visibleStartIndex > currentCharacters.Count - maxVisible)
            visibleStartIndex = Mathf.Max(0, currentCharacters.Count - maxVisible);
        
        // Si l'avatar sélectionné n'est plus visible, sélectionner le premier visible
        int visibleEndIndex = visibleStartIndex + maxVisible - 1;
        if (selectedCharacterIndex < visibleStartIndex || selectedCharacterIndex > visibleEndIndex)
        {
            selectedCharacterIndex = visibleStartIndex;
        }
        
        // Rafraîchir l'affichage
        RefreshVisibleAvatars();
        
        // Mettre à jour la sélection
        UpdateSelectedCharacter();
    }
    
    void SelectCharacter(int index)
    {
        if (currentCharacters == null || index < 0 || index >= currentCharacters.Count) return;
        
        selectedCharacterIndex = index;
        UpdateSelectedCharacter();
    }
    
    void UpdateSelectedCharacter()
    {
        if (currentCharacters == null || currentCharacters.Count == 0) return;
        
        var mosaicConfig = config.characterMosaic;
        QuestCharacter character = currentCharacters[selectedCharacterIndex];
        
        // Mettre à jour la bordure pointillée des avatars visibles
        for (int i = 0; i < avatarButtons.Count; i++)
        {
            int characterIndex = visibleStartIndex + i;
            bool isSelected = (characterIndex == selectedCharacterIndex);
            
            // Afficher/cacher la bordure pointillée
            Transform dashedBorder = avatarButtons[i].transform.Find("DashedBorder");
            if (dashedBorder != null)
            {
                dashedBorder.gameObject.SetActive(isSelected);
            }
        }
        
        // Mettre à jour le panneau de détail (sans le nom)
        if (characterDescriptionText != null)
        {
            characterDescriptionText.text = character.GetDescription() ?? "";
        }
        
        if (characterLongDescriptionText != null)
        {
            characterLongDescriptionText.text = character.GetLongDescription() ?? "";
        }
        
        Debug.Log($"[QuetesTab] 👤 Personnage sélectionné: {character.GetName()}");
    }
    
    IEnumerator LoadAvatarSprite(string url, Image targetImage)
    {
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                Sprite sprite = Sprite.Create(texture, new Rect(0, 0, texture.width, texture.height), 
                    new Vector2(0.5f, 0.5f), 100f);
                targetImage.sprite = sprite;
                targetImage.color = Color.white;
            }
            else
            {
                Debug.LogWarning($"[QuetesTab] Échec du chargement de l'avatar: {url}");
            }
        }
    }
    
    string GetAvatarUrl(string avatarFileName)
    {
        // Si c'est déjà une URL complète, la retourner directement
        if (!string.IsNullOrEmpty(avatarFileName) && (avatarFileName.StartsWith("http://") || avatarFileName.StartsWith("https://")))
        {
            return avatarFileName;
        }
        
        // Sinon, ajouter le préfixe
        string uiPath = generalConfig?.assetsPaths?.uiPath ?? "https://ujsa.studioplc.fr/datas/UI/";
        return uiPath + avatarFileName;
    }
    
    void ShowBadgesContent()
    {
        // Nettoyer le contenu existant
        foreach (Transform child in detailsContent.transform)
        {
            Destroy(child.gameObject);
        }
        
        if (selectedQuest == null)
        {
            ShowNoQuestSelectedMessage();
            return;
        }
        
        
        CreateEmptyMessage("Fonctionnalité BADGES - À venir");
    }
    
    void ShowFormationsContent()
    {
        Debug.Log("[QuetesTab] 🎓 ShowFormationsContent() appelé");
        
        // Nettoyer le contenu existant
        foreach (Transform child in detailsContent.transform)
        {
            Destroy(child.gameObject);
        }
        
        if (selectedQuest == null)
        {
            Debug.LogWarning("[QuetesTab] ⚠️ selectedQuest est NULL");
            ShowNoQuestSelectedMessage();
            return;
        }
        
        
        if (selectedQuest.trainings == null || selectedQuest.trainings.Count == 0)
        {
            Debug.LogWarning($"[QuetesTab] ⚠️ Aucune formation pour la quête {selectedQuest.id}");
            CreateEmptyMessage("Aucune formation pour cette quête");
            return;
        }
        
        Debug.Log($"[QuetesTab] 📚 Affichage de {selectedQuest.trainings.Count} formations");
        
        // Container simple avec positionnement absolu
        float yPos = -30f;
        
        foreach (Training training in selectedQuest.trainings)
        {
            // Créer l'item formation (conteneur)
            GameObject item = new GameObject($"Formation_{training.title}");
            item.transform.SetParent(detailsContent.transform, false);
            
            RectTransform itemRect = item.AddComponent<RectTransform>();
            itemRect.anchorMin = new Vector2(0, 1);
            itemRect.anchorMax = new Vector2(1, 1);
            itemRect.pivot = new Vector2(0, 1);
            itemRect.anchoredPosition = new Vector2(30, yPos);
            itemRect.sizeDelta = new Vector2(-60, 50);
            
            // Image flèche (40x40)
            GameObject arrowObj = new GameObject("Arrow");
            arrowObj.transform.SetParent(item.transform, false);
            
            RectTransform arrowRect = arrowObj.AddComponent<RectTransform>();
            arrowRect.anchorMin = new Vector2(0, 0.5f);
            arrowRect.anchorMax = new Vector2(0, 0.5f);
            arrowRect.pivot = new Vector2(0, 0.5f);
            arrowRect.anchoredPosition = Vector2.zero;
            arrowRect.sizeDelta = new Vector2(40, 40);
            
            Image arrowImage = arrowObj.AddComponent<Image>();
            arrowImage.preserveAspect = true;
            
            // Charger l'image de la flèche
            StartCoroutine(LoadFormationArrowSprite(arrowImage));
            
            // Texte du lien
            GameObject textObj = new GameObject("Text");
            textObj.transform.SetParent(item.transform, false);
            
            RectTransform textRect = textObj.AddComponent<RectTransform>();
            textRect.anchorMin = new Vector2(0, 0);
            textRect.anchorMax = new Vector2(1, 1);
            textRect.offsetMin = new Vector2(50, 0); // Décalé après la flèche (40 + 10 marge)
            textRect.offsetMax = Vector2.zero;
            
            TextMeshProUGUI text = textObj.AddComponent<TextMeshProUGUI>();
            text.text = $"<u>{training.title}</u>";
            text.fontSize = 22;
            text.color = HexToColor("#64477f");
            text.alignment = TextAlignmentOptions.MidlineLeft;
            text.richText = true;
            text.enableWordWrapping = true;
            if (textFont != null) text.font = textFont;
            
            // Rendre cliquable
            if (!string.IsNullOrEmpty(training.url))
            {
                Button button = textObj.AddComponent<Button>();
                button.transition = Selectable.Transition.None;
                
                string url = training.url;
                button.onClick.AddListener(() => {
                    Debug.Log($"[QuetesTab] 🔗 Ouverture: {training.title} → {url}");
                    Application.OpenURL(url);
                });
            }
            
            Debug.Log($"[QuetesTab] ✓ Formation créée: {training.title} à Y={yPos}");
            yPos -= 60f; // Hauteur 50 + 10 espacement
        }
        
        Debug.Log($"[QuetesTab] ✅ {selectedQuest.trainings.Count} formations affichées");
    }
    
    void CreateEmptyMessage(string message)
    {
        GameObject messageObj = new GameObject("EmptyMessage");
        messageObj.transform.SetParent(detailsContent.transform, false);
        
        RectTransform rect = messageObj.AddComponent<RectTransform>();
        rect.anchorMin = Vector2.zero;
        rect.anchorMax = Vector2.one;
        rect.offsetMin = Vector2.zero;
        rect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI text = messageObj.AddComponent<TextMeshProUGUI>();
        text.text = message;
        text.fontSize = 24;
        text.color = HexToColor("#8a8a8a");
        text.alignment = TextAlignmentOptions.Center;
        if (textFont != null) text.font = textFont;
    }
    
    List<QuestCharacter> GetQuestCharacters(int questId)
    {
        if (questCharacters.ContainsKey(questId))
        {
            return questCharacters[questId];
        }
        return new List<QuestCharacter>();
    }
    
    // ========== Utilitaires ==========
    
    Sprite CreateRoundedSprite(int width, int height, float radius, Color fillColor)
    {
        Texture2D texture = new Texture2D(width, height);
        Color[] pixels = new Color[width * height];

        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                float alpha = 1f;

                // Coins arrondis
                if (x < radius && y > height - radius)
                {
                    float dx = radius - x;
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }
                else if (x > width - radius && y > height - radius)
                {
                    float dx = x - (width - radius);
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }
                else if (x < radius && y < radius)
                {
                    float dx = radius - x;
                    float dy = radius - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }
                else if (x > width - radius && y < radius)
                {
                    float dx = x - (width - radius);
                    float dy = radius - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }

                pixels[y * width + x] = new Color(fillColor.r, fillColor.g, fillColor.b, fillColor.a * alpha);
            }
        }

        texture.SetPixels(pixels);
        texture.Apply();
        
        dynamicTextures.Add(texture);

        return Sprite.Create(texture, new Rect(0, 0, width, height), new Vector2(0.5f, 0.5f), 100f);
    }
    
    Color HexToColor(string hex)
    {
        if (string.IsNullOrEmpty(hex)) return Color.white;
        
        hex = hex.TrimStart('#');
        
        if (hex.Length == 6)
            hex = hex + "FF";
        
        if (hex.Length == 8)
        {
            byte r = byte.Parse(hex.Substring(0, 2), System.Globalization.NumberStyles.HexNumber);
            byte g = byte.Parse(hex.Substring(2, 2), System.Globalization.NumberStyles.HexNumber);
            byte b = byte.Parse(hex.Substring(4, 2), System.Globalization.NumberStyles.HexNumber);
            byte a = byte.Parse(hex.Substring(6, 2), System.Globalization.NumberStyles.HexNumber);
            
            return new Color(r / 255f, g / 255f, b / 255f, a / 255f);
        }
        
        return Color.white;
    }
}

// ========== Classes de données ==========

[Serializable]
public class CharactersData
{
    public List<QuestCharacter> personnages;
}

[Serializable]
public class QuestCharacter
{
    // Champs français (depuis persos.json local)
    public string nom;
    public string description_courte;
    public string description;
    public string avatar;
    public string avatar_vide;
    public string png;
    
    // Champs anglais (depuis API)
    public string name;
    public string short_description;
    public string long_description;
    public string avatar_url;
    public string image_url;
    
    // Propriétés pour obtenir les valeurs (priorité à l'anglais car c'est ce que renvoie l'API)
    public string GetName() => !string.IsNullOrEmpty(name) ? name : nom;
    public string GetDescription() => !string.IsNullOrEmpty(short_description) ? short_description : description_courte;
    public string GetLongDescription() => !string.IsNullOrEmpty(long_description) ? long_description : description;
    public string GetAvatarUrl() => !string.IsNullOrEmpty(avatar_url) ? avatar_url : (!string.IsNullOrEmpty(image_url) ? image_url : avatar);
}

// Réponse de l'API pour une quête individuelle
[Serializable]
public class QuestDetailApiResponse
{
    public string status;
    public string message;
    public QuestDetailData data;
}

[Serializable]
public class QuestDetailData
{
    public int id;
    public string title;
    public string description;
    public string duration;
    public string trailer_url;
    public List<Training> trainings;
    public List<QuestCharacter> characters; // Personnages de la quête
    public List<QuestBadge> badges;
}

[Serializable]
public class QuestBadge
{
    public int id;
    public string name;
    public string description;
    public string image_url;
    public bool unlocked;
}

/// <summary>
/// Marqueur pour identifier les quêtes verrouillées dans l'onglet Quêtes
/// </summary>
public class LockedQuestMarker : MonoBehaviour { }
