using UnityEngine;
using UnityEngine.UI;

/// <summary>
/// Helper pour créer une ScrollView complète avec la bonne hiérarchie
/// Placez ce script dans un dossier Editor et utilisez le menu Tools
/// </summary>
public class ScrollViewCreator
{
    [UnityEditor.MenuItem("Tools/Create ScrollView for Levels")]
    public static void CreateScrollView()
    {
        Canvas canvas = Object.FindFirstObjectByType<Canvas>();
        if (canvas == null)
        {
            Debug.LogError("Aucun Canvas trouvé dans la scène");
            return;
        }

        // 1. ScrollView parent
        GameObject scrollView = new GameObject("ScrollView");
        scrollView.transform.SetParent(canvas.transform, false);
        
        RectTransform scrollRT = scrollView.AddComponent<RectTransform>();
        scrollRT.anchorMin = Vector2.zero;
        scrollRT.anchorMax = Vector2.one;
        scrollRT.offsetMin = new Vector2(50, 50);
        scrollRT.offsetMax = new Vector2(-50, -50);
        
        ScrollRect scrollRect = scrollView.AddComponent<ScrollRect>();
        
        // 2. Viewport
        GameObject viewport = new GameObject("Viewport");
        viewport.transform.SetParent(scrollView.transform, false);
        
        RectTransform viewportRT = viewport.AddComponent<RectTransform>();
        viewportRT.anchorMin = Vector2.zero;
        viewportRT.anchorMax = Vector2.one;
        viewportRT.offsetMin = Vector2.zero;
        viewportRT.offsetMax = Vector2.zero;
        
        viewport.AddComponent<RectMask2D>();
        
        // 3. Content
        GameObject content = new GameObject("Content");
        content.transform.SetParent(viewport.transform, false);
        
        RectTransform contentRT = content.AddComponent<RectTransform>();
        contentRT.anchorMin = new Vector2(0, 1);
        contentRT.anchorMax = new Vector2(1, 1);
        contentRT.pivot = new Vector2(0.5f, 1);
        contentRT.anchoredPosition = Vector2.zero;
        contentRT.sizeDelta = new Vector2(0, 600); // Hauteur initiale
        
        GridLayoutGroup grid = content.AddComponent<GridLayoutGroup>();
        grid.cellSize = new Vector2(280, 200);
        grid.spacing = new Vector2(15, 15);
        grid.padding = new RectOffset(30, 30, 30, 30);
        grid.constraint = GridLayoutGroup.Constraint.FixedColumnCount;
        grid.constraintCount = 3;
        grid.startCorner = GridLayoutGroup.Corner.UpperLeft;
        grid.startAxis = GridLayoutGroup.Axis.Horizontal;
        grid.childAlignment = TextAnchor.UpperLeft;
        
        // 4. Scrollbar Vertical
        GameObject scrollbar = new GameObject("Scrollbar Vertical");
        scrollbar.transform.SetParent(scrollView.transform, false);
        
        RectTransform scrollbarRT = scrollbar.AddComponent<RectTransform>();
        scrollbarRT.anchorMin = new Vector2(1, 0);
        scrollbarRT.anchorMax = new Vector2(1, 1);
        scrollbarRT.offsetMin = new Vector2(0, 0);
        scrollbarRT.offsetMax = new Vector2(20, 0);
        
        Scrollbar scrollbarComp = scrollbar.AddComponent<Scrollbar>();
        scrollbarComp.direction = Scrollbar.Direction.BottomToTop;
        
        // Scrollbar Handle
        GameObject handle = new GameObject("Handle");
        handle.transform.SetParent(scrollbar.transform, false);
        
        RectTransform handleRT = handle.AddComponent<RectTransform>();
        handleRT.anchorMin = Vector2.zero;
        handleRT.anchorMax = Vector2.one;
        handleRT.offsetMin = Vector2.zero;
        handleRT.offsetMax = Vector2.zero;
        
        Image handleImg = handle.AddComponent<Image>();
        handleImg.color = new Color(0.8f, 0.8f, 0.8f, 0.8f);
        
        scrollbarComp.handleRect = handleRT;
        scrollbarComp.targetGraphic = handleImg;
        
        // 5. Configuration ScrollRect
        scrollRect.content = contentRT;
        scrollRect.viewport = viewportRT;
        scrollRect.verticalScrollbar = scrollbarComp;
        scrollRect.horizontal = false;
        scrollRect.vertical = true;
        scrollRect.movementType = ScrollRect.MovementType.Clamped;
        scrollRect.inertia = true;
        scrollRect.scrollSensitivity = 25f;
        
        // Sélectionner le ScrollView créé
        UnityEditor.Selection.activeGameObject = scrollView;
        
        Debug.Log("ScrollView créée avec succès ! Assignez maintenant les références dans MainMenuManager.");
    }
}