# 🎯 Solution : Crosshair invisible en WebGL (Problème CORS)

## ❌ Le VRAI Problème

Le crosshair ne s'affiche **PLUS en WebGL** après avoir changé l'URL de l'image vers :
```
https://d27dezxjuxsdf7.cloudfront.net/media/2026/01/curseur-arrow-taxi-jaune_6964e52ae822e.png
```

### Cause Identifiée : Headers HTTP + CORS

Le problème vient de **deux sources** :

#### 1. Headers HTTP Interdits en WebGL ❌

Le fichier `MacImageLoader.cs` ajoutait ces headers :
```csharp
www.SetRequestHeader("User-Agent", "Unity/" + Application.unityVersion);
www.SetRequestHeader("Cache-Control", "no-cache");
```

**En WebGL**, les navigateurs **INTERDISENT** ces headers pour des raisons de sécurité, ce qui fait **échouer silencieusement** la requête.

#### 2. Problème CORS Potentiel ⚠️

CloudFront doit envoyer le header :
```
Access-Control-Allow-Origin: *
```

Sans ce header, WebGL ne peut pas charger l'image depuis un autre domaine.

---

## ✅ Solutions Implémentées

### 1. MacImageLoader.cs Corrigé

**Avant** (ligne 51-52) :
```csharp
// Problématique en WebGL
www.SetRequestHeader("User-Agent", "Unity/" + Application.unityVersion);
www.SetRequestHeader("Cache-Control", "no-cache");
```

**Après** :
```csharp
#if !UNITY_WEBGL || UNITY_EDITOR
// Headers pour Mac/Standalone uniquement (pas WebGL)
www.SetRequestHeader("User-Agent", "Unity/" + Application.unityVersion);
www.SetRequestHeader("Cache-Control", "no-cache");
#else
// Mode WebGL : pas de headers personnalisés
Debug.Log("[MacImageLoader] 🌐 Mode WebGL : headers par défaut uniquement");
#endif
```

### 2. Logs de Diagnostic Améliorés

Le fichier affiche maintenant des messages d'erreur spécifiques pour WebGL :
```
🌐 PROBLÈME WEBGL DÉTECTÉ
Causes possibles:
  - CORS bloqué (serveur doit envoyer Access-Control-Allow-Origin)
  - Headers interdits par le navigateur
  - Format d'image non supporté
```

### 3. Script de Test Créé

**Nouveau fichier** : `CrosshairDebugTester.cs`

Ce script permet de tester le chargement de l'image directement en WebGL.

**Utilisation** :
1. Créer un GameObject dans la scène
2. Ajouter le composant `CrosshairDebugTester`
3. Entrer l'URL à tester
4. Compiler en WebGL et lancer
5. Ouvrir la console du navigateur (F12) pour voir les logs

---

## 🧪 Comment Tester

### 1. Dans Unity Editor

Le jeu fonctionne normalement (les headers sont autorisés).

### 2. En WebGL

#### Étape A : Compiler
1. File → Build Settings → WebGL
2. Build and Run

#### Étape B : Vérifier dans le Navigateur
1. Ouvrir la console (F12 → Console)
2. Chercher les logs `[CrosshairManager]` ou `[MacImageLoader]`
3. Regarder s'il y a des erreurs

#### Étape C : Vérifier CORS
Dans la console du navigateur, onglet **Network** :
1. Recharger la page
2. Trouver la requête vers l'image du crosshair
3. Vérifier les **Response Headers**
4. Doit contenir : `Access-Control-Allow-Origin: *`

---

## 🔧 Solutions si le Problème Persiste

### Solution 1 : Configurer CORS sur CloudFront (RECOMMANDÉ)

CloudFront doit envoyer les headers CORS appropriés.

**Configuration CloudFront** :
1. Aller dans la distribution CloudFront
2. Behaviors → Edit
3. Allowed HTTP Methods : GET, HEAD, OPTIONS
4. Cache Policy → Create new ou modifier
5. Headers : Include → `Access-Control-Allow-Origin`

**Ou créer une Custom Response Headers Policy** :
```json
{
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Methods": "GET, OPTIONS",
  "Access-Control-Allow-Headers": "Content-Type"
}
```

### Solution 2 : Héberger l'Image sur le Même Domaine

Si le jeu est sur `ujsa.studioplc.fr`, hébergez aussi l'image là-bas :
```json
{
  "assets": {
    "crosshair": "https://ujsa.studioplc.fr/datas/images/curseur-arrow-taxi-jaune.png"
  }
}
```

Pas de CORS = pas de problème ! ✅

### Solution 3 : Utiliser un Proxy CORS

En dernier recours, utiliser un proxy CORS (non recommandé en production) :
```
https://cors-anywhere.herokuapp.com/https://d27dezxjuxsdf7.cloudfront.net/...
```

### Solution 4 : Encoder l'Image en Base64 (Petit Fichier)

Si l'image est petite (<100KB), l'encoder en base64 dans le JSON :

```json
{
  "assets": {
    "crosshair": "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAA..."
  }
}
```

Le code Unity `UnityWebRequestTexture.GetTexture()` supporte les data URLs.

---

## 🔍 Diagnostic avec CrosshairDebugTester

### Utilisation

1. **Dans Unity** :
   - Créer un GameObject
   - Ajouter `CrosshairDebugTester`
   - Entrer l'URL du crosshair
   - Cliquer sur "Test Crosshair URL" dans le menu contextuel

2. **En WebGL** :
   - Le test s'exécute automatiquement au démarrage
   - Ouvrir F12 → Console pour voir les résultats

### Interprétation des Résultats

#### ✅ Succès
```
✅ SUCCÈS ! Texture chargée:
  - Dimensions: 512x512
  - Format: RGBA32
✅ Sprite créé avec succès !
CONCLUSION: L'image se charge correctement.
```

→ Le problème n'est **PAS** le chargement, mais l'affichage (alpha, visibilité, z-order)

#### ❌ Échec CORS
```
❌ ÉCHEC DE LA REQUÊTE !
Code HTTP: 0
Result: ConnectionError
🌐 PROBLÈME WEBGL DÉTECTÉ
```

→ Configurer CORS sur CloudFront (Solution 1)

#### ❌ Échec 403 Forbidden
```
Code HTTP: 403
```

→ Vérifier les permissions d'accès au fichier

---

## 📊 Comparaison : Pourquoi ça marche sur les autres jeux ?

### Jeux qui Fonctionnent

Images hébergées sur **le même domaine** :
```json
{
  "crosshair": "https://ujsa.studioplc.fr/datas/images/crosshair.png"
}
```
→ Pas de CORS, pas de problème ✅

### Ce Jeu (Problème)

Image hébergée sur **CloudFront** (domaine différent) :
```json
{
  "crosshair": "https://d27dezxjuxsdf7.cloudfront.net/media/2026/01/curseur-arrow-taxi-jaune_6964e52ae822e.png"
}
```
→ CORS requis + Headers WebGL incompatibles ❌

---

## 🎯 Résolution Rapide (Action Immédiate)

### Option A : Déplacer l'Image (5 minutes)

1. Télécharger l'image depuis CloudFront
2. Uploader sur `ujsa.studioplc.fr/datas/images/`
3. Modifier le JSON :
```json
{
  "crosshair": "https://ujsa.studioplc.fr/datas/images/curseur-arrow-taxi-jaune.png"
}
```
4. Rebuild WebGL → FONCTIONNE ✅

### Option B : Configurer CORS CloudFront (30 minutes)

1. AWS Console → CloudFront
2. Modifier la distribution
3. Ajouter les headers CORS
4. Attendre la propagation (5-10 min)
5. Rebuild WebGL → FONCTIONNE ✅

---

## 🛡️ Prévention Future

Pour éviter ce problème à l'avenir :

1. **Toujours héberger les assets sur le même domaine que le jeu**
2. Si CloudFront est nécessaire, **configurer CORS dès le début**
3. **Tester en WebGL** après chaque changement d'URL d'asset
4. Utiliser `CrosshairDebugTester` pour valider les URLs

---

## 📝 Vérification Console Navigateur

### Ce que vous devriez voir (SUCCÈS) :

```
[MacImageLoader] 📥 Chargement depuis: https://...
[MacImageLoader] Platform: WebGLPlayer
[MacImageLoader] 🌐 Mode WebGL : headers par défaut uniquement
[MacImageLoader] ✅ Texture chargée: 512x512
[CrosshairManager] ✅ Sprite appliqué et rendu visible
```

### Ce que vous voyez actuellement (ÉCHEC) :

```
[MacImageLoader] ❌ Erreur réseau: ...
[MacImageLoader] 🌐 PROBLÈME WEBGL DÉTECTÉ
[CrosshairManager] ❌ Échec téléchargement
[CrosshairManager] ⚠️ Création du crosshair de fallback
```

---

## 📞 Checklist de Résolution

- [x] ✅ `MacImageLoader.cs` : Headers WebGL désactivés
- [x] ✅ `CrosshairManager.cs` : Logs WebGL améliorés
- [x] ✅ `CrosshairDebugTester.cs` : Script de test créé
- [ ] ⏳ **CORS CloudFront** : À configurer (ou déplacer l'image)
- [ ] ⏳ **Test WebGL** : Compiler et vérifier

---

## 🎮 Résultat Attendu Final

Après configuration CORS **OU** déplacement de l'image :

1. ✅ Le crosshair se charge en WebGL
2. ✅ Visible et fonctionnel
3. ✅ Change d'apparence sur les zones
4. ✅ Pas de croix de fallback

---

## 📌 Résumé Ultra-Court

**Problème** : Headers HTTP interdits en WebGL + CORS manquant  
**Solution rapide** : Déplacer l'image sur `ujsa.studioplc.fr`  
**Solution propre** : Configurer CORS sur CloudFront  
**Vérification** : Utiliser `CrosshairDebugTester` en WebGL  

---

**Date** : 12 janvier 2026  
**Fichiers modifiés** :
- ✅ `MacImageLoader.cs` (headers WebGL corrigés)
- ✅ `CrosshairManager.cs` (logs améliorés)
- ✅ `CrosshairDebugTester.cs` (créé pour diagnostic)
- ✅ `WebGLCursor.jslib` (bonus : masquer curseur système)
