# 🎯 Solution : Curseur invisible en WebGL

## ❌ Problème Identifié

Le curseur personnalisé (crosshair) n'apparaît pas en **WebGL** alors qu'il fonctionne en **Unity Player**.

### Cause du problème

En WebGL, l'API Unity `Cursor.visible = false` est **ignorée par les navigateurs** pour des raisons de sécurité. Le navigateur continue d'afficher le curseur HTML par défaut, ce qui empêche le crosshair personnalisé de remplacer le curseur système.

---

## ✅ Solution Implémentée

### 1. Plugin JavaScript créé

**Fichier** : `Assets/Plugins/WebGLCursor.jslib`

Ce plugin permet de contrôler le CSS du canvas WebGL pour masquer/afficher le curseur HTML :

```javascript
mergeInto(LibraryManager.library, {
    HideCursor: function() {
        var canvas = document.getElementById("unity-canvas");
        if (canvas) {
            canvas.style.cursor = "none";
        }
        document.body.style.cursor = "none";
    },
    
    ShowCursor: function() {
        var canvas = document.getElementById("unity-canvas");
        if (canvas) {
            canvas.style.cursor = "auto";
        }
        document.body.style.cursor = "auto";
    }
});
```

### 2. Modifications du CrosshairManager.cs

Le fichier `CrosshairManager.cs` a été modifié pour :

- Importer les fonctions JavaScript avec `[DllImport("__Internal")]`
- Créer des méthodes helpers `HideSystemCursor()` et `ShowSystemCursor()`
- Utiliser ces méthodes au lieu de `Cursor.visible` directement
- Distinguer automatiquement entre WebGL et Unity Player avec `#if UNITY_WEBGL`

**Modifications principales** :
- ✅ Ligne ~93 : Import des fonctions JavaScript
- ✅ Ligne ~112 : Méthodes helpers ajoutées
- ✅ Tous les appels à `Cursor.visible` remplacés par les helpers

---

## 🧪 Comment Tester

### 1. Dans Unity Editor
Le jeu fonctionnera **exactement comme avant** - le curseur système sera masqué normalement.

### 2. En WebGL (Build)

1. **Compiler le jeu en WebGL** :
   - File → Build Settings
   - Platform : WebGL
   - Build

2. **Tester dans un navigateur** :
   - Démarrer le jeu
   - Le curseur système devrait **disparaître**
   - Le crosshair personnalisé devrait être **visible**

3. **Vérifier dans la console du navigateur** (F12) :
   - Rechercher les logs : `"🌐 Curseur masqué via JavaScript (WebGL)"`
   - Pas d'erreurs JavaScript liées au curseur

---

## 🔍 Vérifications Supplémentaires

Si le curseur ne disparaît toujours pas en WebGL :

### 1. Vérifier l'ID du canvas

Le script JavaScript cherche un élément avec l'ID `"unity-canvas"`. Vérifiez que votre template WebGL utilise bien cet ID.

**Dans votre template HTML** (`Assets/WebGLTemplates/...`), cherchez :

```html
<canvas id="unity-canvas"></canvas>
```

Si l'ID est différent, modifiez le fichier `WebGLCursor.jslib` pour utiliser le bon ID.

### 2. Vérifier les Player Settings

Dans Unity :
- Edit → Project Settings → Player → WebGL
- Publishing Settings → **Compression Format** : Assurez-vous qu'il n'y a pas de problème de compression

### 3. Logs de debug

Ajoutez dans la console du navigateur :

```javascript
// Vérifier si le canvas a bien cursor:none
var canvas = document.getElementById("unity-canvas");
console.log("Canvas cursor style:", canvas.style.cursor);
```

---

## 🛠️ Solutions Alternatives

Si la solution principale ne fonctionne pas, voici d'autres options :

### Option A : CSS Global dans le template HTML

Ajoutez dans votre fichier `index.html` du template WebGL :

```html
<style>
    #unity-canvas {
        cursor: none !important;
    }
    
    /* Alternative pour tout le body */
    body {
        cursor: none !important;
    }
</style>
```

### Option B : Cursor.lockState (avec restrictions)

Modifiez `CrosshairManager.cs` ligne ~193 :

```csharp
#if UNITY_WEBGL && !UNITY_EDITOR
    HideCursor();
    // Alternative : verrouiller le curseur (demande permission utilisateur)
    Cursor.lockState = CursorLockMode.Locked;
#else
    Cursor.visible = false;
    Cursor.lockState = CursorLockMode.Confined;
#endif
```

**⚠️ Attention** : `CursorLockMode.Locked` demande une permission de l'utilisateur et nécessite un clic sur la page.

### Option C : Sprite Custom en HTML/CSS

Remplacer le curseur par un sprite CSS personnalisé :

```html
<style>
    #unity-canvas {
        cursor: url('data:image/png;base64,...'), auto;
    }
</style>
```

---

## 📝 Configuration du JSON

Votre configuration JSON actuelle semble correcte :

```json
{
    "crosshairConfig": {
        "alwaysShowCrosshair": true,
        "defaultSize": 70,
        "onTargetAlpha": 1,
        "offTargetAlpha": 0.3,
        "onTargetSizeMultiplier": 1,
        "offTargetSizeMultiplier": 0.75
    },
    "assets": {
        "crosshair": "https://d27dezxjuxsdf7.cloudfront.net/media/2026/01/curseur-arrow-taxi-jaune_6964e52ae822e.png"
    }
}
```

Le crosshair sera :
- ✅ Toujours visible
- ✅ Opaque (alpha=1) sur les zones cibles
- ✅ Semi-transparent (alpha=0.3) hors zones
- ✅ Taille normale sur zone, réduit à 75% ailleurs

---

## 🎮 Résultat Attendu

Après compilation WebGL :

1. **Au démarrage du jeu** :
   - Le curseur système disparaît ✅
   - Le crosshair personnalisé apparaît ✅

2. **Pendant le jeu** :
   - Le crosshair suit la souris ✅
   - Change d'apparence sur les zones (taille + opacité) ✅

3. **Pendant le feedback** :
   - Le crosshair disparaît ✅
   - Le curseur système réapparaît ✅

---

## 📞 Support Technique

Si le problème persiste après compilation WebGL :

1. **Ouvrir la console du navigateur** (F12)
2. **Partager les logs** Unity et JavaScript
3. **Vérifier** qu'il n'y a pas d'erreurs liées à `HideCursor` ou `ShowCursor`
4. **Tester** dans différents navigateurs (Chrome, Firefox, Edge)

---

## ✨ Avantages de cette Solution

- ✅ **Compatible** tous navigateurs modernes
- ✅ **Pas de modification** du template HTML
- ✅ **Fonctionne automatiquement** en WebGL
- ✅ **Pas d'impact** sur Unity Player/Editor
- ✅ **Réversible** facilement si besoin
- ✅ **Code propre** avec séparation WebGL/Standalone

---

**Date de modification** : 12 janvier 2026  
**Fichiers modifiés** :
- ✅ `Assets/Plugins/WebGLCursor.jslib` (créé)
- ✅ `Assets/Scripts/Game/CrosshairManager.cs` (modifié)
