using UnityEngine;
using UnityEngine.Networking;
using System.Collections;

/// <summary>
/// Script de debug pour tester le chargement du crosshair en WebGL
/// Attachez ce script à un GameObject dans votre scène pour diagnostiquer les problèmes
/// </summary>
public class CrosshairDebugTester : MonoBehaviour
{
    [Header("Test Configuration")]
    [Tooltip("URL de l'image du crosshair à tester")]
    public string crosshairURL = "https://d27dezxjuxsdf7.cloudfront.net/media/2026/01/curseur-arrow-taxi-jaune_6964e52ae822e.png";
    
    [Tooltip("Tester automatiquement au démarrage")]
    public bool testOnStart = true;
    
    [Header("Debug Info")]
    public bool showDetailedLogs = true;
    
    void Start()
    {
        if (testOnStart)
        {
            TestCrosshairURL();
        }
    }
    
    [ContextMenu("Test Crosshair URL")]
    public void TestCrosshairURL()
    {
        Debug.Log("========== TEST CROSSHAIR WEBGL ==========");
        Debug.Log($"Platform: {Application.platform}");
        Debug.Log($"URL à tester: {crosshairURL}");
        Debug.Log($"IsWebGL: {Application.platform == RuntimePlatform.WebGLPlayer}");
        Debug.Log("==========================================");
        
        StartCoroutine(TestLoadImage());
    }
    
    IEnumerator TestLoadImage()
    {
        if (string.IsNullOrEmpty(crosshairURL))
        {
            Debug.LogError("❌ URL vide !");
            yield break;
        }
        
        Debug.Log($"📥 [1/4] Création de la requête UnityWebRequest...");
        
        using (UnityWebRequest www = UnityWebRequestTexture.GetTexture(crosshairURL))
        {
            www.timeout = 30;
            
            // En WebGL, NE PAS ajouter de headers personnalisés
            #if UNITY_WEBGL && !UNITY_EDITOR
            Debug.Log("🌐 Mode WebGL : pas de headers personnalisés");
            #else
            Debug.Log("🖥️ Mode Standalone : ajout de headers");
            #endif
            
            Debug.Log($"📤 [2/4] Envoi de la requête...");
            float startTime = Time.time;
            
            yield return www.SendWebRequest();
            
            float loadTime = Time.time - startTime;
            Debug.Log($"⏱️ Temps de chargement: {loadTime:F2}s");
            
            Debug.Log($"🔍 [3/4] Analyse de la réponse...");
            Debug.Log($"  - Code HTTP: {www.responseCode}");
            Debug.Log($"  - Result: {www.result}");
            Debug.Log($"  - Error: {(string.IsNullOrEmpty(www.error) ? "AUCUNE" : www.error)}");
            
            if (showDetailedLogs)
            {
                Debug.Log($"  - Data received: {www.downloadHandler?.data?.Length ?? 0} bytes");
                Debug.Log($"  - Download progress: {www.downloadProgress:P0}");
                Debug.Log($"  - Is network error: {www.result == UnityWebRequest.Result.ConnectionError}");
                Debug.Log($"  - Is HTTP error: {www.result == UnityWebRequest.Result.ProtocolError}");
            }
            
            if (www.result == UnityWebRequest.Result.Success)
            {
                try
                {
                    Texture2D texture = ((DownloadHandlerTexture)www.downloadHandler).texture;
                    
                    if (texture != null)
                    {
                        Debug.Log($"✅ [4/4] SUCCÈS ! Texture chargée:");
                        Debug.Log($"  - Dimensions: {texture.width}x{texture.height}");
                        Debug.Log($"  - Format: {texture.format}");
                        Debug.Log($"  - MipMap: {texture.mipmapCount}");
                        Debug.Log($"  - Readable: {texture.isReadable}");
                        
                        // Tester la création du sprite
                        try
                        {
                            Sprite sprite = Sprite.Create(
                                texture,
                                new Rect(0, 0, texture.width, texture.height),
                                new Vector2(0.5f, 0.5f),
                                100f
                            );
                            
                            if (sprite != null)
                            {
                                Debug.Log("✅ Sprite créé avec succès !");
                                Debug.Log("========================================");
                                Debug.Log("CONCLUSION: L'image se charge correctement.");
                                Debug.Log("Le problème est ailleurs (Alpha? Visibilité?)");
                                Debug.Log("========================================");
                            }
                        }
                        catch (System.Exception e)
                        {
                            Debug.LogError($"❌ Erreur création sprite: {e.Message}");
                        }
                    }
                    else
                    {
                        Debug.LogError("❌ [4/4] ÉCHEC ! Texture null après téléchargement");
                        PrintWebGLTroubleshooting();
                    }
                }
                catch (System.Exception e)
                {
                    Debug.LogError($"❌ [4/4] EXCEPTION: {e.Message}");
                    Debug.LogError($"Stack trace: {e.StackTrace}");
                    PrintWebGLTroubleshooting();
                }
            }
            else
            {
                Debug.LogError($"❌ [4/4] ÉCHEC DE LA REQUÊTE !");
                PrintWebGLTroubleshooting();
            }
        }
    }
    
    void PrintWebGLTroubleshooting()
    {
        Debug.Log("========================================");
        Debug.Log("🔧 GUIDE DE DÉPANNAGE");
        Debug.Log("========================================");
        
        #if UNITY_WEBGL && !UNITY_EDITOR
        Debug.Log("Vous êtes en WebGL. Problèmes possibles:");
        Debug.Log("");
        Debug.Log("1. CORS (Cross-Origin Resource Sharing)");
        Debug.Log("   ➜ Le serveur doit envoyer:");
        Debug.Log("     Access-Control-Allow-Origin: *");
        Debug.Log("   ➜ Testez l'URL dans le navigateur (F12 → Network)");
        Debug.Log("");
        Debug.Log("2. Mixed Content (HTTP vs HTTPS)");
        Debug.Log("   ➜ Si votre jeu est en HTTPS, l'image doit l'être aussi");
        Debug.Log($"   ➜ URL actuelle: {crosshairURL}");
        Debug.Log($"   ➜ Protocole: {(crosshairURL.StartsWith("https://") ? "HTTPS ✅" : "HTTP ⚠️")}");
        Debug.Log("");
        Debug.Log("3. Format d'image");
        Debug.Log("   ➜ Formats supportés: PNG, JPG");
        Debug.Log("   ➜ Évitez les PNG avec transparence complexe");
        Debug.Log("");
        Debug.Log("SOLUTION RAPIDE:");
        Debug.Log("➜ Hébergez l'image sur le même domaine que le jeu");
        Debug.Log("➜ Ou configurez CORS sur CloudFront");
        #else
        Debug.Log("Vous êtes en mode Standalone/Editor.");
        Debug.Log("Ce test est surtout utile en WebGL.");
        Debug.Log("Compilez en WebGL pour tester les problèmes CORS.");
        #endif
        
        Debug.Log("========================================");
    }
    
    [ContextMenu("Test with MacImageLoader")]
    public void TestWithMacImageLoader()
    {
        Debug.Log("========== TEST AVEC MACIMAGELOADER ==========");
        StartCoroutine(MacImageLoader.LoadTexture(
            crosshairURL,
            (texture) => {
                Debug.Log($"✅ MacImageLoader SUCCÈS: {texture.width}x{texture.height}");
            },
            (error) => {
                Debug.LogError($"❌ MacImageLoader ÉCHEC: {error}");
            }
        ));
    }
}
