using System;

/// <summary>
/// Représente une entrée de jeu pour la mosaïque du menu principal
/// Structure: id, title, type, questId, questTitle, isIntro, isOutro
/// </summary>
[Serializable]
public class GameEntry
{
    /// <summary>
    /// ID du jeu (utilisé pour appeler l'API)
    /// </summary>
    public int id;
    
    /// <summary>
    /// Titre du jeu à afficher
    /// </summary>
    public string title;
    
    /// <summary>
    /// Type de jeu (shooting, calculator, trous, intro_video, outro_video, text_hole)
    /// </summary>
    public string type;
    
    /// <summary>
    /// ID de la quête à laquelle appartient ce jeu
    /// </summary>
    public int questId;
    
    /// <summary>
    /// Titre de la quête
    /// </summary>
    public string questTitle;
    
    /// <summary>
    /// Indique si c'est une intro de quête
    /// </summary>
    public bool isIntro;
    
    /// <summary>
    /// Indique si c'est un outro de quête
    /// </summary>
    public bool isOutro;
    
    /// <summary>
    /// Position du jeu dans la quête (1, 2, 3...) - 0 pour intro/outro
    /// </summary>
    public int positionInQuest;
    
    /// <summary>
    /// Nombre total de jeux dans la quête (sans compter intro/outro)
    /// </summary>
    public int totalGamesInQuest;
    
    /// <summary>
    /// Indique si c'est un séparateur vide entre les quêtes
    /// </summary>
    public bool isSeparator;
    
    /// <summary>
    /// ID du step (pour l'URL admin)
    /// </summary>
    public string stepId;
    
    /// <summary>
    /// ID de la zone (pour l'URL admin)
    /// </summary>
    public string zoneId;
    
    /// <summary>
    /// Constructeur par défaut
    /// </summary>
    public GameEntry() { }
    
    /// <summary>
    /// Constructeur avec paramètres
    /// </summary>
    public GameEntry(int id, string title, string type, int questId = 0, string questTitle = "", bool isIntro = false, bool isOutro = false, int positionInQuest = 0, int totalGamesInQuest = 0, bool isSeparator = false, string stepId = "", string zoneId = "")
    {
        this.id = id;
        this.title = title;
        this.type = type;
        this.questId = questId;
        this.questTitle = questTitle;
        this.isIntro = isIntro;
        this.isOutro = isOutro;
        this.positionInQuest = positionInQuest;
        this.totalGamesInQuest = totalGamesInQuest;
        this.isSeparator = isSeparator;
        this.stepId = stepId;
        this.zoneId = zoneId;
    }
    
    /// <summary>
    /// Retourne le type de jeu formaté pour l'affichage
    /// </summary>
    public string GetDisplayType()
    {
        if (isIntro) return "INTRO";
        if (isOutro) return "OUTRO";
        
        switch (type?.ToLower())
        {
            case "shooting": return "TIR";
            case "text_hole": return "TEXTE";
            case "trous": return "TEXTE";
            case "calculator": return "CALCUL";
            case "intro_video": return "INTRO";
            case "outro_video": return "OUTRO";
            default: return type?.ToUpper() ?? "";
        }
    }
    
    /// <summary>
    /// Retourne le type de jeu normalisé pour le système
    /// </summary>
    public string GetNormalizedType()
    {
        switch (type?.ToLower())
        {
            case "shooting": return "shooting";
            case "text_hole": return "trous";
            case "trous": return "trous";
            case "calculator": return "calculator";
            case "intro_video": return "intro_video";
            case "outro_video": return "outro_video";
            default: return type?.ToLower() ?? "shooting";
        }
    }
}

