using System.Collections;
using UnityEngine;
using UnityEngine.UI;

/// <summary>
/// Script de diagnostic pour identifier pourquoi la scène Map reste noire
/// À attacher à un GameObject dans la scène Map
/// </summary>
public class DiagnosticMapLoading : MonoBehaviour
{
    void Start()
    {
        StartCoroutine(RunDiagnostics());
    }
    
    IEnumerator RunDiagnostics()
    {
        Debug.Log("═══════════════════════════════════════");
        Debug.Log("    DIAGNOSTIC CHARGEMENT MAP");
        Debug.Log("═══════════════════════════════════════");
        
        // Test 1 : Caméra
        Camera mainCam = Camera.main;
        if (mainCam == null)
        {
            Debug.LogError("❌ PROBLÈME : Pas de caméra principale trouvée !");
            Debug.LogError("   Solution : Ajouter une caméra avec le tag 'MainCamera'");
        }
        else
        {
            Debug.Log($"✅ Caméra trouvée : {mainCam.name}");
            Debug.Log($"   - Position : {mainCam.transform.position}");
            Debug.Log($"   - Clear Flags : {mainCam.clearFlags}");
            Debug.Log($"   - Background Color : {mainCam.backgroundColor}");
            Debug.Log($"   - Culling Mask : {mainCam.cullingMask}");
        }
        
        // Test 2 : Canvas
        Canvas[] canvases = FindObjectsByType<Canvas>(FindObjectsSortMode.None);
        Debug.Log($"✅ {canvases.Length} Canvas trouvé(s)");
        foreach (Canvas canvas in canvases)
        {
            Debug.Log($"   - Canvas : {canvas.name}, RenderMode: {canvas.renderMode}, Active: {canvas.gameObject.activeInHierarchy}");
        }
        
        // Test 3 : GeneralConfigManager
        if (GeneralConfigManager.Instance == null)
        {
            Debug.LogWarning("⚠️ GeneralConfigManager.Instance est NULL - Création automatique...");
            GameObject configManagerObj = new GameObject("GeneralConfigManager");
            configManagerObj.AddComponent<GeneralConfigManager>();
            DontDestroyOnLoad(configManagerObj);
            Debug.Log("✅ GeneralConfigManager créé automatiquement");
            
            // Attendre un peu que le manager soit initialisé
            yield return new WaitForSeconds(0.1f);
        }
        
        if (GeneralConfigManager.Instance != null)
        {
            Debug.Log("✅ GeneralConfigManager trouvé");
            Debug.Log($"   - Config chargée : {GeneralConfigManager.Instance.IsConfigLoaded()}");
            if (GeneralConfigManager.Instance.IsConfigLoaded())
            {
                var config = GeneralConfigManager.Instance.GetConfig();
                Debug.Log($"   - Projet : {config.projectName}");
                Debug.Log($"   - Map par défaut : {GeneralConfigManager.Instance.GetDefaultMapId()}");
            }
        }
        
        // Test 4 : UnifiedLoadingManager
        if (UnifiedLoadingManager.Instance == null)
        {
            GameObject loadingManagerObj = new GameObject("UnifiedLoadingManager");
            loadingManagerObj.AddComponent<UnifiedLoadingManager>();
            DontDestroyOnLoad(loadingManagerObj);
            
            // Attendre un peu que le manager soit initialisé
            yield return new WaitForSeconds(0.1f);
        }
        
        if (UnifiedLoadingManager.Instance != null)
        {
            Debug.Log("✅ UnifiedLoadingManager trouvé");
            // Forcer la fermeture du loading screen
            Debug.Log("   → Tentative de fermeture forcée du loading screen...");
            UnifiedLoadingManager.HideLoadingAfterDelay(0.1f);
        }
        
        // Test 5 : MapManager
        MapManager mapManager = FindFirstObjectByType<MapManager>();
        if (mapManager == null)
        {
            Debug.LogError("❌ PROBLÈME : MapManager introuvable !");
            Debug.LogError("   Solution : Ajouter un GameObject avec le script MapManager dans la scène");
        }
        else
        {
            Debug.Log($"✅ MapManager trouvé : {mapManager.name}");
        }
        
        // Test 6 : Objets UI actifs
        Image[] images = FindObjectsByType<Image>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        Debug.Log($"✅ {images.Length} Image(s) trouvée(s) dans la scène (incluant inactives)");
        int activeImages = 0;
        foreach (Image img in images)
        {
            if (img.gameObject.activeInHierarchy)
            {
                activeImages++;
            }
        }
        Debug.Log($"   - Images actives : {activeImages}");
        
        // Test 7 : EventSystem
        UnityEngine.EventSystems.EventSystem eventSystem = FindFirstObjectByType<UnityEngine.EventSystems.EventSystem>();
        if (eventSystem == null)
        {
            Debug.LogWarning("⚠️ ATTENTION : Pas d'EventSystem trouvé !");
            Debug.LogWarning("   Les interactions UI ne fonctionneront pas");
        }
        else
        {
            Debug.Log($"✅ EventSystem trouvé : {eventSystem.name}");
        }
        
        // Test 8 : Loading Screen forcé caché
        Canvas[] loadingCanvases = FindObjectsByType<Canvas>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        foreach (Canvas canvas in loadingCanvases)
        {
            if (canvas.name.Contains("Loading") || canvas.name.Contains("Chargement"))
            {
                Debug.Log($"⚠️ Canvas de chargement trouvé : {canvas.name}, Active : {canvas.gameObject.activeInHierarchy}");
                if (canvas.gameObject.activeInHierarchy)
                {
                    Debug.Log($"   → Tentative de désactivation de {canvas.name}...");
                    canvas.gameObject.SetActive(false);
                }
            }
        }
        
        Debug.Log("═══════════════════════════════════════");
        Debug.Log("    FIN DU DIAGNOSTIC");
        Debug.Log("═══════════════════════════════════════");
    }
}

