using UnityEngine;
using UnityEngine.SceneManagement;

#if ENABLE_INPUT_SYSTEM
using UnityEngine.InputSystem;
#endif

/// <summary>
/// Gère la touche ESC selon le contexte (jeu ou menu)
/// À attacher à un GameObject dans chaque scène
/// </summary>
public class EscapeKeyHandler : MonoBehaviour
{
    [Header("Configuration")]
    [Tooltip("Action à effectuer quand ESC est pressée")]
    public EscapeAction action = EscapeAction.BackToMenu;
    
    [Tooltip("Nom de la scène menu (pour retour au menu)")]
    public string menuSceneName = "menu";
    
    [Tooltip("Afficher une confirmation avant de quitter")]
    public bool showQuitConfirmation = false;
    
    [Header("Debug")]
    public bool showDebugLogs = false;
    
    private bool isProcessing = false;
    
    void Update()
    {
        // Ne pas traiter si déjà en cours
        if (isProcessing) return;
        
        // Détecter la touche ESC selon le système d'input
        bool escapePressed = false;
        
        #if ENABLE_INPUT_SYSTEM
        // Nouveau Input System
        if (Keyboard.current != null && Keyboard.current.escapeKey.wasPressedThisFrame)
        {
            escapePressed = true;
        }
        #else
        // Ancien Input System
        if (Input.GetKeyDown(KeyCode.Escape))
        {
            escapePressed = true;
        }
        #endif
        
        if (escapePressed)
        {
            HandleEscapeKey();
        }
    }
    
    void HandleEscapeKey()
    {
        isProcessing = true;
        
        if (showDebugLogs)
        {
            Debug.Log($"[EscapeKeyHandler] ESC pressée - Action: {action}");
        }
        
        switch (action)
        {
            case EscapeAction.BackToMenu:
                BackToMenu();
                break;
                
            case EscapeAction.QuitGame:
                QuitGame();
                break;
                
            case EscapeAction.DoNothing:
                if (showDebugLogs)
                {
                    Debug.Log("[EscapeKeyHandler] ESC ignorée (DoNothing)");
                }
                isProcessing = false;
                break;
        }
    }
    
    void BackToMenu()
    {
        if (showDebugLogs)
        {
            Debug.Log($"[EscapeKeyHandler] Retour au menu: {menuSceneName}");
        }
        
        StartCoroutine(BackToMenuWithCleanup());
    }
    
    private System.Collections.IEnumerator BackToMenuWithCleanup()
    {
        // Afficher l'écran de chargement si disponible
        if (UnifiedLoadingManager.Instance != null)
        {
            UnifiedLoadingManager.ShowLoading("Retour au menu...", LoadingContext.Menu);
        }
        
#if UNITY_WEBGL && !UNITY_EDITOR
        Debug.Log("[EscapeKeyHandler] 🧹 WebGL - Nettoyage mémoire avant retour menu...");
        
        // Purger les caches persistants
        try { if (RemoteSpriteCache.GetExisting() != null) RemoteSpriteCache.GetExisting().ClearAll(); } catch { }
        try { if (GameDataManager.Instance != null) GameDataManager.Instance.ClearAssetCache(); } catch { }
        
        // Attendre un frame pour que les Destroy() prennent effet
        yield return null;
        yield return new WaitForEndOfFrame();
        
        // Libérer les assets non référencés
        yield return Resources.UnloadUnusedAssets();
        try { System.GC.Collect(); } catch { }
        
        yield return null;
        Debug.Log("[EscapeKeyHandler] ✅ Nettoyage mémoire WebGL terminé");
#else
        yield return null;
#endif
        
        // Charger la scène menu
        SceneManager.LoadScene(menuSceneName);
    }
    
    void QuitGame()
    {
        if (showDebugLogs)
        {
            Debug.Log("[EscapeKeyHandler] Quitter le jeu");
        }
        
        if (showQuitConfirmation)
        {
            // TODO: Afficher une popup de confirmation si nécessaire
            // Pour l'instant, on quitte directement
            Debug.Log("[EscapeKeyHandler] Confirmation non implémentée, quitter directement");
        }
        
        #if UNITY_EDITOR
        // Dans l'éditeur Unity, arrêter le mode Play
        UnityEditor.EditorApplication.isPlaying = false;
        Debug.Log("[EscapeKeyHandler] Mode Play arrêté (éditeur)");
        #else
        // En build, quitter l'application
        Application.Quit();
        Debug.Log("[EscapeKeyHandler] Application.Quit() appelé");
        #endif
    }
}

/// <summary>
/// Actions possibles pour la touche ESC
/// </summary>
public enum EscapeAction
{
    BackToMenu,  // Retour au menu principal
    QuitGame,    // Quitter complètement le jeu
    DoNothing    // Ne rien faire (pour désactiver temporairement)
}

