using System.Collections;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;

/// <summary>
/// Carte de jeu simplifiée pour la mosaïque du menu
/// Affiche le titre, la position et les boutons JOUER/ADMIN
/// </summary>
public class GameCard : MonoBehaviour, IPointerEnterHandler, IPointerExitHandler
{
    [Header("UI Components")]
    public TextMeshProUGUI titleText;
    public TextMeshProUGUI typeText;
    public Image backgroundImage;
    public Button cardButton;

    [Header("Visual Effects")]
    public float hoverScale = 1.05f;
    public float animationDuration = 0.2f;

    [Header("Colors (not used anymore - colors are now determined by quest)")]
    public Color shootingColor = new Color(0.39f, 0.28f, 0.5f, 1f);
    public Color textHoleColor = new Color(0.8f, 0.58f, 0.26f, 1f);
    public Color calculatorColor = new Color(0.33f, 0.61f, 0.36f, 1f);
    public Color introOutroColor = new Color(0.2f, 0.5f, 0.8f, 1f);
    public Color defaultColor = new Color(0.5f, 0.5f, 0.5f, 1f);

    // Données
    private GameEntry gameEntry;
    private Vector3 originalScale;
    private CanvasGroup canvasGroup;

    void Awake()
    {
        // Ne rien créer ici - MainMenuManager s'en charge
    }
    
    void Start()
    {
        // Initialiser la scale après que le RectTransform soit configuré
        if (originalScale == Vector3.zero)
            originalScale = transform.localScale;
        
        if (originalScale == Vector3.zero)
            originalScale = Vector3.one;
            
        // Récupérer les composants existants
        if (canvasGroup == null)
            canvasGroup = GetComponent<CanvasGroup>();
            
        if (cardButton == null)
            cardButton = GetComponent<Button>();
    }

    /// <summary>
    /// Initialise la carte avec les données du jeu
    /// </summary>
    public void Initialize(GameEntry entry)
    {
        gameEntry = entry;

        if (gameEntry == null)
        {
            Debug.LogError("[GameCard] Données de jeu nulles !");
            return;
        }

        SetupCardVisuals();
        SetupCardInteraction();

        Debug.Log($"[GameCard] Carte initialisée : {gameEntry.title} (ID: {gameEntry.id})");
    }

    /// <summary>
    /// Configure l'apparence visuelle de la carte
    /// </summary>
    void SetupCardVisuals()
    {
        // Titre du jeu
        if (titleText != null)
        {
            titleText.text = gameEntry.title;
            titleText.color = Color.white;
            titleText.fontSize = 16; // Augmenté pour meilleure lisibilité
        }

        // Zone inférieure : Afficher la POSITION + TYPE pour les jeux normaux, le TYPE pour intro/outro
        if (typeText != null)
        {
            // Pour intro/outro : afficher le type
            if (gameEntry.isIntro || gameEntry.isOutro)
            {
                typeText.text = gameEntry.GetDisplayType();
            }
            // Pour les jeux normaux : afficher la position + type (SANS l'ID)
            else if (gameEntry.positionInQuest > 0 && gameEntry.totalGamesInQuest > 0)
            {
                string displayType = gameEntry.GetDisplayType();
                typeText.text = $"{gameEntry.positionInQuest}/{gameEntry.totalGamesInQuest} - {displayType}";
            }
            // Fallback : afficher le type
            else
            {
                typeText.text = gameEntry.GetDisplayType();
            }
            
            typeText.color = new Color(1f, 1f, 1f, 0.9f); // Plus opaque pour mieux voir
            typeText.fontSize = 16; // Police plus grosse
            typeText.fontStyle = FontStyles.Bold; // En gras
        }

        // Couleur de fond selon le type
        if (backgroundImage != null)
        {
            backgroundImage.color = GetTypeColor(gameEntry.type);
        }
    }

    /// <summary>
    /// Configure l'interaction de la carte (maintenant géré par les boutons JOUER et ADMIN)
    /// </summary>
    void SetupCardInteraction()
    {
        // Plus d'interaction sur la carte entière - les boutons JOUER et ADMIN gèrent les actions
    }

    /// <summary>
    /// Retourne la couleur basée sur la quête et le type de jeu
    /// Chaque quête a une couleur de base, avec 3 variantes selon le type
    /// </summary>
    Color GetTypeColor(string type)
    {
        if (gameEntry == null) return defaultColor;
        
        // Obtenir la couleur de base pour cette quête
        Color baseColor = GetQuestBaseColor(gameEntry.questId);
        
        // Appliquer une variante selon le type de jeu
        return GetColorVariant(baseColor, type);
    }
    
    /// <summary>
    /// Retourne une couleur de base pour une quête donnée
    /// Utilise une palette qui alterne pour éviter que deux quêtes consécutives aient la même couleur
    /// </summary>
    Color GetQuestBaseColor(int questId)
    {
        // Palette de 6 couleurs de base
        Color[] palette = new Color[]
        {
            new Color(0.45f, 0.30f, 0.60f, 1f), // Violet
            new Color(0.25f, 0.55f, 0.70f, 1f), // Bleu
            new Color(0.35f, 0.65f, 0.40f, 1f), // Vert
            new Color(0.85f, 0.50f, 0.25f, 1f), // Orange
            new Color(0.70f, 0.30f, 0.45f, 1f), // Rose/Rouge
            new Color(0.55f, 0.60f, 0.35f, 1f)  // Vert olive
        };
        
        // Utiliser le modulo pour cycler à travers la palette
        int colorIndex = questId % palette.Length;
        return palette[colorIndex];
    }
    
    /// <summary>
    /// Applique une variante à une couleur de base selon le type de jeu
    /// </summary>
    Color GetColorVariant(Color baseColor, string type)
    {
        float factor = 1f;
        
        switch (type?.ToLower())
        {
            case "shooting":
            case "tir":
                factor = 1.0f; // Couleur normale (moyenne)
                break;
                
            case "text_hole":
            case "trous":
            case "texte":
                factor = 1.2f; // Plus clair
                break;
                
            case "calculator":
            case "calcul":
                factor = 0.8f; // Plus foncé
                break;
                
            default:
                factor = 1.0f;
                break;
        }
        
        // Appliquer le facteur tout en gardant l'alpha à 1
        return new Color(
            Mathf.Clamp01(baseColor.r * factor),
            Mathf.Clamp01(baseColor.g * factor),
            Mathf.Clamp01(baseColor.b * factor),
            1f
        );
    }

    // Les méthodes de clic et lancement sont maintenant gérées par MainMenuManager via les boutons JOUER/ADMIN

    // ========================================
    // ANIMATIONS
    // ========================================

    IEnumerator AnimateScale(Vector3 targetScale, float duration)
    {
        Vector3 startScale = transform.localScale;
        float elapsed = 0f;

        while (elapsed < duration)
        {
            elapsed += Time.deltaTime;
            float t = Mathf.SmoothStep(0f, 1f, elapsed / duration);
            transform.localScale = Vector3.Lerp(startScale, targetScale, t);
            yield return null;
        }

        transform.localScale = targetScale;
    }

    // ========================================
    // POINTER EVENTS
    // ========================================

    public void OnPointerEnter(PointerEventData eventData)
    {
        Vector3 hoverScaleVector = originalScale * hoverScale;
        StartCoroutine(AnimateScale(hoverScaleVector, animationDuration));
        if (canvasGroup != null) canvasGroup.alpha = 1f;
    }

    public void OnPointerExit(PointerEventData eventData)
    {
        StartCoroutine(AnimateScale(originalScale, animationDuration));
    }

    // ========================================
    // MÉTHODES PUBLIQUES
    // ========================================

    public GameEntry GetGameEntry() => gameEntry;

    void OnDestroy()
    {
        StopAllCoroutines();
    }
}

