using UnityEngine;
using System.Collections;

/// <summary>
/// Script à placer dans CHAQUE scène pour gérer l'écran de transition
/// Affiche un écran noir jusqu'à ce que la scène soit complètement chargée
/// </summary>
public class SceneReadyManager : MonoBehaviour
{
    [Header("Configuration")]
    [Tooltip("Délai minimum d'affichage de l'écran de chargement (en secondes)")]
    [SerializeField] private float minimumLoadingTime = 0.5f;
    
    [Tooltip("Temps d'attente supplémentaire après que tous les managers soient prêts")]
    [SerializeField] private float additionalWaitTime = 0.3f;
    
    [Tooltip("Message de chargement personnalisé pour cette scène")]
    [SerializeField] private string loadingMessage = "Chargement en cours";
    
    [Header("Managers à attendre (optionnel)")]
    [Tooltip("Liste des scripts/managers qui doivent être prêts avant de masquer le loading")]
    [SerializeField] private MonoBehaviour[] managersToWaitFor;
    
    [Header("Debug")]
    [SerializeField] private bool showDebugLogs = true;
    
    private float sceneStartTime;
    private bool isReady = false;
    
    void Awake()
    {
        sceneStartTime = Time.time;
        
        // IMPORTANT : Afficher l'écran de chargement IMMÉDIATEMENT
        if (LoadingScreenManager.Instance != null)
        {
            LoadingScreenManager.Instance.ForceShowLoading(loadingMessage);
            if (showDebugLogs)
                Debug.Log($"[SceneReadyManager] Écran de chargement affiché pour : {UnityEngine.SceneManagement.SceneManager.GetActiveScene().name}");
        }
        // Note: LoadingScreenManager peut ne pas être présent dans toutes les scènes
    }
    
    void Start()
    {
        StartCoroutine(WaitForSceneReady());
    }
    
    IEnumerator WaitForSceneReady()
    {
        if (showDebugLogs)
            Debug.Log($"[SceneReadyManager] Attente du chargement complet de la scène...");
        
        // Attendre le délai minimum
        float elapsedTime = Time.time - sceneStartTime;
        if (elapsedTime < minimumLoadingTime)
        {
            yield return new WaitForSeconds(minimumLoadingTime - elapsedTime);
        }
        
        // Attendre que tous les managers spécifiés soient prêts
        if (managersToWaitFor != null && managersToWaitFor.Length > 0)
        {
            bool allReady = false;
            int maxWaitFrames = 300; // Maximum 5 secondes à 60 FPS
            int frameCount = 0;
            
            while (!allReady && frameCount < maxWaitFrames)
            {
                allReady = true;
                foreach (var manager in managersToWaitFor)
                {
                    if (manager == null || !manager.enabled || !manager.gameObject.activeInHierarchy)
                    {
                        allReady = false;
                        break;
                    }
                }
                
                if (!allReady)
                {
                    yield return null;
                    frameCount++;
                }
            }
            
            if (showDebugLogs)
            {
                if (allReady)
                    Debug.Log($"[SceneReadyManager] Tous les managers sont prêts ({frameCount} frames)");
                else
                    Debug.LogWarning($"[SceneReadyManager] Timeout atteint - certains managers ne sont peut-être pas prêts");
            }
        }
        
        // Attendre quelques frames supplémentaires pour que tout soit bien initialisé
        yield return new WaitForSeconds(additionalWaitTime);
        
        // Attendre la fin de frame pour être sûr que tout est rendu
        yield return new WaitForEndOfFrame();
        
        // Marquer comme prêt
        isReady = true;
        
        if (showDebugLogs)
        {
            float totalTime = Time.time - sceneStartTime;
            Debug.Log($"[SceneReadyManager] Scène prête ! Temps total : {totalTime:F2}s");
        }
        
        // Masquer l'écran de chargement
        if (LoadingScreenManager.Instance != null)
        {
            LoadingScreenManager.Instance.HideLoading();
        }
    }
    
    /// <summary>
    /// Méthode appelable par d'autres scripts pour indiquer que la scène est prête
    /// </summary>
    public void SetSceneReady()
    {
        if (!isReady)
        {
            StopAllCoroutines();
            StartCoroutine(FadeOutLoading());
        }
    }
    
    IEnumerator FadeOutLoading()
    {
        yield return new WaitForSeconds(additionalWaitTime);
        isReady = true;
        
        if (LoadingScreenManager.Instance != null)
        {
            LoadingScreenManager.Instance.HideLoading();
        }
        
        if (showDebugLogs)
            Debug.Log("[SceneReadyManager] Scène marquée comme prête manuellement");
    }
    
    /// <summary>
    /// Propriété pour savoir si la scène est prête
    /// </summary>
    public bool IsReady => isReady;
}

