using UnityEngine;
using UnityEngine.UI;

/// <summary>
/// Dessine un disque (cercle rempli) dans le RectTransform.
/// Utilisé avec le composant Unity UI `Mask` pour obtenir un masquage circulaire fiable (WebGL inclus).
/// </summary>
[RequireComponent(typeof(CanvasRenderer))]
public class CircleMaskGraphic : MaskableGraphic
{
    [Range(12, 128)]
    [SerializeField] private int segments = 48;

    protected override void OnPopulateMesh(VertexHelper vh)
    {
        vh.Clear();

        Rect r = rectTransform.rect;
        float radius = Mathf.Min(r.width, r.height) * 0.5f;
        Vector2 center = r.center;

        // Centre
        UIVertex v = UIVertex.simpleVert;
        v.color = color;
        v.position = center;
        vh.AddVert(v);

        // Points du cercle
        int seg = Mathf.Clamp(segments, 12, 128);
        for (int i = 0; i <= seg; i++)
        {
            float t = (float)i / seg;
            float ang = t * Mathf.PI * 2f;
            float x = Mathf.Cos(ang) * radius + center.x;
            float y = Mathf.Sin(ang) * radius + center.y;

            v.position = new Vector3(x, y, 0f);
            vh.AddVert(v);
        }

        // Triangles (fan)
        for (int i = 1; i <= seg; i++)
        {
            vh.AddTriangle(0, i, i + 1);
        }
    }
}


