using UnityEngine;
using TMPro;
using UnityEngine.UI;
using UnityEngine.Networking;
using System.Collections;
using System.Collections.Generic;

/// <summary>
/// Données d'un joueur dans le classement
/// </summary>
[System.Serializable]
public class ScoreboardPlayer
{
    public int user_id;
    public string first_name;
    public string last_name;
    public string email;
    public string company;
    public string position;
    public int score;
    public int completion;
    public int completed_quests;
    public string badge_type;
    public string badge_url;
    public string visibility;
    public int rank;
}

/// <summary>
/// Données de l'utilisateur depuis /api/ujsa/user
/// </summary>
[System.Serializable]
public class UserApiResponse
{
    public string status;
    public string message;
    public UserData data;
}

[System.Serializable]
public class UserData
{
    public string first_name;
    public string last_name;
    public string email;
    public string position;
    public string company;
    public bool has_membership;
    public int time_used;
    public string leaderboard_visibility;
    public List<SuperBadge> super_badges;
}

[System.Serializable]
public class SuperBadge
{
    public int project_id;
    public string project_slug;
    public string project_title;
    public int score;
    public string badge_url;
    public string badge_type;
}

/// <summary>
/// Données du projet dans le classement
/// </summary>
[System.Serializable]
public class ScoreboardProject
{
    public int id;
    public string title;
    public string slug;
}

/// <summary>
/// Données du scoreboard depuis l'API
/// </summary>
[System.Serializable]
public class ScoreboardData
{
    public ScoreboardProject project;
    public List<ScoreboardPlayer> company;
    public List<ScoreboardPlayer> global;
}

/// <summary>
/// Réponse de l'API scoreboard
/// </summary>
[System.Serializable]
public class ScoreboardApiResponse
{
    public string status;
    public string message;
    public ScoreboardData data;
}

public class ClassementTab : SettingsTab
{
    private GameObject topSection;
    private GameObject tabsSection;
    
    // Références aux images à charger
    private List<(Image image, string url)> pendingImageLoads = new List<(Image, string)>();
    private bool imagesLoaded = false;
    private GameObject rankingListSection;
    
    private Button companyTabButton;
    private Button globalTabButton;
    
    private GameObject companyRankingList;
    private GameObject globalRankingList;
    
    private ScoreboardData scoreboardData;
    private UserData userData;
    private bool isLoadingData = false;
    private bool isLoadingUserData = false;
    
    // Navigation horizontale
    private List<ScoreboardPlayer> currentPlayers = new List<ScoreboardPlayer>();
    private int visibleStartIndex = 0;
    private const int maxVisiblePlayers = 4; // 4 joueurs visibles au lieu de 3
    private GameObject leftArrow;
    private GameObject rightArrow;
    private GameObject playersContainer;
    
    // Checkboxes cliquables
    private Image checkboxPrive;
    private Image checkboxEntreprise;
    private Image checkboxPublic;
    private string currentVisibility = "public"; // Par défaut
    
    // Badge utilisateur
    private Image userBadgeImage;
    private TextMeshProUGUI userScoreText;
    
    // Couleurs depuis l'image
    private Color purpleColor = new Color(0.64f, 0.25f, 1f, 1f); // #a95bfb
    private Color beigeLightColor = new Color(0.96f, 0.93f, 0.90f, 1f); // #f5ece5
    private Color beigeDarkColor = new Color(0.86f, 0.76f, 0.72f, 1f); // #dbc3b7
    private Color textDarkColor = new Color(0.47f, 0.40f, 0.37f, 1f); // #78675f
    
    // Tracking des ressources dynamiques pour éviter les fuites mémoire (WebGL)
    private readonly RuntimeResourceTracker resourceTracker = new RuntimeResourceTracker();
    
    // Configuration
    private ClassementTabConfig config;
    
    protected override void Awake()
    {
        base.Awake();
        tabName = "CLASSEMENT";
        tabOrder = 1;
        
        // Charger la configuration
        LoadConfig();
        
        // Créer le contenu de l'onglet
        CreateContent();
    }
    
    /// <summary>
    /// Charge la configuration depuis GeneralConfigManager
    /// </summary>
    private void LoadConfig()
    {
        if (GeneralConfigManager.Instance != null)
        {
            config = GeneralConfigManager.Instance.GetConfig()?.classementTabConfig;
            
            if (config != null)
            {
                Debug.Log("[ClassementTab] ✅ Configuration chargée depuis general-config.json");
            }
            else
            {
                Debug.LogWarning("[ClassementTab] ⚠️ Configuration non trouvée, utilisation des valeurs par défaut");
                // Créer une config par défaut
                config = new ClassementTabConfig
                {
                    topSection = new ClassementTopSectionConfig { height = 160 },
                    tabsSection = new ClassementTabsSectionConfig { height = 80 },
                    rankingSection = new ClassementRankingSectionConfig { height = 240, maxVisiblePlayers = 4 }
                };
            }
        }
    }
    
    protected override void OnEnable()
    {
        base.OnEnable();
        
        // Charger les données quand l'onglet est activé
        // MAIS pas pendant la création initiale (le panneau est désactivé juste après)
        if (scoreboardData == null && !isLoadingData && gameObject.activeInHierarchy)
        {
            // Attendre une frame pour s'assurer que tout est bien activé
            StartCoroutine(LoadDataAfterFrame());
        }
    }
    
    /// <summary>
    /// Nettoyage des ressources dynamiques pour éviter les fuites mémoire (CRITIQUE en WebGL)
    /// </summary>
    private void OnDestroy()
    {
        Debug.Log("[ClassementTab] 🧹 Nettoyage des ressources dynamiques");
        int count = resourceTracker.Count;
        resourceTracker.Cleanup();
        Debug.Log($"[ClassementTab] ✅ Nettoyage terminé: {count} ressources libérées");
    }
    
    protected override void OnDisable()
    {
        base.OnDisable();
        // Pas de cleanup ici, on le fait seulement dans OnDestroy
    }
    
    /// <summary>
    /// Enregistre une texture et un sprite pour nettoyage ultérieur
    /// </summary>
    private void TrackDynamic(Texture2D tex, Sprite sp)
    {
        resourceTracker.Track(tex, sp);
    }
    
    /// <summary>
    /// Attend une frame avant de charger les données pour s'assurer que l'onglet est bien actif
    /// </summary>
    private IEnumerator LoadDataAfterFrame()
    {
        yield return null; // Attendre 1 frame
        
        // Vérifier qu'on est toujours actif
        if (gameObject.activeInHierarchy && scoreboardData == null && !isLoadingData)
        {
            Debug.Log("[ClassementTab] 🚀 Lancement du chargement des données après activation");
            LoadScoreboardData();
        }
        
        // Charger toutes les images en attente (flèches, checkboxes, etc.)
        if (!imagesLoaded && pendingImageLoads.Count > 0)
        {
            Debug.Log($"[ClassementTab] 🖼️ Chargement de {pendingImageLoads.Count} images en attente");
            StartCoroutine(LoadAllPendingImages());
            imagesLoaded = true;
        }
        
        // Charger les données utilisateur
        if (userData == null && !isLoadingUserData)
        {
            Debug.Log("[ClassementTab] 👤 Chargement des données utilisateur");
            LoadUserData();
        }
    }
    
    /// <summary>
    /// Charge toutes les images en attente
    /// </summary>
    private IEnumerator LoadAllPendingImages()
    {
        foreach (var (image, url) in pendingImageLoads)
        {
            if (image != null && !string.IsNullOrEmpty(url))
            {
                yield return StartCoroutine(LoadImageSprite(url, image));
            }
        }
        pendingImageLoads.Clear();
        Debug.Log("[ClassementTab] ✅ Toutes les images chargées");
    }
    
    /// <summary>
    /// Charge un sprite depuis une URL (méthode universelle)
    /// </summary>
    private IEnumerator LoadImageSprite(string url, Image targetImage)
    {
        Debug.Log($"[ClassementTab] 🔄 Chargement image: {url}");
        
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                Sprite sprite = Sprite.Create(texture, new Rect(0, 0, texture.width, texture.height), 
                    new Vector2(0.5f, 0.5f), 100f);
                
                // Tracker pour nettoyage (WebGL)
                TrackDynamic(texture, sprite);
                
                if (targetImage != null)
                {
                    targetImage.sprite = sprite;
                    Debug.Log($"[ClassementTab] ✅ Image chargée: {texture.width}x{texture.height}");
                }
            }
            else
            {
                Debug.LogError($"[ClassementTab] ❌ Erreur chargement: {url} - {request.error}");
            }
        }
    }
    
    private void CreateContent()
    {
        // Layout vertical principal
        VerticalLayoutGroup mainLayout = gameObject.AddComponent<VerticalLayoutGroup>();
        
        // Récupérer les valeurs de padding depuis la config
        int padLeft = 40, padRight = 40, padTop = 40, padBottom = 40;
        float spacing = 30;
        
        if (config?.mainLayout != null)
        {
            padLeft = (int)config.mainLayout.paddingLeft;
            padRight = (int)config.mainLayout.paddingRight;
            padTop = (int)config.mainLayout.paddingTop;
            padBottom = (int)config.mainLayout.paddingBottom;
            spacing = config.mainLayout.spacing;
        }
        
        mainLayout.padding = new RectOffset(padLeft, padRight, padTop, padBottom);
        mainLayout.spacing = spacing;
        mainLayout.childAlignment = TextAnchor.UpperCenter;
        mainLayout.childControlWidth = true;
        mainLayout.childControlHeight = false;
        mainLayout.childForceExpandWidth = true;
        mainLayout.childForceExpandHeight = false;
        
        // Section supérieure (badge utilisateur)
        CreateTopSection();
        
        // Section des onglets
        CreateTabsSection();
        
        // Section de la liste de classement
        CreateRankingListSection();
    }
    
    /// <summary>
    /// Crée la section supérieure avec badge à gauche et options de confidentialité à droite
    /// </summary>
    private void CreateTopSection()
    {
        topSection = new GameObject("TopSection");
        topSection.transform.SetParent(transform, false);
        
        float height = config?.topSection?.height ?? 160;
        
        RectTransform topRect = topSection.AddComponent<RectTransform>();
        topRect.sizeDelta = new Vector2(0, height);
        
        LayoutElement topLayout = topSection.AddComponent<LayoutElement>();
        topLayout.preferredHeight = height;
        topLayout.minHeight = height;
        
        // Fond depuis le config
        if (config?.topSection != null && !string.IsNullOrEmpty(config.topSection.backgroundColor))
        {
            Image bgImage = topSection.AddComponent<Image>();
            bgImage.color = HexToColor(config.topSection.backgroundColor);
            bgImage.raycastTarget = false;
        }
        
        // Layout horizontal : Badge (gauche) | Options (droite)
        HorizontalLayoutGroup hLayout = topSection.AddComponent<HorizontalLayoutGroup>();
        hLayout.childAlignment = TextAnchor.UpperLeft;
        
        float spacing = config?.topSection?.spacing ?? 40;
        float padLeft = config?.topSection?.padding?.left ?? 60;
        float padRight = config?.topSection?.padding?.right ?? 60;
        float padTop = config?.topSection?.padding?.top ?? 30;
        float padBottom = config?.topSection?.padding?.bottom ?? 30;
        
        hLayout.spacing = spacing;
        hLayout.padding = new RectOffset((int)padLeft, (int)padRight, (int)padTop, (int)padBottom);
        hLayout.childControlWidth = false;
        hLayout.childControlHeight = false;
        hLayout.childForceExpandWidth = false;
        hLayout.childForceExpandHeight = false;
        
        // Badge à gauche
        CreateBadgeVisual(topSection.transform);
        
        // Options de confidentialité à droite
        CreateConfidentialityOptions(topSection.transform);
    }
    
    /// <summary>
    /// Crée le visuel du badge à gauche (avec fond arrondi beige + score)
    /// </summary>
    private void CreateBadgeVisual(Transform parent)
    {
        GameObject badgeContainer = new GameObject("BadgeContainer");
        badgeContainer.transform.SetParent(parent, false);
        
        // Récupérer les valeurs depuis la config
        float badgeWidth = config?.topSection?.badge?.width ?? 150;
        float badgeHeight = config?.topSection?.badge?.height ?? 180;
        float cornerRadius = config?.topSection?.badge?.cornerRadius ?? 20;
        float imageWidth = config?.topSection?.badge?.imageWidth ?? 110;
        float imageHeight = config?.topSection?.badge?.imageHeight ?? 110;
        float scoreFontSize = config?.topSection?.badge?.scoreFontSize ?? 24;
        
        // Récupérer la position du score
        string scoreMode = config?.topSection?.badge?.scorePosition?.mode ?? "below";
        float scoreOffsetX = config?.topSection?.badge?.scorePosition?.offsetX ?? 0;
        float scoreOffsetY = config?.topSection?.badge?.scorePosition?.offsetY ?? 0;
        
        // Couleurs depuis la config
        Color badgeBgColor = beigeLightColor;
        if (config?.topSection?.badge != null && !string.IsNullOrEmpty(config.topSection.badge.backgroundColor))
        {
            badgeBgColor = HexToColor(config.topSection.badge.backgroundColor);
        }
        
        Color scoreColor = purpleColor;
        if (config?.topSection?.badge != null && !string.IsNullOrEmpty(config.topSection.badge.scoreColor))
        {
            scoreColor = HexToColor(config.topSection.badge.scoreColor);
        }
        
        RectTransform badgeRect = badgeContainer.AddComponent<RectTransform>();
        badgeRect.sizeDelta = new Vector2(badgeWidth, badgeHeight);
        
        LayoutElement badgeLayout = badgeContainer.AddComponent<LayoutElement>();
        badgeLayout.preferredWidth = badgeWidth;
        badgeLayout.preferredHeight = badgeHeight;
        
        // Fond arrondi avec couleur de la config
        Image badgeBg = badgeContainer.AddComponent<Image>();
        badgeBg.sprite = CreateRoundedSprite((int)badgeWidth, (int)badgeHeight, cornerRadius, badgeBgColor);
        badgeBg.type = Image.Type.Simple;
        
        bool useOverlay = scoreMode == "overlay";
        
        if (!useOverlay)
        {
            // Mode "below" : Layout vertical classique (badge + score)
            VerticalLayoutGroup vLayout = badgeContainer.AddComponent<VerticalLayoutGroup>();
            vLayout.padding = new RectOffset(10, 10, 15, 10);
            vLayout.spacing = 10;
            vLayout.childAlignment = TextAnchor.UpperCenter;
            vLayout.childControlWidth = false;
            vLayout.childControlHeight = false;
        }
        
        // Image du badge (PNG)
        GameObject badgeImage = new GameObject("BadgeImage");
        badgeImage.transform.SetParent(badgeContainer.transform, false);
        
        RectTransform imgRect = badgeImage.AddComponent<RectTransform>();
        
        if (useOverlay)
        {
            // Mode overlay : badge centré dans le conteneur
            imgRect.anchorMin = new Vector2(0.5f, 0.5f);
            imgRect.anchorMax = new Vector2(0.5f, 0.5f);
            imgRect.pivot = new Vector2(0.5f, 0.5f);
            imgRect.anchoredPosition = Vector2.zero;
        }
        
        imgRect.sizeDelta = new Vector2(imageWidth, imageHeight);
        
        userBadgeImage = badgeImage.AddComponent<Image>();
        userBadgeImage.color = Color.white;
        userBadgeImage.sprite = CreateCircleSprite();
        
        if (!useOverlay)
        {
            LayoutElement imgLayout = badgeImage.AddComponent<LayoutElement>();
            imgLayout.preferredWidth = imageWidth;
            imgLayout.preferredHeight = imageHeight;
        }
        
        // Score
        GameObject scoreObj = new GameObject("ScoreText");
        scoreObj.transform.SetParent(badgeContainer.transform, false);
        
        RectTransform scoreRect = scoreObj.AddComponent<RectTransform>();
        
        if (useOverlay)
        {
            // Mode overlay : score centré sur le badge avec offset
            scoreRect.anchorMin = new Vector2(0.5f, 0.5f);
            scoreRect.anchorMax = new Vector2(0.5f, 0.5f);
            scoreRect.pivot = new Vector2(0.5f, 0.5f);
            scoreRect.anchoredPosition = new Vector2(scoreOffsetX, scoreOffsetY);
            scoreRect.sizeDelta = new Vector2(badgeWidth, 40);
        }
        else
        {
            // Mode below : score positionné normalement sous le badge
            scoreRect.sizeDelta = new Vector2(badgeWidth, 30);
        }
        
        userScoreText = scoreObj.AddComponent<TextMeshProUGUI>();
        userScoreText.text = "0"; // Sera mis à jour
        userScoreText.fontSize = scoreFontSize;
        userScoreText.fontStyle = FontStyles.Bold;
        userScoreText.color = scoreColor;
        userScoreText.alignment = TextAlignmentOptions.Center;
    }
    
    /// <summary>
    /// Crée la section des options de confidentialité à droite
    /// </summary>
    private void CreateConfidentialityOptions(Transform parent)
    {
        GameObject optionsContainer = new GameObject("ConfidentialityOptions");
        optionsContainer.transform.SetParent(parent, false);
        
        // Récupérer les valeurs depuis la config
        float width = config?.topSection?.confidentiality?.width ?? 700;
        float spacing = config?.topSection?.confidentiality?.spacing ?? 12;
        float questionFontSize = config?.topSection?.confidentiality?.questionFontSize ?? 18;
        
        Color questionColor = new Color(0.35f, 0.31f, 0.42f, 1f);
        if (config?.topSection?.confidentiality != null && !string.IsNullOrEmpty(config.topSection.confidentiality.questionColor))
        {
            questionColor = HexToColor(config.topSection.confidentiality.questionColor);
        }
        
        RectTransform optionsRect = optionsContainer.AddComponent<RectTransform>();
        optionsRect.sizeDelta = new Vector2(width, 150);
        
        LayoutElement optionsLayout = optionsContainer.AddComponent<LayoutElement>();
        optionsLayout.preferredWidth = width;
        optionsLayout.preferredHeight = 150;
        
        // Layout vertical pour organiser : Question | Confidentialité | Bouton
        VerticalLayoutGroup vLayout = optionsContainer.AddComponent<VerticalLayoutGroup>();
        vLayout.spacing = spacing;
        vLayout.childAlignment = TextAnchor.UpperLeft;
        vLayout.childControlWidth = false; // NE PAS contrôler la largeur
        vLayout.childControlHeight = false;
        vLayout.childForceExpandWidth = false; // NE PAS forcer l'expansion
        vLayout.childForceExpandHeight = false;
        
        // Question (prend toute la largeur)
        GameObject questionObj = new GameObject("QuestionText");
        questionObj.transform.SetParent(optionsContainer.transform, false);
        
        RectTransform questionRect = questionObj.AddComponent<RectTransform>();
        questionRect.sizeDelta = new Vector2(width, 60); // Largeur complète
        
        LayoutElement questionLayoutElem = questionObj.AddComponent<LayoutElement>();
        questionLayoutElem.preferredWidth = width;
        questionLayoutElem.preferredHeight = 60;
        
        TextMeshProUGUI questionText = questionObj.AddComponent<TextMeshProUGUI>();
        questionText.text = "Souhaites-tu rendre ton super-badge public pour pouvoir te comparer aux autres joueurs ?";
        questionText.fontSize = questionFontSize;
        questionText.fontStyle = FontStyles.Bold;
        questionText.color = questionColor;
        questionText.alignment = TextAlignmentOptions.Left;
        questionText.textWrappingMode = TextWrappingModes.Normal;
        
        // Section Confidentialité
        CreateConfidentialityRadioButtons(optionsContainer.transform);
        
        // Bouton "Exporter mon super-badge"
        CreateExportButton(optionsContainer.transform);
    }
    
    /// <summary>
    /// Crée les boutons radio pour la confidentialité
    /// </summary>
    private void CreateConfidentialityRadioButtons(Transform parent)
    {
        GameObject radioContainer = new GameObject("RadioButtonsContainer");
        radioContainer.transform.SetParent(parent, false);
        
        // Récupérer les valeurs depuis la config
        float radioSpacing = config?.topSection?.confidentiality?.radioSpacing ?? 15;
        float labelFontSize = config?.topSection?.confidentiality?.labelFontSize ?? 15;
        
        RectTransform radioRect = radioContainer.AddComponent<RectTransform>();
        radioRect.sizeDelta = new Vector2(0, 35);
        
        LayoutElement radioLayoutElem = radioContainer.AddComponent<LayoutElement>();
        radioLayoutElem.preferredHeight = 35;
        
        // Layout horizontal pour : "Confidentialité :" + 3 options
        HorizontalLayoutGroup hLayout = radioContainer.AddComponent<HorizontalLayoutGroup>();
        hLayout.spacing = radioSpacing;
        hLayout.childAlignment = TextAnchor.MiddleLeft;
        hLayout.childControlWidth = false;
        hLayout.childControlHeight = false;
        hLayout.childForceExpandWidth = false;
        hLayout.childForceExpandHeight = false;
        
        // Label "Confidentialité :"
        GameObject labelObj = new GameObject("ConfidentialityLabel");
        labelObj.transform.SetParent(radioContainer.transform, false);
        
        TextMeshProUGUI labelText = labelObj.AddComponent<TextMeshProUGUI>();
        labelText.text = "Confidentialité :";
        labelText.fontSize = 16;
        labelText.fontStyle = FontStyles.Bold;
        labelText.color = textDarkColor;
        labelText.alignment = TextAlignmentOptions.Left;
        
        // 3 options radio (cliquables)
        checkboxPrive = CreateRadioOption(radioContainer.transform, "Privé", "private", false, labelFontSize);
        checkboxEntreprise = CreateRadioOption(radioContainer.transform, "Mon entreprise", "company", false, labelFontSize);
        checkboxPublic = CreateRadioOption(radioContainer.transform, "Public", "public", true, labelFontSize); // Coché par défaut
    }
    
    /// <summary>
    /// Crée une option radio avec case à cocher (cliquable)
    /// </summary>
    private Image CreateRadioOption(Transform parent, string label, string visibilityValue, bool isChecked, float fontSize)
    {
        GameObject optionContainer = new GameObject($"RadioOption_{label}");
        optionContainer.transform.SetParent(parent, false);
        
        // Récupérer la taille de la checkbox depuis la config
        float checkboxSize = config?.topSection?.confidentiality?.checkboxSize ?? 58;
        
        RectTransform optionRect = optionContainer.AddComponent<RectTransform>();
        optionRect.sizeDelta = new Vector2(150, 30);
        
        LayoutElement optionLayout = optionContainer.AddComponent<LayoutElement>();
        optionLayout.preferredWidth = 150;
        optionLayout.preferredHeight = 30;
        
        // Layout horizontal pour : case + label
        HorizontalLayoutGroup hLayout = optionContainer.AddComponent<HorizontalLayoutGroup>();
        hLayout.spacing = 8;
        hLayout.childAlignment = TextAnchor.MiddleLeft;
        hLayout.childControlWidth = false;
        hLayout.childControlHeight = false;
        
        // Case à cocher (PNG depuis uiPath)
        GameObject checkboxObj = new GameObject("Checkbox");
        checkboxObj.transform.SetParent(optionContainer.transform, false);
        
        RectTransform checkRect = checkboxObj.AddComponent<RectTransform>();
        checkRect.sizeDelta = new Vector2(checkboxSize, checkboxSize);
        
        Image checkBg = checkboxObj.AddComponent<Image>();
        checkBg.color = Color.white; // Blanc pour que le PNG s'affiche correctement
        
        LayoutElement checkLayout = checkboxObj.AddComponent<LayoutElement>();
        checkLayout.preferredWidth = checkboxSize;
        checkLayout.preferredHeight = checkboxSize;
        
        // Ajouter à la liste des images à charger (sera chargé quand l'onglet est actif)
        string imageName = isChecked ? "check_box_checked.png" : "check_box_unchecked.png";
        string imageUrl = GeneralConfigManager.Instance?.GetUIUrl(imageName) ?? "";
        if (!string.IsNullOrEmpty(imageUrl))
        {
            pendingImageLoads.Add((checkBg, imageUrl));
        }
        
        // Rendre la checkbox cliquable
        Button checkboxButton = checkboxObj.AddComponent<Button>();
        checkboxButton.targetGraphic = checkBg;
        checkboxButton.transition = Selectable.Transition.None;
        checkboxButton.onClick.AddListener(() => OnVisibilityChanged(visibilityValue));
        
        // Label
        GameObject labelObj = new GameObject("Label");
        labelObj.transform.SetParent(optionContainer.transform, false);
        
        TextMeshProUGUI labelText = labelObj.AddComponent<TextMeshProUGUI>();
        labelText.text = label;
        labelText.fontSize = fontSize;
        labelText.color = textDarkColor;
        labelText.alignment = TextAlignmentOptions.Left;
        
        return checkBg; // Retourner l'Image pour pouvoir la mettre à jour
    }
    
    /// <summary>
    /// Appelé quand l'utilisateur change la visibilité (boutons radios)
    /// </summary>
    private void OnVisibilityChanged(string newVisibility)
    {
        if (currentVisibility == newVisibility)
        {
            return; // Déjà sélectionné, ne rien faire
        }
        
        currentVisibility = newVisibility;
        Debug.Log($"[ClassementTab] 🔒 Visibilité changée: {newVisibility}");
        
        // Mettre à jour les checkboxes (comportement radio: une seule cochée)
        UpdateCheckboxes();
        
        // Envoyer à l'API pour sauvegarder la préférence
        StartCoroutine(SaveVisibilityToApi(newVisibility));
    }
    
    /// <summary>
    /// Envoie la nouvelle visibilité à l'API
    /// </summary>
    private IEnumerator SaveVisibilityToApi(string visibility)
    {
        string baseUrl = GeneralConfigManager.Instance?.GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[ClassementTab] ❌ BaseUrl non définie");
            yield break;
        }
        
        string url = $"{baseUrl}/api/ujsa/user/leaderboard-visibility";
        
        // Créer le JSON
        string jsonData = $"{{\"visibility\": \"{visibility}\"}}";
        
        Debug.Log($"[ClassementTab] 📤 POST vers {url} avec données: {jsonData}");
        
        using (UnityWebRequest www = UnityWebRequest.Post(url, jsonData, "application/json"))
        {
            // Ajouter le token d'authentification
            string token = UserDataManager.Instance?.token;
            if (!string.IsNullOrEmpty(token))
            {
                www.SetRequestHeader("Authorization", $"Bearer {token}");
            }
            
            yield return www.SendWebRequest();
            
            if (www.result == UnityWebRequest.Result.Success)
            {
                Debug.Log($"[ClassementTab] ✅ Visibilité sauvegardée: {visibility}");
                
                // Attendre 0.2 secondes puis recharger les données
                yield return new WaitForSeconds(0.2f);
                
                Debug.Log("[ClassementTab] 🔄 Rechargement des données après changement de visibilité");
                
                // Recharger les données utilisateur
                isLoadingUserData = false; // Réinitialiser le flag
                LoadUserData();
                
                // Recharger le classement
                isLoadingData = false; // Réinitialiser le flag
                LoadScoreboardData();
            }
            else
            {
                Debug.LogError($"[ClassementTab] ❌ Erreur sauvegarde visibilité: {www.error}");
                Debug.LogError($"[ClassementTab] ❌ Response: {www.downloadHandler?.text}");
            }
        }
    }
    
    /// <summary>
    /// Met à jour l'affichage des checkboxes
    /// </summary>
    private void UpdateCheckboxes()
    {
        if (checkboxPrive != null)
        {
            string url = GeneralConfigManager.Instance?.GetUIUrl(
                currentVisibility == "private" ? "check_box_checked.png" : "check_box_unchecked.png") ?? "";
            if (!string.IsNullOrEmpty(url))
            {
                StartCoroutine(LoadImageSprite(url, checkboxPrive));
            }
        }
        
        if (checkboxEntreprise != null)
        {
            string url = GeneralConfigManager.Instance?.GetUIUrl(
                currentVisibility == "company" ? "check_box_checked.png" : "check_box_unchecked.png") ?? "";
            if (!string.IsNullOrEmpty(url))
            {
                StartCoroutine(LoadImageSprite(url, checkboxEntreprise));
            }
        }
        
        if (checkboxPublic != null)
        {
            string url = GeneralConfigManager.Instance?.GetUIUrl(
                currentVisibility == "public" ? "check_box_checked.png" : "check_box_unchecked.png") ?? "";
            if (!string.IsNullOrEmpty(url))
            {
                StartCoroutine(LoadImageSprite(url, checkboxPublic));
            }
        }
    }
    
    /// <summary>
    /// Crée le bouton "Exporter mon super-badge" avec le style validationDefault (PAS DE VALEURS EN DUR)
    /// </summary>
    private void CreateExportButton(Transform parent)
    {
        // Récupérer le style depuis la config
        ButtonStyleConfig buttonStyle = null;
        float buttonWidth = 300f; // Valeur par défaut si config échoue
        float buttonHeight = 80f;
        float borderRadius = 35f;
        float borderWidth = 4f;
        Color startColor = HexToColor("#CE9BFD");
        Color endColor = HexToColor("#9A2DFF");
        Color borderColor = HexToColor("#ffffff");
        Color textColor = Color.white;
        float textFontSize = 28f;
        string fontFamily = "Anton-Regular SDF";
        
        if (GeneralConfigManager.Instance != null)
        {
            buttonStyle = GeneralConfigManager.Instance.GetButtonStyle("exportBadgeButton");
            if (buttonStyle != null)
            {
                // Lire TOUTES les valeurs depuis le config
                buttonWidth = buttonStyle.width;
                buttonHeight = buttonStyle.height;
                borderRadius = buttonStyle.borderRadius;
                borderWidth = buttonStyle.borderWidth;
                
                if (buttonStyle.gradient != null && buttonStyle.gradient.enabled)
                {
                    startColor = HexToColor(buttonStyle.gradient.startColor ?? "#CE9BFD");
                    endColor = HexToColor(buttonStyle.gradient.endColor ?? "#9A2DFF");
                }
                
                borderColor = HexToColor(buttonStyle.borderColor ?? "#ffffff");
                
                if (buttonStyle.text != null)
                {
                    textColor = HexToColor(buttonStyle.text.color ?? "#FFFFFF");
                    textFontSize = buttonStyle.text.fontSize;
                    if (!string.IsNullOrEmpty(buttonStyle.text.fontFamily))
                    {
                        fontFamily = buttonStyle.text.fontFamily;
                    }
                }
                
                Debug.Log($"[ClassementTab] ✅ Style exportBadgeButton chargé: {buttonWidth}x{buttonHeight}");
            }
            else
            {
                Debug.LogWarning("[ClassementTab] ⚠️ Style exportBadgeButton introuvable, utilisation des valeurs par défaut");
            }
        }
        
        GameObject buttonObj = new GameObject("ExportButton");
        buttonObj.transform.SetParent(parent, false);
        
        RectTransform buttonRect = buttonObj.AddComponent<RectTransform>();
        buttonRect.sizeDelta = new Vector2(buttonWidth, buttonHeight);
        
        LayoutElement buttonLayout = buttonObj.AddComponent<LayoutElement>();
        buttonLayout.preferredWidth = buttonWidth;
        buttonLayout.preferredHeight = buttonHeight;
        buttonLayout.minWidth = buttonWidth;
        buttonLayout.minHeight = buttonHeight;
        buttonLayout.flexibleWidth = 0;
        buttonLayout.flexibleHeight = 0;
        
        // Image du bouton avec gradient
        Image buttonBg = buttonObj.AddComponent<Image>();
        buttonBg.sprite = CreateGradientSpriteWithBorder((int)buttonWidth, (int)buttonHeight, borderRadius, 
            startColor, endColor, borderColor, borderWidth);
        buttonBg.color = Color.white;
        buttonBg.raycastTarget = true;
        
        // Ombre depuis le style
        if (buttonStyle?.shadow != null && buttonStyle.shadow.enabled)
        {
            Shadow shadow = buttonObj.GetComponent<Shadow>();
            if (shadow == null) shadow = buttonObj.AddComponent<Shadow>();
            shadow.effectColor = HexToColor(buttonStyle.shadow.color ?? "#00000040");
            shadow.effectDistance = new Vector2(buttonStyle.shadow.offsetX, -buttonStyle.shadow.offsetY);
        }
        
        // Ajouter le composant Button
        Button button = buttonObj.AddComponent<Button>();
        button.targetGraphic = buttonBg;
        button.interactable = true;
        
        // Configurer les couleurs du bouton (effets hover/pressed)
        ColorBlock colors = button.colors;
        colors.normalColor = Color.white;
        colors.highlightedColor = new Color(1.1f, 1.1f, 1.1f, 1f);
        colors.pressedColor = new Color(0.9f, 0.9f, 0.9f, 1f);
        button.colors = colors;
        
        // Texte du bouton
        GameObject textObj = new GameObject("ButtonText");
        textObj.transform.SetParent(buttonObj.transform, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.sizeDelta = Vector2.zero;
        textRect.anchoredPosition = Vector2.zero;
        
        TextMeshProUGUI buttonText = textObj.AddComponent<TextMeshProUGUI>();
        buttonText.text = "EXPORTER MON SUPER-BADGE";
        buttonText.fontSize = textFontSize;
        buttonText.fontStyle = FontStyles.Bold;
        buttonText.alignment = TextAlignmentOptions.Center;
        buttonText.verticalAlignment = VerticalAlignmentOptions.Middle;
        buttonText.color = textColor;
        buttonText.enableAutoSizing = false;
        buttonText.raycastTarget = false;
        
        // Charger la police
        TMP_FontAsset customFont = Resources.Load<TMP_FontAsset>($"Fonts/{fontFamily}");
        if (customFont != null)
        {
            buttonText.font = customFont;
        }
        
        // Action du bouton (placeholder pour l'instant)
        button.onClick.AddListener(() => {
            Debug.Log("[ClassementTab] 📤 Export du super-badge demandé");
        });
    }
    
    /// <summary>
    /// Crée la section des onglets (CLASSEMENT CHEZ {COMPANY} / CLASSEMENT GLOBAL)
    /// </summary>
    private void CreateTabsSection()
    {
        tabsSection = new GameObject("TabsSection");
        tabsSection.transform.SetParent(transform, false);
        
        float height = config?.tabsSection?.height ?? 80;
        
        RectTransform tabsRect = tabsSection.AddComponent<RectTransform>();
        tabsRect.sizeDelta = new Vector2(0, height);
        
        LayoutElement tabsLayout = tabsSection.AddComponent<LayoutElement>();
        tabsLayout.preferredHeight = height;
        tabsLayout.minHeight = height;
        
        // Layout horizontal pour les onglets
        HorizontalLayoutGroup hLayout = tabsSection.AddComponent<HorizontalLayoutGroup>();
        
        float spacing = config?.tabsSection?.spacing ?? 20;
        
        hLayout.spacing = spacing;
        hLayout.childAlignment = TextAnchor.MiddleCenter;
        hLayout.childControlWidth = false;
        hLayout.childControlHeight = false;
        hLayout.childForceExpandWidth = false;
        hLayout.childForceExpandHeight = false;
        
        // Récupérer le nom de la compagnie de l'utilisateur
        string companyName = "VOTRE ENTREPRISE"; // Valeur par défaut
        
        if (UserDataManager.Instance != null && !string.IsNullOrEmpty(UserDataManager.Instance.companyName))
        {
            companyName = UserDataManager.Instance.companyName.ToUpper();
        }
        else if (scoreboardData != null && scoreboardData.company != null && scoreboardData.company.Count > 0)
        {
            // Fallback : utiliser le nom de la compagnie du premier joueur
            companyName = scoreboardData.company[0].company.ToUpper();
        }
        
        Debug.Log($"[ClassementTab] 🏢 Nom de la compagnie pour l'onglet: {companyName}");
        
        // Onglet "CLASSEMENT CHEZ {COMPANY}"
        string companyTabLabel = $"CLASSEMENT CHEZ {companyName}";
        companyTabButton = CreateRankingTabButton(companyTabLabel, true);
        companyTabButton.transform.SetParent(tabsSection.transform, false);
        companyTabButton.onClick.AddListener(() => ShowCompanyRanking());
        
        // Onglet "CLASSEMENT GLOBAL"
        globalTabButton = CreateRankingTabButton("CLASSEMENT GLOBAL", false);
        globalTabButton.transform.SetParent(tabsSection.transform, false);
        globalTabButton.onClick.AddListener(() => ShowGlobalRanking());
    }
    
    /// <summary>
    /// Crée un bouton d'onglet pour le classement
    /// </summary>
    private Button CreateRankingTabButton(string label, bool isActive)
    {
        GameObject buttonObj = new GameObject($"Tab_{label}");
        
        RectTransform buttonRect = buttonObj.AddComponent<RectTransform>();
        buttonRect.sizeDelta = new Vector2(340, 70);
        
        LayoutElement buttonLayout = buttonObj.AddComponent<LayoutElement>();
        buttonLayout.preferredWidth = 340;
        buttonLayout.preferredHeight = 70;
        
        // Image du bouton
        Image buttonImage = buttonObj.AddComponent<Image>();
        buttonImage.sprite = CreateRoundedSprite(340, 70, 35f, isActive ? beigeDarkColor : Color.clear);
        buttonImage.type = Image.Type.Simple;
        buttonImage.raycastTarget = true;
        
        Button button = buttonObj.AddComponent<Button>();
        button.targetGraphic = buttonImage;
        button.transition = Selectable.Transition.None;
        
        // Texte du bouton
        GameObject textObj = new GameObject("Text");
        textObj.transform.SetParent(buttonObj.transform, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.offsetMin = Vector2.zero;
        textRect.offsetMax = Vector2.zero;
        
        TextMeshProUGUI text = textObj.AddComponent<TextMeshProUGUI>();
        text.text = label;
        text.fontSize = 26;
        text.fontStyle = FontStyles.Bold;
        text.color = isActive ? Color.white : textDarkColor;
        text.alignment = TextAlignmentOptions.Center;
        
        // Charger la police Anton
        TMP_FontAsset antonFont = Resources.Load<TMP_FontAsset>("Fonts/Anton-Regular SDF");
        if (antonFont != null)
        {
            text.font = antonFont;
        }
        
        return button;
    }
    
    /// <summary>
    /// Crée la section de la liste de classement scrollable (HORIZONTALE avec navigation - PLUS HAUTE)
    /// </summary>
    private void CreateRankingListSection()
    {
        rankingListSection = new GameObject("RankingListSection");
        rankingListSection.transform.SetParent(transform, false);
        
        float height = config?.rankingSection?.height ?? 240;
        
        RectTransform listRect = rankingListSection.AddComponent<RectTransform>();
        listRect.sizeDelta = new Vector2(0, height);
        
        LayoutElement listLayout = rankingListSection.AddComponent<LayoutElement>();
        listLayout.preferredHeight = height;
        listLayout.minHeight = height;
        listLayout.flexibleHeight = 1f; // Prendre tout l'espace restant
        
        // Fond semi-transparent
        Image sectionBg = rankingListSection.AddComponent<Image>();
        sectionBg.color = new Color(0.9f, 0.9f, 0.85f, 0.3f);
        sectionBg.raycastTarget = false;
        
        // Layout horizontal pour: Flèche gauche | Container joueurs | Flèche droite
        HorizontalLayoutGroup hLayout = rankingListSection.AddComponent<HorizontalLayoutGroup>();
        hLayout.childAlignment = TextAnchor.MiddleCenter;
        
        float arrowSpacing = config?.rankingSection?.arrowSpacing ?? 10;
        
        // Padding depuis le config ou valeur par défaut
        int padLeft = 20, padRight = 20, padTop = 20, padBottom = 20;
        if (config?.rankingSection?.padding != null)
        {
            padLeft = (int)config.rankingSection.padding.left;
            padRight = (int)config.rankingSection.padding.right;
            padTop = (int)config.rankingSection.padding.top;
            padBottom = (int)config.rankingSection.padding.bottom;
        }
        
        hLayout.spacing = arrowSpacing;
        hLayout.childControlWidth = false;
        hLayout.childControlHeight = false;
        hLayout.childForceExpandWidth = false;
        hLayout.childForceExpandHeight = false;
        hLayout.padding = new RectOffset(padLeft, padRight, padTop, padBottom);
        
        // Flèche gauche
        leftArrow = CreateNavigationArrow(rankingListSection.transform, true);
        
        // Container pour les joueurs
        playersContainer = new GameObject("PlayersContainer");
        playersContainer.transform.SetParent(rankingListSection.transform, false);
        
        int maxPlayers = config?.rankingSection?.maxVisiblePlayers ?? maxVisiblePlayers;
        float containerWidth = maxPlayers * 220 + (maxPlayers - 1) * 25 + 80; // 4×220 + 3×25 + marge
        
        RectTransform containerRect = playersContainer.AddComponent<RectTransform>();
        containerRect.sizeDelta = new Vector2(containerWidth, 200);
        
        LayoutElement containerLayout = playersContainer.AddComponent<LayoutElement>();
        containerLayout.preferredWidth = containerWidth;
        containerLayout.preferredHeight = 200;
        containerLayout.flexibleWidth = 0;
        
        // Layout horizontal pour les joueurs
        HorizontalLayoutGroup playersLayout = playersContainer.AddComponent<HorizontalLayoutGroup>();
        playersLayout.spacing = 25;
        playersLayout.childAlignment = TextAnchor.MiddleCenter;
        playersLayout.childControlWidth = false;
        playersLayout.childControlHeight = false;
        playersLayout.childForceExpandWidth = false;
        playersLayout.childForceExpandHeight = false;
        
        // Flèche droite
        rightArrow = CreateNavigationArrow(rankingListSection.transform, false);
        
        Debug.Log($"[ClassementTab] 📊 RankingListSection créée (horizontale avec navigation - hauteur: {height})");
    }
    
    /// <summary>
    /// Crée une flèche de navigation (gauche ou droite) - COPIE EXACTE de QuetesTab
    /// </summary>
    private GameObject CreateNavigationArrow(Transform parent, bool isLeft)
    {
        GameObject arrow = new GameObject(isLeft ? "LeftArrow" : "RightArrow");
        arrow.transform.SetParent(parent, false);
        
        RectTransform arrowRect = arrow.AddComponent<RectTransform>();
        arrowRect.sizeDelta = new Vector2(60, 60);
        
        LayoutElement arrowLayout = arrow.AddComponent<LayoutElement>();
        arrowLayout.preferredWidth = 60;
        arrowLayout.preferredHeight = 60;
        arrowLayout.flexibleWidth = 0;
        
        // Image de la flèche
        Image arrowImage = arrow.AddComponent<Image>();
        arrowImage.color = Color.white;
        arrowImage.raycastTarget = true;
        
        // Ajouter à la liste des images à charger (sera chargé quand l'onglet est actif)
        string imageName = isLeft ? "nav_gauche.png" : "nav_droite.png";
        string imageUrl = GeneralConfigManager.Instance?.GetUIUrl(imageName) ?? "";
        if (!string.IsNullOrEmpty(imageUrl))
        {
            pendingImageLoads.Add((arrowImage, imageUrl));
        }
        
        // Bouton
        Button button = arrow.AddComponent<Button>();
        button.transition = Selectable.Transition.ColorTint;
        button.targetGraphic = arrowImage;
        
        ColorBlock colors = button.colors;
        colors.normalColor = Color.white;
        colors.highlightedColor = new Color(1.2f, 1.2f, 1.2f, 1f);
        colors.pressedColor = new Color(0.8f, 0.8f, 0.8f, 1f);
        button.colors = colors;
        
        int direction = isLeft ? -1 : 1;
        button.onClick.AddListener(() => NavigatePlayers(direction));
        
        return arrow;
    }
    
    /// <summary>
    /// Navigue dans la liste des joueurs (gauche/droite)
    /// </summary>
    private void NavigatePlayers(int direction)
    {
        int maxPlayers = config?.rankingSection?.maxVisiblePlayers ?? maxVisiblePlayers;
        visibleStartIndex += direction;
        visibleStartIndex = Mathf.Clamp(visibleStartIndex, 0, Mathf.Max(0, currentPlayers.Count - maxPlayers));
        
        RefreshVisiblePlayers();
        UpdateArrowsVisibility();
    }
    
    /// <summary>
    /// Rafraîchit l'affichage des joueurs visibles
    /// </summary>
    private void RefreshVisiblePlayers()
    {
        int maxPlayers = config?.rankingSection?.maxVisiblePlayers ?? maxVisiblePlayers;
        
        // Nettoyer le container
        foreach (Transform child in playersContainer.transform)
        {
            Destroy(child.gameObject);
        }
        
        // Afficher les joueurs visibles
        int endIndex = Mathf.Min(visibleStartIndex + maxPlayers, currentPlayers.Count);
        
        for (int i = visibleStartIndex; i < endIndex; i++)
        {
            CreatePlayerCard(currentPlayers[i], playersContainer.transform);
        }
        
        Debug.Log($"[ClassementTab] 🔄 Affichage de {endIndex - visibleStartIndex} joueurs ({visibleStartIndex} à {endIndex - 1})");
    }
    
    /// <summary>
    /// Met à jour la visibilité des flèches de navigation
    /// </summary>
    private void UpdateArrowsVisibility()
    {
        int maxPlayers = config?.rankingSection?.maxVisiblePlayers ?? maxVisiblePlayers;
        
        if (leftArrow != null)
        {
            leftArrow.SetActive(visibleStartIndex > 0);
        }
        
        if (rightArrow != null)
        {
            int maxStartIndex = Mathf.Max(0, currentPlayers.Count - maxPlayers);
            rightArrow.SetActive(visibleStartIndex < maxStartIndex);
        }
    }
    
    /// <summary>
    /// Charge les données utilisateur depuis l'API
    /// </summary>
    private void LoadUserData()
    {
        if (GeneralConfigManager.Instance == null)
        {
            Debug.LogError("[ClassementTab] GeneralConfigManager n'est pas initialisé");
            return;
        }
        
        StartCoroutine(LoadUserDataCoroutine());
    }
    
    private IEnumerator LoadUserDataCoroutine()
    {
        isLoadingUserData = true;
        
        // Construire l'URL de l'API
        string baseUrl = GeneralConfigManager.Instance.GetApiBaseUrl();
        string url = $"{baseUrl}/api/ujsa/user";
        
        Debug.Log($"[ClassementTab] 🌐 Appel API utilisateur: {url}");
        
        using (UnityWebRequest www = UnityWebRequest.Get(url))
        {
            // Ajouter le token d'authentification
            string token = UserDataManager.Instance?.token;
            if (!string.IsNullOrEmpty(token))
            {
                www.SetRequestHeader("Authorization", $"Bearer {token}");
            }
            www.SetRequestHeader("Content-Type", "application/json");
            
            yield return www.SendWebRequest();
            
            if (www.result == UnityWebRequest.Result.Success)
            {
                string json = www.downloadHandler.text;
                Debug.Log($"[ClassementTab] ✅ Données utilisateur reçues: {json.Substring(0, Mathf.Min(200, json.Length))}...");
                
                try
                {
                    UserApiResponse response = JsonUtility.FromJson<UserApiResponse>(json);
                    
                    if (response != null && response.data != null)
                    {
                        userData = response.data;
                        Debug.Log($"[ClassementTab] ✅ Utilisateur: {userData.first_name} {userData.last_name}");
                        
                        // Mettre à jour le badge et le score
                        UpdateUserBadge();
                        
                        // Mettre à jour la visibilité
                        if (!string.IsNullOrEmpty(userData.leaderboard_visibility))
                        {
                            currentVisibility = userData.leaderboard_visibility;
                            UpdateCheckboxes();
                        }
                    }
                }
                catch (System.Exception e)
                {
                    Debug.LogError($"[ClassementTab] ❌ Erreur parsing JSON utilisateur: {e.Message}");
                }
            }
            else
            {
                Debug.LogError($"[ClassementTab] ❌ Erreur API utilisateur: {www.error}");
            }
        }
        
        isLoadingUserData = false;
    }
    
    /// <summary>
    /// Met à jour le badge et le score de l'utilisateur
    /// </summary>
    private void UpdateUserBadge()
    {
        if (userData == null || userData.super_badges == null || userData.super_badges.Count == 0)
        {
            Debug.LogWarning("[ClassementTab] ⚠️ Pas de super badge pour l'utilisateur");
            return;
        }
        
        // Trouver le badge du projet actuel
        string projectSlug = GeneralConfigManager.Instance.GetConfig()?.slug;
        SuperBadge badge = userData.super_badges.Find(b => b.project_slug == projectSlug);
        
        if (badge == null)
        {
            badge = userData.super_badges[0]; // Prendre le premier si pas trouvé
        }
        
        // Mettre à jour le score
        if (userScoreText != null)
        {
            userScoreText.text = badge.score.ToString();
        }
        
        // Charger l'image du badge
        if (userBadgeImage != null && !string.IsNullOrEmpty(badge.badge_url))
        {
            StartCoroutine(LoadImageSprite(badge.badge_url, userBadgeImage));
        }
        
        Debug.Log($"[ClassementTab] 🏅 Badge mis à jour: {badge.badge_type}, Score: {badge.score}");
    }
    
    /// <summary>
    /// Charge les données du scoreboard depuis l'API
    /// </summary>
    private void LoadScoreboardData()
    {
        if (GeneralConfigManager.Instance == null)
        {
            Debug.LogError("[ClassementTab] GeneralConfigManager n'est pas initialisé");
            return;
        }
        
        StartCoroutine(LoadScoreboardDataCoroutine());
    }
    
    private IEnumerator LoadScoreboardDataCoroutine()
    {
        isLoadingData = true;
        
        // Construire l'URL de l'API
        string baseUrl = GeneralConfigManager.Instance.GetApiBaseUrl();
        var config = GeneralConfigManager.Instance.GetConfig();
        string projectSlug = config?.slug ?? "ujsa-default";
        
        string apiUrl = $"{baseUrl}/api/ujsa/projects/{projectSlug}/scoreboard";
        
        Debug.Log($"[ClassementTab] 📊 Chargement du scoreboard depuis: {apiUrl}");
        
        // Récupérer le token d'authentification (comme dans UserDataManager et GameDataManager)
        string token = UserDataManager.Instance?.token;
        
        if (!string.IsNullOrEmpty(token))
        {
            Debug.Log($"[ClassementTab] 🔑 Token présent, ajout à la requête");
        }
        else
        {
            Debug.LogWarning("[ClassementTab] ⚠️ Pas de token - requête sans authentification");
        }
        
        using (UnityWebRequest www = UnityWebRequest.Get(apiUrl))
        {
            // Ajouter l'authentification Bearer Token (comme dans UserDataManager)
            if (!string.IsNullOrEmpty(token))
            {
                www.SetRequestHeader("Authorization", $"Bearer {token}");
            }
            
            www.SetRequestHeader("Content-Type", "application/json");
            
            yield return www.SendWebRequest();
            
            if (www.result != UnityWebRequest.Result.Success)
            {
                Debug.LogError($"[ClassementTab] ❌ Erreur chargement scoreboard: {www.error}");
                Debug.LogError($"[ClassementTab] ❌ Code HTTP: {www.responseCode}");
                if (!string.IsNullOrEmpty(www.downloadHandler?.text))
                {
                    Debug.LogError($"[ClassementTab] ❌ Réponse serveur: {www.downloadHandler.text}");
                }
                // Afficher un message d'erreur basique
                currentPlayers = new List<ScoreboardPlayer>();
                RefreshVisiblePlayers();
                isLoadingData = false;
                yield break;
            }
            
            string jsonData = www.downloadHandler.text;
            Debug.Log($"[ClassementTab] ✅ Données brutes reçues ({jsonData.Length} caractères)");
            Debug.Log($"[ClassementTab] 📋 JSON complet: {jsonData}");
            
            try
            {
                ScoreboardApiResponse response = JsonUtility.FromJson<ScoreboardApiResponse>(jsonData);
                
                Debug.Log($"[ClassementTab] 🔍 Parsing terminé - response null? {response == null}");
                if (response != null)
                {
                    Debug.Log($"[ClassementTab] 🔍 response.status = {response.status}");
                    Debug.Log($"[ClassementTab] 🔍 response.data null? {response.data == null}");
                    if (response.data != null)
                    {
                        Debug.Log($"[ClassementTab] 🔍 response.data.company null? {response.data.company == null}");
                        Debug.Log($"[ClassementTab] 🔍 response.data.global null? {response.data.global == null}");
                        if (response.data.company != null)
                        {
                            Debug.Log($"[ClassementTab] 🔍 response.data.company.Count = {response.data.company.Count}");
                        }
                        if (response.data.global != null)
                        {
                            Debug.Log($"[ClassementTab] 🔍 response.data.global.Count = {response.data.global.Count}");
                        }
                    }
                }
                
                if (response != null && response.status == "success" && response.data != null)
                {
                    scoreboardData = response.data;
                    Debug.Log($"[ClassementTab] ✅ Scoreboard chargé: {scoreboardData.company?.Count ?? 0} joueurs (compagnie), {scoreboardData.global?.Count ?? 0} joueurs (global)");
                    
                    // Mettre à jour l'affichage
                    PopulateRankingLists();
                }
                else
                {
                    Debug.LogError($"[ClassementTab] ❌ Réponse API invalide: status={response?.status}, data null? {response?.data == null}");
                    currentPlayers = new List<ScoreboardPlayer>();
                    RefreshVisiblePlayers();
                }
            }
            catch (System.Exception e)
            {
                Debug.LogError($"[ClassementTab] ❌ Erreur parsing JSON: {e.Message}");
                Debug.LogError($"[ClassementTab] ❌ Stack trace: {e.StackTrace}");
                currentPlayers = new List<ScoreboardPlayer>();
                RefreshVisiblePlayers();
            }
        }
        
        isLoadingData = false;
    }
    
    /// <summary>
    /// Remplit les listes de classement avec les données de l'API
    /// </summary>
    private void PopulateRankingLists()
    {
        Debug.Log("[ClassementTab] 🎯 PopulateRankingLists appelé");
        
        if (scoreboardData == null)
        {
            Debug.LogWarning("[ClassementTab] ⚠️ scoreboardData est null");
            return;
        }
        
        Debug.Log($"[ClassementTab] 📊 scoreboardData.company: {scoreboardData.company?.Count ?? -1} joueurs");
        Debug.Log($"[ClassementTab] 🌍 scoreboardData.global: {scoreboardData.global?.Count ?? -1} joueurs");
        
        // Déterminer quelle liste afficher
        bool showCompany = companyTabButton != null && companyTabButton.GetComponent<Image>().color != Color.clear;
        
        if (showCompany && scoreboardData.company != null && scoreboardData.company.Count > 0)
        {
            currentPlayers = new List<ScoreboardPlayer>(scoreboardData.company);
            Debug.Log($"[ClassementTab] 📊 Affichage classement compagnie: {currentPlayers.Count} joueurs");
        }
        else if (scoreboardData.global != null && scoreboardData.global.Count > 0)
        {
            currentPlayers = new List<ScoreboardPlayer>(scoreboardData.global);
            Debug.Log($"[ClassementTab] 🌍 Affichage classement global: {currentPlayers.Count} joueurs");
        }
        else
        {
            currentPlayers = new List<ScoreboardPlayer>();
            Debug.LogWarning("[ClassementTab] ⚠️ Aucune donnée à afficher");
        }
        
        // Réinitialiser l'index de navigation
        visibleStartIndex = 0;
        
        // Afficher les joueurs
        RefreshVisiblePlayers();
        UpdateArrowsVisibility();
        
        Debug.Log("[ClassementTab] ✅ PopulateRankingLists terminé");
    }
    
    
    /// <summary>
    /// Crée une card pour un joueur dans le classement (COMME LE VISUEL FOURNI)
    /// </summary>
    private void CreatePlayerCard(ScoreboardPlayer player, Transform parent)
    {
        GameObject card = new GameObject($"PlayerCard_{player.user_id}");
        card.transform.SetParent(parent, false);
        
        RectTransform cardRect = card.AddComponent<RectTransform>();
        cardRect.sizeDelta = new Vector2(220, 190); // Même taille que le visuel
        
        LayoutElement cardLayout = card.AddComponent<LayoutElement>();
        cardLayout.preferredWidth = 220;
        cardLayout.preferredHeight = 190;
        cardLayout.flexibleWidth = 0;
        
        // Fond beige clair avec coins arrondis (EXACTEMENT comme le visuel)
        Image cardBg = card.AddComponent<Image>();
        cardBg.sprite = CreateRoundedSprite(220, 190, 15f, beigeLightColor);
        cardBg.type = Image.Type.Simple;
        cardBg.raycastTarget = false;
        
        // Layout vertical pour organiser: Badge | Rang | Nom | Entreprise
        VerticalLayoutGroup vLayout = card.AddComponent<VerticalLayoutGroup>();
        vLayout.padding = new RectOffset(15, 15, 15, 10);
        vLayout.spacing = 8;
        vLayout.childAlignment = TextAnchor.UpperCenter;
        vLayout.childControlWidth = false;
        vLayout.childControlHeight = false;
        vLayout.childForceExpandWidth = false;
        vLayout.childForceExpandHeight = false;
        
        // Badge du joueur (en haut, centré)
        GameObject badgeObj = new GameObject("BadgeImage");
        badgeObj.transform.SetParent(card.transform, false);
        
        // Récupérer les dimensions depuis la config
        float badgeWidth = config?.rankingSection?.badge?.width ?? 80;
        float badgeHeight = config?.rankingSection?.badge?.height ?? 80;
        
        RectTransform badgeRect = badgeObj.AddComponent<RectTransform>();
        badgeRect.sizeDelta = new Vector2(badgeWidth, badgeHeight);
        
        Image badgeImage = badgeObj.AddComponent<Image>();
        badgeImage.sprite = CreateCircleSprite();
        badgeImage.color = GetBadgeColor(player.badge_type);
        
        LayoutElement badgeLayout = badgeObj.AddComponent<LayoutElement>();
        badgeLayout.preferredWidth = badgeWidth;
        badgeLayout.preferredHeight = badgeHeight;
        
        // Charger l'image du badge depuis l'URL
        if (!string.IsNullOrEmpty(player.badge_url))
        {
            StartCoroutine(LoadBadgeImage(player.badge_url, badgeImage));
        }
        
        // RANG juste en dessous du badge (GROS et VIOLET comme demandé)
        string rankSuffix = player.rank == 1 ? "er" : "ème";
        CreatePlayerInfoTextCompact(card.transform, $"{player.rank}{rankSuffix}", 28, FontStyles.Bold, purpleColor);
        
        // Nom du joueur (plus petit)
        CreatePlayerInfoTextCompact(card.transform, $"{player.first_name} {player.last_name}", 16, FontStyles.Bold, textDarkColor);
        
        // Entreprise (encore plus petit, style "Entreprise" gris comme le visuel)
        CreatePlayerInfoTextCompact(card.transform, player.company, 14, FontStyles.Normal, new Color(0.5f, 0.45f, 0.42f, 1f));
    }
    
    /// <summary>
    /// Crée un texte d'information pour un joueur (VERSION COMPACTE)
    /// </summary>
    private GameObject CreatePlayerInfoTextCompact(Transform parent, string text, float fontSize, FontStyles fontStyle, Color textColor)
    {
        GameObject textObj = new GameObject("InfoText");
        textObj.transform.SetParent(parent, false);
        
        TextMeshProUGUI textComponent = textObj.AddComponent<TextMeshProUGUI>();
        textComponent.text = text;
        textComponent.fontSize = fontSize;
        textComponent.fontStyle = fontStyle;
        textComponent.color = textColor;
        textComponent.alignment = TextAlignmentOptions.Center;
        textComponent.raycastTarget = false;
        textComponent.textWrappingMode = TextWrappingModes.NoWrap;
        textComponent.overflowMode = TextOverflowModes.Ellipsis; // Tronquer si trop long
        
        LayoutElement textLayout = textObj.AddComponent<LayoutElement>();
        textLayout.preferredHeight = fontSize + 2;
        
        return textObj;
    }
    
    /// <summary>
    /// Retourne une couleur selon le type de badge
    /// </summary>
    private Color GetBadgeColor(string badgeType)
    {
        switch (badgeType?.ToLower())
        {
            case "gold":
            case "or":
                return new Color(1f, 0.84f, 0f, 1f); // Or
            case "silver":
            case "argent":
                return new Color(0.75f, 0.75f, 0.75f, 1f); // Argent
            case "bronze":
                return new Color(0.8f, 0.5f, 0.2f, 1f); // Bronze
            case "unlocked":
            default:
                return new Color(0.64f, 0.25f, 1f, 1f); // Violet par défaut
        }
    }
    
    /// <summary>
    /// Charge l'image du badge depuis une URL (utilise LoadImageSprite)
    /// </summary>
    private IEnumerator LoadBadgeImage(string url, Image targetImage)
    {
        using (UnityWebRequest request = UnityWebRequestTexture.GetTexture(url))
        {
            yield return request.SendWebRequest();
            
            if (request.result == UnityWebRequest.Result.Success)
            {
                Texture2D texture = DownloadHandlerTexture.GetContent(request);
                Sprite sprite = Sprite.Create(
                    texture,
                    new Rect(0, 0, texture.width, texture.height),
                    new Vector2(0.5f, 0.5f),
                    100f
                );
                
                // Tracker pour nettoyage (WebGL)
                TrackDynamic(texture, sprite);
                
                targetImage.sprite = sprite;
                targetImage.color = Color.white; // Enlever la teinte de couleur
            }
            else
            {
                Debug.LogWarning($"[ClassementTab] ⚠️ Impossible de charger l'image du badge: {url}");
            }
        }
    }
    
    /// <summary>
    /// Affiche le classement de la compagnie
    /// </summary>
    private void ShowCompanyRanking()
    {
        // Mettre à jour les styles des onglets
        UpdateTabButtonStyle(companyTabButton, true);
        UpdateTabButtonStyle(globalTabButton, false);
        
        // Recharger les joueurs
        if (scoreboardData != null && scoreboardData.company != null)
        {
            currentPlayers = new List<ScoreboardPlayer>(scoreboardData.company);
            visibleStartIndex = 0;
            RefreshVisiblePlayers();
            UpdateArrowsVisibility();
        }
    }
    
    /// <summary>
    /// Affiche le classement global
    /// </summary>
    private void ShowGlobalRanking()
    {
        // Mettre à jour les styles des onglets
        UpdateTabButtonStyle(companyTabButton, false);
        UpdateTabButtonStyle(globalTabButton, true);
        
        // Recharger les joueurs
        if (scoreboardData != null && scoreboardData.global != null)
        {
            currentPlayers = new List<ScoreboardPlayer>(scoreboardData.global);
            visibleStartIndex = 0;
            RefreshVisiblePlayers();
            UpdateArrowsVisibility();
        }
    }
    
    /// <summary>
    /// Met à jour le style d'un bouton d'onglet
    /// </summary>
    private void UpdateTabButtonStyle(Button button, bool isActive)
    {
        if (button == null) return;
        
        Image buttonImage = button.GetComponent<Image>();
        TextMeshProUGUI buttonText = button.GetComponentInChildren<TextMeshProUGUI>();
        
        if (buttonImage != null)
        {
            buttonImage.color = isActive ? beigeDarkColor : Color.clear;
        }
        
        if (buttonText != null)
        {
            buttonText.color = isActive ? Color.white : textDarkColor;
        }
    }
    
    // Helper methods pour créer des sprites
    
    private Sprite CreateRoundedSprite(int width, int height, float radius, Color fillColor)
    {
        Texture2D texture = new Texture2D(width, height);
        Color[] pixels = new Color[width * height];
        
        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                float alpha = 1f;
                
                // Coins arrondis (simplifié)
                if (x < radius && y > height - radius)
                {
                    float dx = radius - x;
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }
                else if (x > width - radius && y > height - radius)
                {
                    float dx = x - (width - radius);
                    float dy = (height - radius) - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }
                else if (x < radius && y < radius)
                {
                    float dx = radius - x;
                    float dy = radius - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }
                else if (x > width - radius && y < radius)
                {
                    float dx = x - (width - radius);
                    float dy = radius - y;
                    float distance = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distance > radius) alpha = 0f;
                }
                
                pixels[y * width + x] = new Color(fillColor.r, fillColor.g, fillColor.b, fillColor.a * alpha);
            }
        }
        
        texture.SetPixels(pixels);
        texture.Apply();
        
        Sprite sprite = Sprite.Create(texture, new Rect(0, 0, width, height), new Vector2(0.5f, 0.5f), 100f);
        TrackDynamic(texture, sprite);
        return sprite;
    }
    
    private Sprite CreateCircleSprite()
    {
        int size = 128;
        Texture2D texture = new Texture2D(size, size);
        Color[] pixels = new Color[size * size];
        
        float center = size / 2f;
        float radius = size / 2f;
        
        for (int y = 0; y < size; y++)
        {
            for (int x = 0; x < size; x++)
            {
                float distance = Vector2.Distance(new Vector2(x, y), new Vector2(center, center));
                pixels[y * size + x] = distance <= radius ? Color.white : Color.clear;
            }
        }
        
        texture.SetPixels(pixels);
        texture.Apply();
        
        Sprite sprite = Sprite.Create(texture, new Rect(0, 0, size, size), new Vector2(0.5f, 0.5f));
        TrackDynamic(texture, sprite);
        return sprite;
    }
    
    /// <summary>
    /// Crée un sprite avec dégradé vertical et bordure
    /// </summary>
    private Sprite CreateGradientSpriteWithBorder(int width, int height, float radius, Color startColor, Color endColor, Color borderColor, float borderWidth)
    {
        Texture2D texture = new Texture2D(width, height);
        Color[] pixels = new Color[width * height];

        for (int y = 0; y < height; y++)
        {
            for (int x = 0; x < width; x++)
            {
                float alpha = 1f;
                bool isBorder = false;
                
                // Vérifier si on est dans la bordure
                if (x < borderWidth || x >= width - borderWidth || y < borderWidth || y >= height - borderWidth)
                {
                    isBorder = true;
                }

                // Coins arrondis
                float distanceFromCorner = float.MaxValue;
                
                // Coin supérieur gauche
                if (x < radius && y > height - radius)
                {
                    float dx = radius - x;
                    float dy = (height - radius) - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }
                // Coin supérieur droit
                else if (x > width - radius && y > height - radius)
                {
                    float dx = x - (width - radius);
                    float dy = (height - radius) - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }
                // Coin inférieur gauche
                else if (x < radius && y < radius)
                {
                    float dx = radius - x;
                    float dy = radius - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }
                // Coin inférieur droit
                else if (x > width - radius && y < radius)
                {
                    float dx = x - (width - radius);
                    float dy = radius - y;
                    distanceFromCorner = Mathf.Sqrt(dx * dx + dy * dy);
                    if (distanceFromCorner > radius) alpha = 0f;
                    else if (distanceFromCorner > radius - borderWidth) isBorder = true;
                }

                Color pixelColor;
                if (isBorder && alpha > 0)
                {
                    pixelColor = borderColor;
                }
                else
                {
                    // Dégradé vertical
                    float t = (float)y / height;
                    pixelColor = Color.Lerp(endColor, startColor, t);
                }
                
                pixels[y * width + x] = new Color(pixelColor.r, pixelColor.g, pixelColor.b, pixelColor.a * alpha);
            }
        }

        texture.SetPixels(pixels);
        texture.Apply();
        
        Sprite sprite = Sprite.Create(texture, new Rect(0, 0, width, height), new Vector2(0.5f, 0.5f), 100f);
        TrackDynamic(texture, sprite);
        return sprite;
    }
    
    /// <summary>
    /// Convertit une couleur hexadécimale en Color Unity
    /// </summary>
    private Color HexToColor(string hex)
    {
        hex = hex.TrimStart('#');
        
        if (hex.Length == 6)
            hex += "FF"; // Ajouter l'alpha si absent
        
        if (hex.Length != 8)
            return Color.white;
        
        byte r = System.Convert.ToByte(hex.Substring(0, 2), 16);
        byte g = System.Convert.ToByte(hex.Substring(2, 2), 16);
        byte b = System.Convert.ToByte(hex.Substring(4, 2), 16);
        byte a = System.Convert.ToByte(hex.Substring(6, 2), 16);
        
        return new Color32(r, g, b, a);
    }
}