using UnityEngine;

/// <summary>
/// Système de logging centralisé avec niveaux de verbosité.
/// Permet de réduire le bruit dans la console en production.
/// </summary>
public static class GameLogger
{
    /// <summary>
    /// Niveau de verbosité des logs
    /// </summary>
    public enum LogLevel
    {
        None = 0,      // Aucun log
        Error = 1,     // Erreurs uniquement
        Warning = 2,   // Erreurs + Warnings
        Info = 3,      // Logs importants (connexion, chargement terminé, etc.)
        Verbose = 4    // Tous les logs (debug)
    }
    
    /// <summary>
    /// Niveau de log actuel. Modifier cette valeur pour ajuster la verbosité.
    /// En production, utiliser LogLevel.Info ou LogLevel.Warning
    /// </summary>
#if UNITY_EDITOR
    public static LogLevel CurrentLevel = LogLevel.Info; // Plus de logs en éditeur
#else
    public static LogLevel CurrentLevel = LogLevel.Warning; // Moins de logs en build
#endif
    
    /// <summary>
    /// Log de niveau Verbose (debug détaillé)
    /// </summary>
    public static void Verbose(string tag, string message)
    {
        if (CurrentLevel >= LogLevel.Verbose)
        {
            Debug.Log($"[{tag}] {message}");
        }
    }
    
    /// <summary>
    /// Log de niveau Info (informations importantes)
    /// </summary>
    public static void Info(string tag, string message)
    {
        if (CurrentLevel >= LogLevel.Info)
        {
            Debug.Log($"[{tag}] {message}");
        }
    }
    
    /// <summary>
    /// Log de niveau Warning
    /// </summary>
    public static void Warning(string tag, string message)
    {
        if (CurrentLevel >= LogLevel.Warning)
        {
            Debug.LogWarning($"[{tag}] {message}");
        }
    }
    
    /// <summary>
    /// Log de niveau Error (toujours affiché sauf si None)
    /// </summary>
    public static void Error(string tag, string message)
    {
        if (CurrentLevel >= LogLevel.Error)
        {
            Debug.LogError($"[{tag}] ❌ {message}");
        }
    }
    
    /// <summary>
    /// Active le mode debug (tous les logs)
    /// </summary>
    public static void EnableDebugMode()
    {
        CurrentLevel = LogLevel.Verbose;
        Debug.Log("[GameLogger] Mode debug activé - tous les logs sont visibles");
    }
    
    /// <summary>
    /// Désactive les logs (sauf erreurs)
    /// </summary>
    public static void EnableProductionMode()
    {
        CurrentLevel = LogLevel.Error;
    }
}

