using UnityEngine;
using UnityEngine.UI;
using System.Runtime.InteropServices;

/// <summary>
/// Handler pour capturer les événements de scroll en WebGL via JavaScript
/// et les transmettre au ScrollRect
/// </summary>
[RequireComponent(typeof(ScrollRect))]
public class WebGLScrollHandler : MonoBehaviour
{
#if UNITY_WEBGL && !UNITY_EDITOR
    [DllImport("__Internal")]
    private static extern void InitWebGLScrollListener(string gameObjectName, string callbackName);
#endif

    private ScrollRect scrollRect;
    
    [Header("Paramètres de scroll")]
    [Tooltip("Sensibilité du scroll (plus élevé = scroll plus rapide)")]
    public float scrollSpeed = 20f;
    
    [Tooltip("Activer les logs de debug")]
    public bool debugLogs = true;

    void Start()
    {
        scrollRect = GetComponent<ScrollRect>();
        
        if (scrollRect == null)
        {
            Debug.LogError("[WebGLScrollHandler] Pas de ScrollRect trouvé sur ce GameObject!");
            enabled = false;
            return;
        }
        
        Debug.Log("[WebGLScrollHandler] 🚀 Initialisation du handler de scroll WebGL");
        
#if UNITY_WEBGL && !UNITY_EDITOR
        try
        {
            Debug.Log($"[WebGLScrollHandler] Appel de InitWebGLScrollListener pour '{gameObject.name}'");
            InitWebGLScrollListener(gameObject.name, "OnWebGLScroll");
            Debug.Log("[WebGLScrollHandler] ✅ Listener JavaScript initialisé");
        }
        catch (System.Exception e)
        {
            Debug.LogError($"[WebGLScrollHandler] ❌ Erreur lors de l'initialisation: {e.Message}");
        }
#else
        Debug.Log("[WebGLScrollHandler] Mode Editor/Standalone - Listener JS non initialisé");
#endif
    }

    /// <summary>
    /// Méthode appelée par JavaScript quand un événement wheel est détecté
    /// </summary>
    /// <param name="deltaY">Delta vertical du scroll (négatif = scroll vers le haut)</param>
    public void OnWebGLScroll(float deltaY)
    {
        if (scrollRect == null) return;
        
        if (debugLogs)
        {
            Debug.Log($"[WebGLScrollHandler] 🖱️ Scroll reçu de JS ! deltaY: {deltaY}");
        }
        
        // Convertir le delta en mouvement de scroll
        // deltaY positif = scroll vers le bas = diminuer normalizedPosition
        // deltaY négatif = scroll vers le haut = augmenter normalizedPosition
        
        float scrollDelta = -deltaY / scrollSpeed;
        float newPosition = Mathf.Clamp01(scrollRect.verticalNormalizedPosition + scrollDelta);
        
        if (debugLogs)
        {
            Debug.Log($"[WebGLScrollHandler] Position: {scrollRect.verticalNormalizedPosition:F3} -> {newPosition:F3}");
        }
        
        scrollRect.verticalNormalizedPosition = newPosition;
    }
    
    void OnDestroy()
    {
        Debug.Log("[WebGLScrollHandler] Handler détruit");
    }
}
