# Script PowerShell pour télécharger les images depuis CloudFront
# Et les préparer pour upload sur ujsa.studioplc.fr

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "  TÉLÉCHARGEMENT IMAGES CLOUDFRONT" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host ""

# Créer un dossier pour les téléchargements
$downloadFolder = Join-Path $PSScriptRoot "images_downloaded"
if (-not (Test-Path $downloadFolder)) {
    New-Item -ItemType Directory -Path $downloadFolder | Out-Null
    Write-Host "✅ Dossier créé: $downloadFolder" -ForegroundColor Green
} else {
    Write-Host "📁 Dossier existe déjà: $downloadFolder" -ForegroundColor Yellow
}

# Liste des images à télécharger
$images = @(
    @{
        Url = "https://d27dezxjuxsdf7.cloudfront.net/media/2026/01/curseur-arrow-taxi-jaune_6964e52ae822e.png"
        Nom = "curseur-arrow-taxi-jaune.png"
        Description = "Crosshair (viseur)"
    },
    @{
        Url = "https://d27dezxjuxsdf7.cloudfront.net/media/2026/01/impact-flash-light_6964e940c1faf.png"
        Nom = "impact-flash-light.png"
        Description = "Impact (effet de tir)"
    }
)

Write-Host "Téléchargement de $($images.Count) images..." -ForegroundColor Cyan
Write-Host ""

$success = 0
$failed = 0

foreach ($image in $images) {
    $destination = Join-Path $downloadFolder $image.Nom
    
    Write-Host "📥 Téléchargement: $($image.Description)" -ForegroundColor White
    Write-Host "   URL: $($image.Url)" -ForegroundColor Gray
    Write-Host "   Destination: $($image.Nom)" -ForegroundColor Gray
    
    try {
        # Télécharger avec Invoke-WebRequest
        $ProgressPreference = 'SilentlyContinue' # Désactiver la barre de progression (plus rapide)
        Invoke-WebRequest -Uri $image.Url -OutFile $destination -UseBasicParsing -TimeoutSec 30
        
        # Vérifier que le fichier existe et n'est pas vide
        if (Test-Path $destination) {
            $fileSize = (Get-Item $destination).Length
            if ($fileSize -gt 0) {
                $fileSizeKB = [math]::Round($fileSize / 1KB, 2)
                Write-Host "   ✅ Téléchargé avec succès ($fileSizeKB KB)" -ForegroundColor Green
                $success++
            } else {
                Write-Host "   ❌ Fichier vide" -ForegroundColor Red
                $failed++
            }
        } else {
            Write-Host "   ❌ Fichier non créé" -ForegroundColor Red
            $failed++
        }
    }
    catch {
        Write-Host "   ❌ Erreur: $($_.Exception.Message)" -ForegroundColor Red
        $failed++
    }
    
    Write-Host ""
}

Write-Host "========================================" -ForegroundColor Cyan
Write-Host "  RÉSUMÉ" -ForegroundColor Cyan
Write-Host "========================================" -ForegroundColor Cyan
Write-Host "✅ Réussis: $success" -ForegroundColor Green
Write-Host "❌ Échoués: $failed" -ForegroundColor Red
Write-Host ""

if ($success -gt 0) {
    Write-Host "📂 Les images sont dans:" -ForegroundColor Cyan
    Write-Host "   $downloadFolder" -ForegroundColor White
    Write-Host ""
    Write-Host "========================================" -ForegroundColor Cyan
    Write-Host "  PROCHAINES ÉTAPES" -ForegroundColor Cyan
    Write-Host "========================================" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "1. 📤 Uploader ces images sur votre serveur:" -ForegroundColor Yellow
    Write-Host "   https://ujsa.studioplc.fr/datas/images/" -ForegroundColor White
    Write-Host ""
    Write-Host "2. 📝 Modifier le JSON du jeu:" -ForegroundColor Yellow
    Write-Host '   "assets": {' -ForegroundColor White
    Write-Host '     "gun": "https://ujsa.studioplc.fr/datas/images/vide.png",' -ForegroundColor White
    Write-Host '     "crosshair": "https://ujsa.studioplc.fr/datas/images/curseur-arrow-taxi-jaune.png",' -ForegroundColor Green
    Write-Host '     "impact": "https://ujsa.studioplc.fr/datas/images/impact-flash-light.png"' -ForegroundColor Green
    Write-Host '   }' -ForegroundColor White
    Write-Host ""
    Write-Host "3. ✅ Tester le jeu WebGL - le crosshair devrait apparaître !" -ForegroundColor Yellow
    Write-Host ""
    
    # Ouvrir l'explorateur de fichiers
    Write-Host "📁 Ouverture du dossier..." -ForegroundColor Cyan
    Start-Process explorer.exe $downloadFolder
}

if ($failed -gt 0) {
    Write-Host ""
    Write-Host "⚠️  Certains téléchargements ont échoué." -ForegroundColor Yellow
    Write-Host "   Causes possibles:" -ForegroundColor Gray
    Write-Host "   - Pas de connexion Internet" -ForegroundColor Gray
    Write-Host "   - URLs invalides ou fichiers supprimés" -ForegroundColor Gray
    Write-Host "   - Firewall bloquant" -ForegroundColor Gray
    Write-Host ""
    Write-Host "   Solution: Téléchargez manuellement depuis le navigateur" -ForegroundColor Gray
}

Write-Host ""
Write-Host "Appuyez sur une touche pour fermer..." -ForegroundColor Gray
$null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
