using UnityEngine;
using UnityEngine.UI;

#if UNITY_EDITOR
using UnityEditor;
using UnityEditor.SceneManagement;
#endif

public class MainSceneGenerator : ScriptableObject
{
    [ContextMenu("Generate Main Scene")]
    public void GenerateMainScene()
    {
        Debug.Log("[MainSceneGenerator] Génération de la scène Main...");
        
        try
        {
            #if UNITY_EDITOR
            // Créer ou charger la scène main
            UnityEngine.SceneManagement.Scene scene = EditorSceneManager.GetSceneByName("main");
            
            if (!scene.IsValid())
            {
                // Créer une nouvelle scène
                scene = EditorSceneManager.NewScene(NewSceneSetup.DefaultGameObjects, NewSceneMode.Single);
                Debug.Log("[MainSceneGenerator] Nouvelle scène créée");
            }
            else
            {
                // Charger la scène existante
                scene = EditorSceneManager.OpenScene("Assets/Scenes/main.unity", OpenSceneMode.Single);
                Debug.Log("[MainSceneGenerator] Scène existante chargée");
            }
            
            // Nettoyer les objets existants
            ClearExistingObjects();
            
            // Créer le Canvas principal
            Debug.Log("[MainSceneGenerator] Création du Canvas...");
            GameObject canvas = CreateMainCanvas();
            if (canvas == null)
            {
                Debug.LogError("[MainSceneGenerator] Échec de création du Canvas");
                return;
            }
            
            // Créer le MainSceneManager
            Debug.Log("[MainSceneGenerator] Création du MainSceneManager...");
            GameObject managerObj = CreateMainSceneManager();
            if (managerObj == null)
            {
                Debug.LogError("[MainSceneGenerator] Échec de création du MainSceneManager");
                return;
            }
            
            // Créer l'EventSystem (nécessaire pour les clics)
            Debug.Log("[MainSceneGenerator] Création de l'EventSystem...");
            CreateEventSystem();
            
            // Enregistrer tous les objets dans la scène
            Debug.Log("[MainSceneGenerator] Enregistrement des objets dans la scène...");
            
            GameObject[] allObjects = { canvas, managerObj };
            
            foreach (GameObject obj in allObjects)
            {
                if (obj != null)
                {
                    Undo.RegisterCreatedObjectUndo(obj, "Create Main Scene Objects");
                    EditorUtility.SetDirty(obj);
                }
            }
            
            // Marquer la scène comme modifiée
            EditorSceneManager.MarkSceneDirty(scene);
            
            Debug.Log("[MainSceneGenerator] ✅ Scène Main générée avec succès !");
            Debug.Log("⚠️ IMPORTANT : Sauvegardez la scène avec Ctrl+S pour que les changements persistent !");
            #else
            Debug.LogError("[MainSceneGenerator] Ce script ne peut être exécuté que dans l'éditeur Unity");
            #endif
        }
        catch (System.Exception e)
        {
            Debug.LogError($"[MainSceneGenerator] Erreur détaillée: {e.Message}\nStackTrace: {e.StackTrace}");
        }
    }
    
    private void ClearExistingObjects()
    {
        #if UNITY_EDITOR
        // Supprimer les objets existants avec les noms spécifiques
        string[] objectNames = {
            "Canvas", "MainSceneManager", "EventSystem"
        };
        
        foreach (string objectName in objectNames)
        {
            GameObject existing = GameObject.Find(objectName);
            if (existing != null)
            {
                DestroyImmediate(existing);
            }
        }
        #endif
    }
    
    private GameObject CreateMainCanvas()
    {
        GameObject canvas = new GameObject("Canvas");
        
        // Canvas Component
        Canvas canvasComponent = canvas.AddComponent<Canvas>();
        canvasComponent.renderMode = RenderMode.ScreenSpaceOverlay;
        
        // CanvasScaler
        CanvasScaler scaler = canvas.AddComponent<CanvasScaler>();
        scaler.uiScaleMode = CanvasScaler.ScaleMode.ScaleWithScreenSize;
        scaler.referenceResolution = new Vector2(1920, 1080);
        scaler.matchWidthOrHeight = 0.5f;
        
        // GraphicRaycaster
        canvas.AddComponent<GraphicRaycaster>();
        
        // RectTransform
        RectTransform rect = canvas.GetComponent<RectTransform>();
        rect.anchorMin = Vector2.zero;
        rect.anchorMax = Vector2.one;
        rect.offsetMin = Vector2.zero;
        rect.offsetMax = Vector2.zero;
        
        return canvas;
    }
    
    private GameObject CreateMainSceneManager()
    {
        GameObject managerObj = new GameObject("MainSceneManager");
        
        // Ajouter le composant MainSceneManager
        MainSceneManager manager = managerObj.AddComponent<MainSceneManager>();
        
        // La configuration sera chargée depuis le JSON au runtime
        // Pas besoin de configurer ici
        
        return managerObj;
    }
    
    private void CreateEventSystem()
    {
        // Vérifier si un EventSystem existe déjà
        UnityEngine.EventSystems.EventSystem existing = Object.FindFirstObjectByType<UnityEngine.EventSystems.EventSystem>();
        if (existing != null)
        {
            Debug.Log("[MainSceneGenerator] EventSystem existe déjà");
            return;
        }
        
        GameObject eventSystemObj = new GameObject("EventSystem");
        eventSystemObj.AddComponent<UnityEngine.EventSystems.EventSystem>();
        
        #if ENABLE_INPUT_SYSTEM
        // Utiliser InputSystemUIInputModule si le nouveau Input System est activé
        eventSystemObj.AddComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
        Debug.Log("[MainSceneGenerator] EventSystem créé avec InputSystemUIInputModule");
        #else
        // Utiliser StandaloneInputModule si l'ancien Input System est utilisé
        eventSystemObj.AddComponent<UnityEngine.EventSystems.StandaloneInputModule>();
        Debug.Log("[MainSceneGenerator] EventSystem créé avec StandaloneInputModule");
        #endif
    }
    
    #if UNITY_EDITOR
    [MenuItem("Tools/Main Scene/Generate Scene")]
    public static void GenerateSceneFromMenu()
    {
        try
        {
            // Créer une instance temporaire sans GameObject
            MainSceneGenerator generator = ScriptableObject.CreateInstance<MainSceneGenerator>();
            
            Debug.Log("[MainSceneGenerator] Générateur créé, début de la génération...");
            
            generator.GenerateMainScene();
            
            Debug.Log("[MainSceneGenerator] Scène générée avec succès depuis le menu !");
            
            // Nettoyer l'instance temporaire
            ScriptableObject.DestroyImmediate(generator);
        }
        catch (System.Exception e)
        {
            Debug.LogError($"[MainSceneGenerator] Erreur: {e.Message}\nStackTrace: {e.StackTrace}");
        }
    }
    #endif
}

