using UnityEngine;

/// <summary>
/// Script de nettoyage global pour supprimer tous les StandaloneInputModule
/// qui causent des exceptions avec le nouveau Input System.
/// S'exécute AVANT le chargement de la scène pour éviter les exceptions.
/// </summary>
public class InputSystemCleanup : MonoBehaviour
{
    [RuntimeInitializeOnLoadMethod(RuntimeInitializeLoadType.BeforeSceneLoad)]
    static void CleanupBeforeSceneLoad()
    {
        #if ENABLE_INPUT_SYSTEM
        // Supprimer tous les StandaloneInputModule AVANT même que la scène ne soit chargée
        // Cela évite les exceptions InvalidOperationException
        UnityEngine.EventSystems.StandaloneInputModule[] allStandaloneModules = 
            Object.FindObjectsByType<UnityEngine.EventSystems.StandaloneInputModule>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        
        foreach (var module in allStandaloneModules)
        {
            if (module != null)
            {
                // Désactiver d'abord
                module.enabled = false;
                
                // Désactiver aussi l'EventSystem parent s'il existe
                var eventSystem = module.GetComponent<UnityEngine.EventSystems.EventSystem>();
                if (eventSystem != null)
                {
                    eventSystem.enabled = false;
                }
                
                // Supprimer complètement
                Object.DestroyImmediate(module);
                Debug.Log("[InputSystemCleanup] ⚠️ StandaloneInputModule supprimé (BeforeSceneLoad)");
                #if UNITY_WEBGL && !UNITY_EDITOR
                UnityEngine.Application.ExternalCall("console.log", "[Unity] StandaloneInputModule supprimé (BeforeSceneLoad)");
                #endif
            }
        }
        
        // Désactiver tous les EventSystem qui ont encore des StandaloneInputModule
        UnityEngine.EventSystems.EventSystem[] allEventSystems = 
            Object.FindObjectsByType<UnityEngine.EventSystems.EventSystem>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        
        foreach (var es in allEventSystems)
        {
            if (es != null)
            {
                var standaloneModule = es.GetComponent<UnityEngine.EventSystems.StandaloneInputModule>();
                if (standaloneModule != null)
                {
                    es.enabled = false;
                    standaloneModule.enabled = false;
                    Object.DestroyImmediate(standaloneModule);
                    Debug.Log($"[InputSystemCleanup] ⚠️ EventSystem '{es.gameObject.name}' DÉSACTIVÉ et StandaloneInputModule supprimé");
                }
            }
        }
        
        // Créer une instance persistante pour le nettoyage continu
        GameObject cleanupObj = new GameObject("InputSystemCleanup");
        cleanupObj.AddComponent<InputSystemCleanup>();
        Object.DontDestroyOnLoad(cleanupObj);
        #endif
    }
    
    void Awake()
    {
        #if ENABLE_INPUT_SYSTEM
        // Nettoyage supplémentaire au démarrage de chaque scène
        CleanupStandaloneInputModules();
        #endif
    }
    
    void Update()
    {
        #if ENABLE_INPUT_SYSTEM
        // Nettoyage continu (toutes les 60 frames pour éviter le spam)
        if (Time.frameCount % 60 == 0)
        {
            CleanupStandaloneInputModules();
        }
        #endif
    }
    
    void CleanupStandaloneInputModules()
    {
        #if ENABLE_INPUT_SYSTEM
        // Supprimer TOUS les StandaloneInputModule de la scène
        UnityEngine.EventSystems.StandaloneInputModule[] allStandaloneModules = 
            FindObjectsByType<UnityEngine.EventSystems.StandaloneInputModule>(FindObjectsSortMode.None);
        
        foreach (var module in allStandaloneModules)
        {
            if (module != null)
            {
                // Désactiver d'abord
                module.enabled = false;
                
                // Désactiver aussi l'EventSystem parent s'il existe
                var eventSystem = module.GetComponent<UnityEngine.EventSystems.EventSystem>();
                if (eventSystem != null)
                {
                    eventSystem.enabled = false;
                }
                
                // Supprimer complètement
                DestroyImmediate(module);
                Debug.Log("[InputSystemCleanup] ⚠️ StandaloneInputModule supprimé (CleanupStandaloneInputModules)");
                #if UNITY_WEBGL && !UNITY_EDITOR
                UnityEngine.Application.ExternalCall("console.log", "[Unity] StandaloneInputModule supprimé (Cleanup)");
                #endif
            }
        }
        
        // Désactiver TOUS les EventSystem qui ont encore des StandaloneInputModule
        UnityEngine.EventSystems.EventSystem[] allEventSystems = 
            FindObjectsByType<UnityEngine.EventSystems.EventSystem>(FindObjectsSortMode.None);
        
        foreach (var es in allEventSystems)
        {
            if (es != null)
            {
                // Supprimer tout StandaloneInputModule restant
                var standaloneModule = es.GetComponent<UnityEngine.EventSystems.StandaloneInputModule>();
                if (standaloneModule != null)
                {
                    // Désactiver l'EventSystem complètement
                    es.enabled = false;
                    standaloneModule.enabled = false;
                    DestroyImmediate(standaloneModule);
                    Debug.Log($"[InputSystemCleanup] ⚠️ EventSystem '{es.gameObject.name}' DÉSACTIVÉ et StandaloneInputModule supprimé");
                    #if UNITY_WEBGL && !UNITY_EDITOR
                    UnityEngine.Application.ExternalCall("console.log", $"[Unity] EventSystem '{es.gameObject.name}' DÉSACTIVÉ");
                    #endif
                }
                else
                {
                    // Si pas de StandaloneInputModule, vérifier s'il y a un InputSystemUIInputModule
                    var inputSystemModule = es.GetComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
                    if (inputSystemModule == null)
                    {
                        // Ajouter InputSystemUIInputModule si absent
                        es.gameObject.AddComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
                        Debug.Log("[InputSystemCleanup] ✅ InputSystemUIInputModule ajouté");
                    }
                    else if (!inputSystemModule.enabled)
                    {
                        // Réactiver s'il est désactivé
                        inputSystemModule.enabled = true;
                        Debug.Log("[InputSystemCleanup] ✅ InputSystemUIInputModule réactivé");
                    }
                }
            }
        }
        #endif
    }
}

