using System;
using System.Collections;
using UnityEngine;
using UnityEngine.Networking;

/// <summary>
/// Helper pour charger des images depuis des URLs avec support Mac amélioré
/// Remplace UnityWebRequestTexture.GetTexture() avec des corrections pour Mac
/// </summary>
public static class MacImageLoader
{
    // Paramètres configurables
    public static int TimeoutSeconds = 30; // Augmenté pour Mac
    public static bool LogVerbose = true;
    public static bool AcceptAllCertificates = true; // Accepter les certificats auto-signés
    
    /// <summary>
    /// Charge une texture depuis une URL avec support Mac amélioré
    /// </summary>
    public static IEnumerator LoadTexture(string url, Action<Texture2D> onSuccess, Action<string> onError = null)
    {
        if (string.IsNullOrEmpty(url))
        {
            string error = "URL vide";
            Debug.LogError($"[MacImageLoader] {error}");
            onError?.Invoke(error);
            yield break;
        }
        
        if (LogVerbose)
        {
            Debug.Log($"[MacImageLoader] 📥 Chargement depuis: {url}");
            Debug.Log($"[MacImageLoader] Platform: {Application.platform}");
        }
        
        // Sur Mac, accepter tous les certificats SSL
        #if UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX
        if (AcceptAllCertificates)
        {
            // Désactiver la vérification SSL (uniquement pour dev)
            ServicePointManagerHelper.SetupMacSSL();
        }
        #endif
        
        using (UnityWebRequest www = UnityWebRequestTexture.GetTexture(url))
        {
            // Configurer le timeout
            www.timeout = TimeoutSeconds;
            
            // Headers pour Mac
            www.SetRequestHeader("User-Agent", "Unity/" + Application.unityVersion);
            www.SetRequestHeader("Cache-Control", "no-cache");
            
            #if UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX
            // Headers spécifiques Mac
            www.SetRequestHeader("Accept", "image/*");
            if (LogVerbose)
            {
                Debug.Log("[MacImageLoader] 🍎 Headers Mac ajoutés");
            }
            #endif
            
            // Lancer la requête
            yield return www.SendWebRequest();
            
            // Vérifier le résultat
            #if UNITY_2020_2_OR_NEWER
            bool success = www.result == UnityWebRequest.Result.Success;
            #else
            bool success = !(www.isNetworkError || www.isHttpError);
            #endif
            
            if (success)
            {
                try
                {
                    Texture2D texture = ((DownloadHandlerTexture)www.downloadHandler).texture;
                    
                    if (texture == null)
                    {
                        string error = "Texture nulle après téléchargement";
                        Debug.LogError($"[MacImageLoader] ❌ {error}");
                        Debug.LogError($"[MacImageLoader] URL: {url}");
                        onError?.Invoke(error);
                    }
                    else
                    {
                        if (LogVerbose)
                        {
                            Debug.Log($"[MacImageLoader] ✅ Texture chargée: {texture.width}x{texture.height}");
                        }
                        onSuccess?.Invoke(texture);
                    }
                }
                catch (Exception e)
                {
                    string error = $"Exception lors de la récupération de la texture: {e.Message}";
                    Debug.LogError($"[MacImageLoader] ❌ {error}");
                    Debug.LogError($"[MacImageLoader] Stack trace: {e.StackTrace}");
                    onError?.Invoke(error);
                }
            }
            else
            {
                string error = $"Erreur réseau: {www.error}";
                Debug.LogError($"[MacImageLoader] ❌ {error}");
                Debug.LogError($"[MacImageLoader] URL: {url}");
                Debug.LogError($"[MacImageLoader] Code HTTP: {www.responseCode}");
                Debug.LogError($"[MacImageLoader] Result: {www.result}");
                
                #if UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX
                Debug.LogError("[MacImageLoader] 🍎 PROBLÈME MAC DÉTECTÉ");
                Debug.LogError("[MacImageLoader] Causes possibles:");
                Debug.LogError("  - Certificat SSL non accepté");
                Debug.LogError("  - CORS bloqué");
                Debug.LogError("  - Timeout (augmentez TimeoutSeconds)");
                Debug.LogError("  - URL non accessible depuis Mac");
                #endif
                
                onError?.Invoke(error);
            }
        }
    }
    
    /// <summary>
    /// Charge une texture et crée un Sprite
    /// </summary>
    public static IEnumerator LoadSprite(string url, Action<Sprite> onSuccess, Action<string> onError = null)
    {
        yield return LoadTexture(url, 
            (texture) => {
                try
                {
                    Sprite sprite = Sprite.Create(
                        texture,
                        new Rect(0, 0, texture.width, texture.height),
                        new Vector2(0.5f, 0.5f),
                        100f
                    );
                    onSuccess?.Invoke(sprite);
                }
                catch (Exception e)
                {
                    string error = $"Erreur création sprite: {e.Message}";
                    Debug.LogError($"[MacImageLoader] ❌ {error}");
                    onError?.Invoke(error);
                }
            },
            onError
        );
    }
}

/// <summary>
/// Helper pour configurer SSL sur Mac
/// </summary>
public static class ServicePointManagerHelper
{
    private static bool _isConfigured = false;
    
    public static void SetupMacSSL()
    {
        if (_isConfigured) return;
        
        #if UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX
        try
        {
            // Accepter tous les certificats SSL (UNIQUEMENT POUR DEV!)
            System.Net.ServicePointManager.ServerCertificateValidationCallback = 
                (sender, certificate, chain, sslPolicyErrors) => true;
            
            // Forcer TLS 1.2
            System.Net.ServicePointManager.SecurityProtocol = 
                System.Net.SecurityProtocolType.Tls12 | 
                System.Net.SecurityProtocolType.Tls11 | 
                System.Net.SecurityProtocolType.Tls;
            
            Debug.Log("[MacImageLoader] 🍎 SSL configuré pour Mac (accepte tous les certificats)");
            _isConfigured = true;
        }
        catch (Exception e)
        {
            Debug.LogWarning($"[MacImageLoader] ⚠️ Impossible de configurer SSL: {e.Message}");
        }
        #endif
    }
}

