using System;
using System.Collections.Generic;
using UnityEngine;

/// <summary>
/// Gère le stockage et la récupération des résultats des jeux
/// </summary>
public class GameResultsManager : MonoBehaviour
{
    private static GameResultsManager instance;
    public static GameResultsManager Instance => instance;

    private const string PLAYER_PREFS_KEY = "GameResults";
    
    [System.Serializable]
    public class GameResult
    {
        public string gameId;           // Identifiant unique du jeu (configUrl ou levelId)
        public string gameName;         // Nom du jeu (optionnel)
        public bool isSuccess;          // true = succès, false = échec
        public int correctAnswers;      // Nombre de bonnes réponses
        public int totalAnswers;        // Nombre total de réponses
        public string dateTime;         // Date et heure du résultat
        public string gameType;         // "shooting", "trous", "calculator", etc.
    }
    
    [System.Serializable]
    public class GameResultsData
    {
        public List<GameResult> results = new List<GameResult>();
    }
    
    private GameResultsData resultsData;
    
    void Awake()
    {
        if (instance == null)
        {
            instance = this;
            DontDestroyOnLoad(gameObject);
            LoadResults();
        }
        else if (instance != this)
        {
            Destroy(gameObject);
        }
    }
    
    /// <summary>
    /// Charge les résultats depuis PlayerPrefs
    /// </summary>
    private void LoadResults()
    {
        string json = PlayerPrefs.GetString(PLAYER_PREFS_KEY, "");
        if (string.IsNullOrEmpty(json))
        {
            resultsData = new GameResultsData();
            Debug.Log("[GameResultsManager] Aucun résultat précédent trouvé");
        }
        else
        {
            try
            {
                resultsData = JsonUtility.FromJson<GameResultsData>(json);
                if (resultsData == null)
                {
                    resultsData = new GameResultsData();
                }
                Debug.Log($"[GameResultsManager] {resultsData.results.Count} résultats chargés");
            }
            catch (Exception e)
            {
                Debug.LogError($"[GameResultsManager] Erreur lors du chargement des résultats: {e.Message}");
                resultsData = new GameResultsData();
            }
        }
    }
    
    /// <summary>
    /// Sauvegarde les résultats dans PlayerPrefs
    /// </summary>
    private void SaveResults()
    {
        try
        {
            string json = JsonUtility.ToJson(resultsData, true);
            PlayerPrefs.SetString(PLAYER_PREFS_KEY, json);
            PlayerPrefs.Save();
            Debug.Log($"[GameResultsManager] {resultsData.results.Count} résultats sauvegardés");
        }
        catch (Exception e)
        {
            Debug.LogError($"[GameResultsManager] Erreur lors de la sauvegarde des résultats: {e.Message}");
        }
    }
    
    /// <summary>
    /// Enregistre un résultat de jeu
    /// </summary>
    public void RecordGameResult(string gameId, string gameName, bool isSuccess, int correctAnswers, int totalAnswers, string gameType = "shooting")
    {
        if (resultsData == null)
        {
            resultsData = new GameResultsData();
        }
        
        GameResult result = new GameResult
        {
            gameId = gameId,
            gameName = gameName,
            isSuccess = isSuccess,
            correctAnswers = correctAnswers,
            totalAnswers = totalAnswers,
            dateTime = DateTime.Now.ToString("yyyy-MM-dd HH:mm:ss"),
            gameType = gameType
        };
        
        resultsData.results.Add(result);
        SaveResults();
        
        Debug.Log($"[GameResultsManager] Résultat enregistré: {gameName} ({gameId}) - {(isSuccess ? "SUCCÈS" : "ÉCHEC")} - {correctAnswers}/{totalAnswers}");
    }
    
    /// <summary>
    /// Récupère tous les résultats
    /// </summary>
    public List<GameResult> GetAllResults()
    {
        if (resultsData == null)
        {
            LoadResults();
        }
        return resultsData?.results ?? new List<GameResult>();
    }
    
    /// <summary>
    /// Récupère les résultats d'un jeu spécifique
    /// </summary>
    public List<GameResult> GetGameResults(string gameId)
    {
        if (resultsData == null)
        {
            LoadResults();
        }
        
        List<GameResult> gameResults = new List<GameResult>();
        if (resultsData != null)
        {
            foreach (var result in resultsData.results)
            {
                if (result.gameId == gameId)
                {
                    gameResults.Add(result);
                }
            }
        }
        return gameResults;
    }
    
    /// <summary>
    /// Récupère le dernier résultat d'un jeu
    /// </summary>
    public GameResult GetLastGameResult(string gameId)
    {
        List<GameResult> gameResults = GetGameResults(gameId);
        if (gameResults.Count > 0)
        {
            // Retourner le plus récent (le dernier dans la liste)
            return gameResults[gameResults.Count - 1];
        }
        return null;
    }
    
    /// <summary>
    /// Efface tous les résultats
    /// </summary>
    public void ClearAllResults()
    {
        resultsData = new GameResultsData();
        PlayerPrefs.DeleteKey(PLAYER_PREFS_KEY);
        PlayerPrefs.Save();
        Debug.Log("[GameResultsManager] Tous les résultats ont été effacés");
    }
    
    /// <summary>
    /// Compte le nombre de succès et d'échecs
    /// </summary>
    public void GetStatistics(out int totalGames, out int successes, out int failures)
    {
        if (resultsData == null)
        {
            LoadResults();
        }
        
        totalGames = resultsData?.results.Count ?? 0;
        successes = 0;
        failures = 0;
        
        if (resultsData != null)
        {
            foreach (var result in resultsData.results)
            {
                if (result.isSuccess)
                    successes++;
                else
                    failures++;
            }
        }
    }
}

