using System.Collections;
using UnityEngine;
using UnityEngine.Networking;

/// <summary>
/// Manager pour charger et gérer les paramètres généraux du projet
/// </summary>
public class GeneralConfigManager : MonoBehaviour
{
    public static GeneralConfigManager Instance { get; private set; }
    
    [Header("Configuration")]
    public string configFileName = "general-config.json";
    
    [Header("Debug")]
    public bool showDebugLogs = false;

    // URL de fallback hardcodée pour WebGL si le fichier local n'est pas trouvé
    private const string FALLBACK_REMOTE_CONFIG_URL = "https://ujsa.studioplc.fr/datas/json/general-config.json";
    
    private GeneralConfig generalConfig;
    private bool isConfigLoaded = false;
    private bool isLoadingInProgress = false; // 🔒 NOUVEAU : Empêche les chargements multiples
    
    void Awake()
    {
        // Singleton pattern
        if (Instance == null)
        {
            Instance = this;
            DontDestroyOnLoad(gameObject);
        }
        else
        {
            Destroy(gameObject);
            return;
        }
    }
    
    void Start()
    {
        StartCoroutine(LoadGeneralConfig());
    }
    
    /// <summary>
    /// Charge la configuration générale depuis le fichier JSON
    /// </summary>
    private IEnumerator LoadGeneralConfig()
    {
        // 🔒 EMPÊCHER LES CHARGEMENTS MULTIPLES
        if (isLoadingInProgress)
        {
            if (showDebugLogs)
            {
                Debug.LogWarning("[GeneralConfigManager] Chargement déjà en cours - Tentative ignorée");
            }
            yield break;
        }
        
        if (isConfigLoaded)
        {
            if (showDebugLogs)
            {
                Debug.Log("[GeneralConfigManager] Configuration déjà chargée - Pas besoin de recharger");
            }
            yield break;
        }
        
        isLoadingInProgress = true;
        
        string filePath = System.IO.Path.Combine(Application.streamingAssetsPath, configFileName);
        
        if (showDebugLogs)
        {
            Debug.Log($"[GeneralConfigManager] Chargement de la configuration depuis: {filePath}");
            Debug.Log($"[GeneralConfigManager] Platform: {Application.platform}");
        }
        
        // DÉTECTION MAC AU RUNTIME (plus fiable que les defines de compilation)
        bool isMacOrIOS = Application.platform == RuntimePlatform.OSXEditor || 
                         Application.platform == RuntimePlatform.OSXPlayer ||
                         Application.platform == RuntimePlatform.IPhonePlayer;
        
        // FIX MAC : Utiliser File.ReadAllText sur Mac/iOS au lieu de UnityWebRequest
        // IMPORTANT : Aussi dans l'éditeur Mac car UnityWebRequest échoue avec file:///
        if (isMacOrIOS)
        {
            Debug.Log("[GeneralConfigManager] 🍎 Détection Mac/iOS - Utilisation de File.ReadAllText");
            
            if (System.IO.File.Exists(filePath))
            {
                try
                {
                    string jsonContent = System.IO.File.ReadAllText(filePath);
                    
                    if (showDebugLogs)
                    {
                        Debug.Log($"[GeneralConfigManager] Fichier lu: {jsonContent.Length} caractères");
                    }
                    
                    generalConfig = JsonUtility.FromJson<GeneralConfig>(jsonContent);
                    isConfigLoaded = true;
                    
                    // Debug des buttonStyles (toujours afficher pour le debug)
                    Debug.Log($"[GeneralConfigManager] 🎨 (Mac/iOS) buttonStyles: {(generalConfig.buttonStyles != null ? "OK" : "NULL")}");
                    if (generalConfig.buttonStyles != null)
                    {
                        Debug.Log($"[GeneralConfigManager] 🎨 validationPurple: {(generalConfig.buttonStyles.validationPurple != null ? $"OK (w={generalConfig.buttonStyles.validationPurple.width})" : "NULL")}");
                        Debug.Log($"[GeneralConfigManager] 🎨 validationBeige: {(generalConfig.buttonStyles.validationBeige != null ? $"OK (w={generalConfig.buttonStyles.validationBeige.width})" : "NULL")}");
                    }
                    Debug.Log($"[GeneralConfigManager] 🔘 connexionPanel: {(generalConfig.connexionPanel != null ? "OK" : "NULL")}");
                    if (generalConfig.connexionPanel != null)
                    {
                        Debug.Log($"[GeneralConfigManager] 🔘 connexionPanel.buttons: {(generalConfig.connexionPanel.buttons != null ? "OK" : "NULL")}");
                        if (generalConfig.connexionPanel.buttons != null)
                        {
                            Debug.Log($"[GeneralConfigManager] 🔘 continueButton.style: '{generalConfig.connexionPanel.buttons.continueButton?.style ?? "NULL"}'");
                            Debug.Log($"[GeneralConfigManager] 🔘 seroniConnectButton.style: '{generalConfig.connexionPanel.buttons.seroniConnectButton?.style ?? "NULL"}'");
                        }
                    }
                    
                    if (showDebugLogs)
                    {
                        Debug.Log($"[GeneralConfigManager] Configuration chargée avec succès (Mac/iOS)!");
                        Debug.Log($"[GeneralConfigManager] Projet: {generalConfig.projectName} v{generalConfig.version}");
                        Debug.Log($"[GeneralConfigManager] Map par défaut: {generalConfig.defaultMapId}");
                    }
                    
                    isLoadingInProgress = false; // 🔓 DÉVERROUILLER - Succès Mac/iOS
                    yield break; // Important : sortir de la coroutine
                }
                    catch (System.Exception e)
                    {
                        isLoadingInProgress = false; // 🔓 DÉVERROUILLER - Erreur Mac/iOS
                        Debug.LogError($"[GeneralConfigManager] Erreur lors du parsing JSON (Mac/iOS): {e.Message}");
                        LoadDefaultConfig();
                        yield break;
                    }
                }
                else
                {
                    isLoadingInProgress = false; // 🔓 DÉVERROUILLER - Fichier introuvable Mac/iOS
                    Debug.LogError($"[GeneralConfigManager] Fichier introuvable sur Mac/iOS: {filePath}");
                    Debug.LogError($"[GeneralConfigManager] Assurez-vous que {configFileName} est dans le dossier StreamingAssets");
                    LoadDefaultConfig();
                    yield break;
                }
        }
        
        // Code normal pour Windows/Android/autres plateformes (ou éditeur Mac)
        if (showDebugLogs)
        {
            Debug.Log("[GeneralConfigManager] Utilisation de UnityWebRequest");
        }
        
        UnityWebRequest request = UnityWebRequest.Get(filePath);
        request.timeout = 10; // Timeout de 10 secondes
        
        Debug.Log($"[GeneralConfigManager] 🌐 Envoi requête vers: {filePath}");
        
        yield return request.SendWebRequest();
        
        Debug.Log($"[GeneralConfigManager] 🌐 Result: {request.result} | Code: {request.responseCode} | Error: {request.error}");
        
        if (request.result == UnityWebRequest.Result.Success)
        {
            Debug.Log($"[GeneralConfigManager] 🎯 Texte reçu: {request.downloadHandler.text.Length} caractères");
            Debug.Log($"[GeneralConfigManager] 🎯 Premiers 100 chars: {request.downloadHandler.text.Substring(0, Mathf.Min(100, request.downloadHandler.text.Length))}");
            
            try
            {
                Debug.Log($"[GeneralConfigManager] 🎯 Début parsing...");
                generalConfig = JsonUtility.FromJson<GeneralConfig>(request.downloadHandler.text);
                Debug.Log($"[GeneralConfigManager] 🎯 Parsing terminé. Config null? {generalConfig == null}");
                isConfigLoaded = true;
                isLoadingInProgress = false; // 🔓 DÉVERROUILLER
                
                Debug.Log($"[GeneralConfigManager] ✅ Configuration chargée avec SUCCÈS!");
                
                // Debug des buttonStyles
                Debug.Log($"[GeneralConfigManager] 🎨 buttonStyles: {(generalConfig.buttonStyles != null ? "OK" : "NULL")}");
                if (generalConfig.buttonStyles != null)
                {
                    Debug.Log($"[GeneralConfigManager] 🎨 validationPurple: {(generalConfig.buttonStyles.validationPurple != null ? "OK" : "NULL")}");
                    Debug.Log($"[GeneralConfigManager] 🎨 validationBeige: {(generalConfig.buttonStyles.validationBeige != null ? "OK" : "NULL")}");
                }
                Debug.Log($"[GeneralConfigManager] 🔘 connexionPanel.buttons: {(generalConfig.connexionPanel?.buttons != null ? "OK" : "NULL")}");
                
                // Debug de apiUrls
                Debug.Log($"[GeneralConfigManager] 🌐 apiUrls: {(generalConfig.apiUrls != null ? "OK" : "NULL")}");
                if (generalConfig.apiUrls != null)
                {
                    Debug.Log($"[GeneralConfigManager] 🌐 apiUrls.baseUrl: '{(string.IsNullOrEmpty(generalConfig.apiUrls.baseUrl) ? "VIDE" : generalConfig.apiUrls.baseUrl)}'");
                }
                
                if (showDebugLogs)
                {
                    Debug.Log($"[GeneralConfigManager] Projet: {generalConfig.projectName} v{generalConfig.version}");
                    Debug.Log($"[GeneralConfigManager] Map par défaut: {generalConfig.defaultMapId}");
                }
            }
            catch (System.Exception e)
            {
                isLoadingInProgress = false; // 🔓 DÉVERROUILLER même en cas d'erreur
                Debug.LogError($"[GeneralConfigManager] ❌ Erreur parsing: {e.Message}");
                Debug.LogError($"[GeneralConfigManager] ❌ Stack: {e.StackTrace}");
                LoadDefaultConfig();
            }
        }
        else
        {
            // NE PAS DÉVERROUILLER ICI - On va essayer le fallback
            Debug.LogError($"[GeneralConfigManager] ❌ Échec local: {request.error} | Code: {request.responseCode}");
            Debug.Log($"[GeneralConfigManager] 🔄 Tentative fallback URL: {FALLBACK_REMOTE_CONFIG_URL}");
            
            // Tentative de chargement depuis l'URL distante
            using (UnityWebRequest remoteRequest = UnityWebRequest.Get(FALLBACK_REMOTE_CONFIG_URL))
            {
                remoteRequest.timeout = 10;
                yield return remoteRequest.SendWebRequest();

                if (remoteRequest.result == UnityWebRequest.Result.Success)
                {
                    try
                    {
                        generalConfig = JsonUtility.FromJson<GeneralConfig>(remoteRequest.downloadHandler.text);
                        isConfigLoaded = true;
                        isLoadingInProgress = false; // 🔓 DÉVERROUILLER - Succès du fallback
                        Debug.Log($"[GeneralConfigManager] ✅ Configuration chargée depuis URL de secours: {FALLBACK_REMOTE_CONFIG_URL}");
                    }
                    catch (System.Exception e)
                    {
                        isLoadingInProgress = false; // 🔓 DÉVERROUILLER - Échec du parsing
                        Debug.LogError($"[GeneralConfigManager] Erreur parsing config distante: {e.Message}");
                        LoadDefaultConfig();
                    }
                }
                else
                {
                    isLoadingInProgress = false; // 🔓 DÉVERROUILLER - Échec du fallback
                    Debug.LogError($"[GeneralConfigManager] ❌ Échec chargement distant: {remoteRequest.error}");
                    LoadDefaultConfig();
                }
            }
        }
        
        request.Dispose();
    }
    
    /// <summary>
    /// Charge une configuration par défaut en cas d'erreur
    /// Essaie de charger le JSON de manière synchrone avant de créer une config vide
    /// </summary>
    private void LoadDefaultConfig()
    {
        // Dernier essai : charger le JSON de manière synchrone
        string filePath = System.IO.Path.Combine(Application.streamingAssetsPath, configFileName);
        
        if (System.IO.File.Exists(filePath))
        {
            try
            {
                string jsonContent = System.IO.File.ReadAllText(filePath);
                generalConfig = JsonUtility.FromJson<GeneralConfig>(jsonContent);
                isConfigLoaded = true;
                Debug.LogWarning("[GeneralConfigManager] Configuration chargée depuis LoadDefaultConfig (dernier essai)");
                return;
            }
            catch (System.Exception e)
            {
                Debug.LogError($"[GeneralConfigManager] Erreur lors du dernier essai de chargement: {e.Message}");
            }
        }
        
        // Si le chargement échoue, créer une configuration vide (sans valeurs en dur)
        // Les scripts devront gérer le cas où les valeurs sont null/vides
        Debug.LogError("[GeneralConfigManager] ❌ Impossible de charger general-config.json - Configuration vide créée");
        Debug.LogError("[GeneralConfigManager] ❌ Le fichier general-config.json est OBLIGATOIRE pour le fonctionnement du jeu");
        
        generalConfig = new GeneralConfig
        {
            // Configuration minimale sans URLs en dur
            // Les scripts devront vérifier si les valeurs sont null/vides
        };
        
        isConfigLoaded = true;
    }
    
    /// <summary>
    /// Attend que la configuration soit chargée (coroutine)
    /// </summary>
    public IEnumerator WaitForConfigLoaded()
    {
        float timeout = 10f; // 10 secondes max
        float elapsed = 0f;
        
        while (!isConfigLoaded && elapsed < timeout)
        {
            yield return new WaitForSeconds(0.1f);
            elapsed += 0.1f;
        }
        
        if (!isConfigLoaded)
        {
            Debug.LogError("[GeneralConfigManager] ⏱️ Timeout : La configuration n'a pas pu être chargée en 10 secondes");
        }
    }
    
    /// <summary>
    /// Vérifie si la configuration est chargée
    /// </summary>
    public bool IsConfigLoaded()
    {
        return isConfigLoaded;
    }
    
    /// <summary>
    /// Obtient la configuration générale (attendre que soit chargée)
    /// </summary>
    public GeneralConfig GetConfig()
    {
        if (!isConfigLoaded)
        {
            Debug.LogWarning("[GeneralConfigManager] Configuration pas encore chargée, retour de la config par défaut");
            LoadDefaultConfig();
        }
        return generalConfig;
    }
    
    // Cache pour la police par défaut
    private TMPro.TMP_FontAsset cachedDefaultFont;
    
    /// <summary>
    /// Obtient la police par défaut configurée
    /// </summary>
    public TMPro.TMP_FontAsset GetDefaultFont()
    {
        if (cachedDefaultFont != null) return cachedDefaultFont;
        
        string fontName = GetConfig()?.defaultFontName ?? "Lato SDF";
        
        // Essayer de charger depuis Resources/Fonts/
        cachedDefaultFont = Resources.Load<TMPro.TMP_FontAsset>($"Fonts/{fontName}");
        
        if (cachedDefaultFont == null)
        {
            // Essayer directement avec le nom
            cachedDefaultFont = Resources.Load<TMPro.TMP_FontAsset>(fontName);
        }
        
        if (cachedDefaultFont == null)
        {
            Debug.LogWarning($"[GeneralConfigManager] Police '{fontName}' non trouvée, utilisation de la police par défaut TMP");
        }
        else
        {
            Debug.Log($"[GeneralConfigManager] Police par défaut chargée: {fontName}");
        }
        
        return cachedDefaultFont;
    }
    
    /// <summary>
    /// Obtient la map par défaut
    /// </summary>
    public string GetDefaultMapId()
    {
        return GetConfig().defaultMapId;
    }
    
    /// <summary>
    /// Obtient l'URL du registre des maps
    /// </summary>
    public string GetMapsRegistryUrl()
    {
        return ResolveStreamingAssetsPath(GetConfig().mapsRegistryUrl);
    }
    
    /// <summary>
    /// Convertit un chemin STREAMING_ASSETS/ en URL file:/// complète
    /// </summary>
    private string ResolveStreamingAssetsPath(string path)
    {
        if (string.IsNullOrEmpty(path))
            return path;
            
        if (path.StartsWith("STREAMING_ASSETS/"))
        {
            string relativePath = path.Substring("STREAMING_ASSETS/".Length);
            string fullPath = System.IO.Path.Combine(Application.streamingAssetsPath, relativePath);

            // WebGL: Application.streamingAssetsPath est déjà une URL accessible par le serveur (pas de file:///)
            // Éditeur/Standalone: on peut utiliser file:///
            fullPath = fullPath.Replace("\\", "/");

            #if UNITY_WEBGL && !UNITY_EDITOR
            return fullPath; // ex: /StreamingAssets/json/… servi par le serveur WebGL
            #else
            return "file:///" + fullPath;
            #endif
        }
        
        return path;
    }
    
    /// <summary>
    /// Force le rechargement de la configuration
    /// </summary>
    public void ReloadConfig()
    {
        isConfigLoaded = false;
        StartCoroutine(LoadGeneralConfig());
    }
    
    // ==========================================
    // MÉTHODES UTILITAIRES POUR LES CHEMINS D'ASSETS
    // ==========================================
    
    /// <summary>
    /// Construit l'URL complète pour une vidéo de popup
    /// </summary>
    /// <param name="fileName">Nom du fichier (peut inclure un sous-dossier comme "focus/video.mp4")</param>
    /// <returns>URL complète</returns>
    public string GetPopupVideoUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.popupVideoPath))
        {
            Debug.LogWarning("[GeneralConfigManager] popupVideoPath non défini dans la config");
            return fileName;
        }
        
        return config.assetsPaths.popupVideoPath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour une image décorative
    /// </summary>
    /// <param name="fileName">Nom du fichier (peut inclure un sous-dossier comme "personnages/image.png")</param>
    /// <returns>URL complète</returns>
    public string GetDecoratorImageUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.decoratorImagePath))
        {
            Debug.LogWarning("[GeneralConfigManager] decoratorImagePath non défini dans la config");
            return fileName;
        }
        
        return config.assetsPaths.decoratorImagePath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour une vidéo générale
    /// </summary>
    /// <param name="fileName">Nom du fichier (peut inclure un sous-dossier comme "focus/video.mp4")</param>
    /// <returns>URL complète</returns>
    public string GetVideoUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.videoPath))
        {
            Debug.LogWarning("[GeneralConfigManager] videoPath non défini dans la config");
            return fileName;
        }
        
        return config.assetsPaths.videoPath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour une image de fond
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetBackgroundImageUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.backgroundImagePath))
        {
            Debug.LogWarning("[GeneralConfigManager] backgroundImagePath non défini dans la config");
            return fileName;
        }
        
        return config.assetsPaths.backgroundImagePath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour une vidéo de fond
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetBackgroundVideoUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.backgroundVideoPath))
        {
            Debug.LogWarning("[GeneralConfigManager] backgroundVideoPath non défini dans la config");
            return fileName;
        }
        
        return config.assetsPaths.backgroundVideoPath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour un fichier UI
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetUIUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.uiPath))
        {
            Debug.LogWarning("[GeneralConfigManager] uiPath non défini dans la config");
            return fileName;
        }
        
        return config.assetsPaths.uiPath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour un crosshair (curseur de visée)
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetCrosshairUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Détecter et ignorer les URLs placeholder (via.placeholder.com)
        if (fileName.Contains("via.placeholder.com"))
        {
            Debug.LogWarning($"[GeneralConfigManager] ⚠️ URL placeholder détectée pour crosshair: '{fileName}'. Ignorée, utilisation du nom de fichier par défaut.");
            // Retourner une chaîne vide pour déclencher un fallback ou une erreur
            return "";
        }
            
        // Si le fichier contient déjà une URL complète valide, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.crosshairPath))
        {
            // Fallback vers gameAssetsPath si crosshairPath n'est pas défini
            Debug.LogWarning("[GeneralConfigManager] crosshairPath non défini, utilisation de gameAssetsPath");
            return GetGameAssetsUrl(fileName);
        }
        
        return config.assetsPaths.crosshairPath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour un asset de jeu (gun, crosshair, impact, etc.)
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetGameAssetsUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.gameAssetsPath))
        {
            // Fallback vers uiPath si gameAssetsPath n'est pas défini
            Debug.LogWarning("[GeneralConfigManager] gameAssetsPath non défini, utilisation de uiPath");
            return GetUIUrl(fileName);
        }
        
        string basePath = config.assetsPaths.gameAssetsPath;
        
        // Si le chemin de base est STREAMING_ASSETS/, le résoudre d'abord
        if (basePath.StartsWith("STREAMING_ASSETS/"))
        {
            basePath = ResolveStreamingAssetsPath(basePath);
        }
        
        return basePath + fileName;
    }
    
    /// <summary>
    /// Construit l'URL complète pour un fichier de dialogue
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetDialogueUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.dialoguePath))
        {
            Debug.LogWarning("[GeneralConfigManager] dialoguePath non défini dans la config");
            return fileName;
        }
        
        string fullUrl = config.assetsPaths.dialoguePath + fileName;
        return ResolveStreamingAssetsPath(fullUrl);
    }
    
    /// <summary>
    /// Construit l'URL complète pour un fichier de questions
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetQuestionsUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.questionsPath))
        {
            Debug.LogWarning("[GeneralConfigManager] questionsPath non défini dans la config");
            return fileName;
        }
        
        string fullUrl = config.assetsPaths.questionsPath + fileName;
        return ResolveStreamingAssetsPath(fullUrl);
    }
    
    /// <summary>
    /// Construit l'URL complète pour un fichier de configuration de map
    /// </summary>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetMapConfigUrl(string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
            
        var config = GetConfig();
        if (config.assetsPaths == null || string.IsNullOrEmpty(config.assetsPaths.mapsPath))
        {
            Debug.LogWarning("[GeneralConfigManager] mapsPath non défini dans la config");
            return fileName;
        }
        
        string fullUrl = config.assetsPaths.mapsPath + fileName;
        return ResolveStreamingAssetsPath(fullUrl);
    }
    
    /// <summary>
    /// Méthode générique pour construire une URL d'asset
    /// </summary>
    /// <param name="assetType">Type d'asset (popupVideo, decoratorImage, video, backgroundImage, backgroundVideo, ui, dialogue, questions, maps)</param>
    /// <param name="fileName">Nom du fichier</param>
    /// <returns>URL complète</returns>
    public string GetAssetUrl(string assetType, string fileName)
    {
        if (string.IsNullOrEmpty(fileName))
            return "";
            
        // Si le fichier contient déjà une URL complète, le retourner tel quel
        if (fileName.StartsWith("http://") || fileName.StartsWith("https://"))
            return fileName;
        
        switch (assetType.ToLower())
        {
            case "popupvideo":
                return GetPopupVideoUrl(fileName);
            case "decoratorimage":
                return GetDecoratorImageUrl(fileName);
            case "video":
                return GetVideoUrl(fileName);
            case "backgroundimage":
                return GetBackgroundImageUrl(fileName);
            case "backgroundvideo":
                return GetBackgroundVideoUrl(fileName);
            case "ui":
                return GetUIUrl(fileName);
            case "dialogue":
                return GetDialogueUrl(fileName);
            case "questions":
                return GetQuestionsUrl(fileName);
            case "maps":
                return GetMapConfigUrl(fileName);
            default:
                Debug.LogWarning($"[GeneralConfigManager] Type d'asset inconnu: {assetType}");
                return fileName;
        }
    }
    
    // ==========================================
    // GETTERS POUR LES URLs D'API
    // ==========================================
    
    /// <summary>
    /// Obtient l'URL de base de l'API
    /// </summary>
    public string GetApiBaseUrl()
    {
        var config = GetConfig();
        if (config == null)
        {
            Debug.LogError("[GeneralConfigManager] ❌ Config est null");
            return "";
        }
        
        if (config.apiUrls == null)
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls est null dans general-config.json");
            return "";
        }
        
        if (string.IsNullOrEmpty(config.apiUrls.baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.baseUrl est vide dans general-config.json");
            return "";
        }
        
        return config.apiUrls.baseUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API de questions
    /// </summary>
    public string GetQuestionsApiUrl(int answersCount, int count = 3)
    {
        var config = GetConfig();
        string baseUrl = GetApiBaseUrl();
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.questionsApi))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.questionsApi non défini dans general-config.json");
            return ""; // Retourner vide plutôt qu'une valeur en dur
        }
        string apiPath = config.apiUrls.questionsApi;
        return $"{baseUrl}{apiPath}?type=qcu&answers={answersCount}&count={count}";
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API de questions pour les jeux de type trous
    /// </summary>
    public string GetTrousQuestionsApiUrl()
    {
        var config = GetConfig();
        string baseUrl = GetApiBaseUrl();
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.questionsApi))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.questionsApi non défini dans general-config.json");
            return ""; // Retourner vide plutôt qu'une valeur en dur
        }
        
        // Récupérer le count depuis la configuration (défaut: 1)
        int count = 1;
        if (config.trousGame != null && config.trousGame.count > 0)
        {
            count = config.trousGame.count;
        }
        
        string apiPath = config.apiUrls.questionsApi;
        return $"{baseUrl}{apiPath}?type=text_blank&count={count}";
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour récupérer la configuration de la scène main
    /// Format: {baseUrl}/api/ujsa/projects/{slug}
    /// </summary>
    public string GetMainSceneConfigApiUrl()
    {
        var config = GetConfig();
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(config.slug))
        {
            Debug.LogError("[GeneralConfigManager] ❌ slug non défini dans general-config.json");
            return ""; // Retourner vide plutôt qu'une valeur en dur
        }
        
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        string projectSlug = config.slug;
        string apiUrl = $"{baseUrl}/api/ujsa/projects/{projectSlug}";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API main scene construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour récupérer la configuration d'une quest (map)
    /// Format: {baseUrl}/api/ujsa/quests/{questId}
    /// </summary>
    public string GetQuestConfigApiUrl(int questId)
    {
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        if (questId <= 0)
        {
            Debug.LogError($"[GeneralConfigManager] ❌ questId invalide: {questId}");
            return "";
        }
        
        string apiUrl = $"{baseUrl}/api/ujsa/quests/{questId}";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API quest construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour récupérer tous les projets avec leurs quêtes
    /// Format: {baseUrl}/api/ujsa/projects/{projectSlug}
    /// </summary>
    public string GetProjectsApiUrl()
    {
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        var config = GetConfig();
        if (config == null)
        {
            Debug.LogError("[GeneralConfigManager] ❌ config non chargée");
            return "";
        }
        
        string projectSlug = config.slug;
        if (string.IsNullOrEmpty(projectSlug))
        {
            Debug.LogError("[GeneralConfigManager] ❌ projectSlug non défini dans general-config.json");
            return "";
        }
        
        string apiUrl = $"{baseUrl}/api/ujsa/projects/{projectSlug}";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API projects construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour récupérer la configuration d'un jeu
    /// Format: {baseUrl}/api/ujsa/games/{gameId}?difficulty={difficulty}
    /// </summary>
    public string GetGameConfigApiUrl(int gameId, string difficulty = "Débutant")
    {
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        if (gameId <= 0)
        {
            Debug.LogError($"[GeneralConfigManager] ❌ gameId invalide: {gameId}");
            return "";
        }
        
        // Encoder le paramètre difficulty pour l'URL (gère les accents)
        string encodedDifficulty = UnityEngine.Networking.UnityWebRequest.EscapeURL(difficulty);
        string apiUrl = $"{baseUrl}/api/ujsa/games/{gameId}?difficulty={encodedDifficulty}";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API game construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL de préconnexion
    /// </summary>
    public string GetAuthPreconnectUrl()
    {
        var config = GetConfig();
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            return "";
        }
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.authPreconnect))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.authPreconnect non défini dans general-config.json");
            return "";
        }
        return baseUrl + config.apiUrls.authPreconnect;
    }
    
    /// <summary>
    /// Obtient l'URL de connexion
    /// </summary>
    public string GetAuthLoginUrl()
    {
        var config = GetConfig();
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            return "";
        }
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.authLogin))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.authLogin non défini dans general-config.json");
            return "";
        }
        return baseUrl + config.apiUrls.authLogin;
    }
    
    /// <summary>
    /// Obtient l'URL de vérification du token
    /// </summary>
    public string GetAuthVerifyTokenUrl()
    {
        var config = GetConfig();
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            return "";
        }
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.authVerifyToken))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.authVerifyToken non défini dans general-config.json");
            return "";
        }
        return baseUrl + config.apiUrls.authVerifyToken;
    }
    
    /// <summary>
    /// Obtient l'URL d'inscription
    /// </summary>
    public string GetRegistrationUrl()
    {
        var config = GetConfig();
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.registrationUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.registrationUrl non défini dans general-config.json");
            return ""; // Retourner vide plutôt qu'une valeur en dur
        }
        return config.apiUrls.registrationUrl;
    }

    public string GetForgotPasswordUrl()
    {
        var config = GetConfig();
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.forgotPasswordUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.forgotPasswordUrl non défini dans general-config.json");
            return "";
        }
        return config.apiUrls.forgotPasswordUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour envoyer les réponses d'un joueur pour un jeu
    /// Format: {baseUrl}/api/ujsa/games/{gameId}/answers
    /// </summary>
    public string GetGameAnswersApiUrl(int gameId)
    {
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        var config = GetConfig();
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.gameAnswersApi))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.gameAnswersApi non défini dans general-config.json");
            return "";
        }
        
        if (gameId <= 0)
        {
            Debug.LogError($"[GeneralConfigManager] ❌ gameId invalide: {gameId}");
            return "";
        }
        
        // Remplacer {gameId} par la valeur réelle
        string apiPath = config.apiUrls.gameAnswersApi.Replace("{gameId}", gameId.ToString());
        string apiUrl = $"{baseUrl}{apiPath}";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API answers construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour reset les réponses avant un jeu
    /// Format: {baseUrl}/api/ujsa/games/{gameId}/answers/reset
    /// </summary>
    public string GetGameAnswersResetApiUrl(int gameId)
    {
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        var config = GetConfig();
        if (config.apiUrls == null || string.IsNullOrEmpty(config.apiUrls.gameAnswersResetApi))
        {
            Debug.LogError("[GeneralConfigManager] ❌ apiUrls.gameAnswersResetApi non défini dans general-config.json");
            return "";
        }
        
        if (gameId <= 0)
        {
            Debug.LogError($"[GeneralConfigManager] ❌ gameId invalide: {gameId}");
            return "";
        }
        
        // Remplacer {gameId} par la valeur réelle
        string apiPath = config.apiUrls.gameAnswersResetApi.Replace("{gameId}", gameId.ToString());
        string apiUrl = $"{baseUrl}{apiPath}";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API answers reset construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour reset la progression courante d'une quête
    /// Format: {baseUrl}/api/ujsa/quests/{questId}/answers/reset-current
    /// </summary>
    public string GetQuestResetCurrentApiUrl(int questId)
    {
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        if (questId <= 0)
        {
            Debug.LogError($"[GeneralConfigManager] ❌ questId invalide: {questId}");
            return "";
        }
        
        string apiUrl = $"{baseUrl}/api/ujsa/quests/{questId}/answers/reset-current";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API quest reset-current construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL complète de l'API pour reset les réponses d'une quête par difficulté
    /// Format: {baseUrl}/api/ujsa/quests/{questId}/answers/reset-by-difficulty
    /// </summary>
    public string GetQuestAnswersResetByDifficultyUrl(int questId)
    {
        string baseUrl = GetApiBaseUrl();
        if (string.IsNullOrEmpty(baseUrl))
        {
            Debug.LogError("[GeneralConfigManager] ❌ baseUrl non défini dans general-config.json");
            return "";
        }
        
        if (questId <= 0)
        {
            Debug.LogError($"[GeneralConfigManager] ❌ questId invalide: {questId}");
            return "";
        }
        
        string apiUrl = $"{baseUrl}/api/ujsa/quests/{questId}/answers/reset-by-difficulty";
        Debug.Log($"[GeneralConfigManager] 🔍 URL API quest reset-by-difficulty construite: {apiUrl}");
        return apiUrl;
    }
    
    /// <summary>
    /// Obtient l'URL de configuration par défaut pour un type de jeu
    /// </summary>
    public string GetDefaultGameConfigUrl(string gameType)
    {
        var config = GetConfig();
        if (config.defaultConfigUrls == null)
        {
            Debug.LogWarning("[GeneralConfigManager] defaultConfigUrls non défini");
            return "";
        }
        
        switch (gameType.ToLower())
        {
            case "shooting":
                return ResolveStreamingAssetsPath(config.defaultConfigUrls.shootingGameConfig);
            case "calculator":
                return ResolveStreamingAssetsPath(config.defaultConfigUrls.calculatorGameConfig);
            case "levels":
                return ResolveStreamingAssetsPath(config.defaultConfigUrls.levelsConfigUrl);
            default:
                Debug.LogWarning($"[GeneralConfigManager] Type de jeu inconnu: {gameType}");
                return "";
        }
    }
    
    /// <summary>
    /// Obtient l'URL racine pour les JSON de niveaux
    /// </summary>
    public string GetLevelsConfigUrlRootJson()
    {
        var config = GetConfig();
        if (config.levelsConfig == null || string.IsNullOrEmpty(config.levelsConfig.urlRootJson))
        {
            Debug.LogError("[GeneralConfigManager] ❌ levelsConfig.urlRootJson non défini dans general-config.json");
            return ""; // Retourner vide plutôt qu'une valeur en dur
        }
        return config.levelsConfig.urlRootJson;
    }
    
    /// <summary>
    /// Obtient l'URL racine pour les thumbnails de niveaux
    /// </summary>
    public string GetLevelsConfigUrlRootThumbnails()
    {
        var config = GetConfig();
        if (config.levelsConfig == null || string.IsNullOrEmpty(config.levelsConfig.urlRootThumbnails))
        {
            Debug.LogError("[GeneralConfigManager] ❌ levelsConfig.urlRootThumbnails non défini dans general-config.json");
            return ""; // Retourner vide plutôt qu'une valeur en dur
        }
        return config.levelsConfig.urlRootThumbnails;
    }
    
    // ==========================================
    // GETTERS POUR LES VALEURS PAR DÉFAUT
    // ==========================================
    
    /// <summary>
    /// Obtient les assets par défaut
    /// </summary>
    public DefaultAssets GetDefaultAssets()
    {
        var config = GetConfig();
        return config.defaultAssets;
    }
    
    /// <summary>
    /// Obtient la palette de couleurs
    /// </summary>
    public ColorPalette GetColorPalette()
    {
        var config = GetConfig();
        return config.colorPalette;
    }
    
    /// <summary>
    /// Obtient la configuration popup par défaut
    /// </summary>
    public DefaultPopupConfig GetDefaultPopupConfig()
    {
        var config = GetConfig();
        return config.defaultPopupConfig;
    }
    
    /// <summary>
    /// Obtient la configuration header par défaut
    /// </summary>
    public DefaultHeaderConfig GetDefaultHeaderConfig()
    {
        var config = GetConfig();
        return config.defaultHeaderConfig;
    }
    
    /// <summary>
    /// Obtient la configuration dialogue par défaut
    /// </summary>
    public DefaultDialogueConfig GetDefaultDialogueConfig()
    {
        var config = GetConfig();
        return config.defaultDialogueConfig;
    }
    
    /// <summary>
    /// Obtient les messages de feedback par défaut
    /// </summary>
    public DefaultFeedbackMessages GetDefaultFeedbackMessages()
    {
        var config = GetConfig();
        return config.defaultFeedbackMessages;
    }
    
    /// <summary>
    /// Obtient un style de bouton par son nom (ex: "validationPurple")
    /// </summary>
    public ButtonStyleConfig GetButtonStyle(string styleName)
    {
        if (string.IsNullOrEmpty(styleName)) return null;
        
        var config = GetConfig();
        if (config?.buttonStyles == null) return null;
        
        return config.buttonStyles.GetStyle(styleName);
    }
    
    /// <summary>
    /// Obtient la configuration UI par défaut
    /// </summary>
    public DefaultUIConfig GetDefaultUIConfig()
    {
        var config = GetConfig();
        if (config == null)
        {
            Debug.LogWarning("[GeneralConfigManager] GetDefaultUIConfig - config est null");
            return null;
        }
        if (config.defaultUIConfig == null)
        {
            Debug.LogError("[GeneralConfigManager] GetDefaultUIConfig - config.defaultUIConfig est null!");
        }
        return config.defaultUIConfig;
    }
    
    /// <summary>
    /// Obtient la résolution par défaut
    /// </summary>
    public DefaultResolution GetDefaultResolution()
    {
        var config = GetConfig();
        return config.defaultResolution;
    }
    
    /// <summary>
    /// Obtient la configuration crosshair par défaut
    /// </summary>
    public DefaultCrosshairConfig GetDefaultCrosshairConfig()
    {
        var config = GetConfig();
        return config.defaultCrosshairConfig;
    }
    
    /// <summary>
    /// Obtient la configuration gun par défaut
    /// </summary>
    public DefaultGunConfig GetDefaultGunConfig()
    {
        var config = GetConfig();
        return config.defaultGunConfig;
    }
    
    /// <summary>
    /// Obtient la configuration zone decorator par défaut
    /// </summary>
    public DefaultZoneDecorator GetDefaultZoneDecorator()
    {
        var config = GetConfig();
        return config.defaultZoneDecorator;
    }
    
    /// <summary>
    /// Obtient les timings d'animations
    /// </summary>
    public AnimationTimings GetAnimationTimings()
    {
        var config = GetConfig();
        return config.animations;
    }
}
