using UnityEngine;
using UnityEngine.UI;
using TMPro;

#if ENABLE_INPUT_SYSTEM
using UnityEngine.InputSystem;
#endif

/// <summary>
/// Script de diagnostic pour le panneau des paramètres dans la scène Map
/// Appuyer sur F12 pour afficher le diagnostic complet
/// </summary>
public class MapSettingsDiagnostic : MonoBehaviour
{
    void Update()
    {
        #if ENABLE_INPUT_SYSTEM
        // Nouveau Input System
        if (Keyboard.current != null && Keyboard.current.f12Key.wasPressedThisFrame)
        {
            RunDiagnostic();
        }
        #else
        // Ancien Input System
        if (Input.GetKeyDown(KeyCode.F12))
        {
            RunDiagnostic();
        }
        #endif
    }
    
    void RunDiagnostic()
    {
        Debug.Log("═══════════════════════════════════════════════════");
        Debug.Log("🔍 DIAGNOSTIC PANNEAU PARAMÈTRES - SCÈNE MAP");
        Debug.Log("═══════════════════════════════════════════════════");
        
        // 1. Trouver le SettingsPanel
        SettingsManager settingsManager = FindFirstObjectByType<SettingsManager>();
        if (settingsManager == null)
        {
            Debug.LogError("❌ SettingsManager introuvable !");
            return;
        }
        
        GameObject settingsPanel = settingsManager.settingsPanel;
        if (settingsPanel == null)
        {
            Debug.LogError("❌ settingsPanel est NULL !");
            return;
        }
        
        Debug.Log($"✅ SettingsPanel trouvé: {settingsPanel.name}");
        Debug.Log($"   Active: {settingsPanel.activeInHierarchy}");
        
        // 2. Analyser le Canvas
        Canvas canvas = settingsPanel.GetComponent<Canvas>();
        if (canvas != null)
        {
            Debug.Log($"📊 Canvas Settings Panel:");
            Debug.Log($"   Override Sorting: {canvas.overrideSorting}");
            Debug.Log($"   Sorting Order: {canvas.sortingOrder}");
            Debug.Log($"   Render Mode: {canvas.renderMode}");
        }
        else
        {
            Debug.LogWarning("⚠️ Pas de Canvas sur le SettingsPanel");
        }
        
        // 3. Analyser le DarkBackground
        GameObject darkBg = settingsManager.darkBackground;
        if (darkBg != null)
        {
            Debug.Log($"🌑 Dark Background:");
            Debug.Log($"   Active: {darkBg.activeInHierarchy}");
            Image bgImage = darkBg.GetComponent<Image>();
            if (bgImage != null)
            {
                Debug.Log($"   Couleur: {bgImage.color}");
                Debug.Log($"   RaycastTarget: {bgImage.raycastTarget}");
            }
        }
        else
        {
            Debug.LogError("❌ darkBackground est NULL !");
        }
        
        // 4. Trouver le QuetesTab
        QuetesTab quetesTab = FindFirstObjectByType<QuetesTab>();
        if (quetesTab == null)
        {
            Debug.LogError("❌ QuetesTab introuvable !");
            return;
        }
        
        Debug.Log($"📋 QuetesTab trouvé: {quetesTab.gameObject.name}");
        Debug.Log($"   Active: {quetesTab.gameObject.activeInHierarchy}");
        
        // 5. Analyser la hiérarchie du QuetesTab
        Transform leftPanel = quetesTab.transform.Find("LeftPanel");
        if (leftPanel != null)
        {
            Debug.Log($"📦 LeftPanel trouvé:");
            Debug.Log($"   Active: {leftPanel.gameObject.activeInHierarchy}");
            RectTransform rect = leftPanel.GetComponent<RectTransform>();
            Debug.Log($"   Position: {rect.anchoredPosition}");
            Debug.Log($"   Taille: {rect.rect.width} x {rect.rect.height}");
            Debug.Log($"   Anchors: ({rect.anchorMin.x}, {rect.anchorMin.y}) -> ({rect.anchorMax.x}, {rect.anchorMax.y})");
            
            Image bgImage = leftPanel.GetComponent<Image>();
            if (bgImage != null)
            {
                Debug.Log($"   Couleur fond: {bgImage.color}");
                Debug.Log($"   RaycastTarget: {bgImage.raycastTarget}");
            }
            
            // 6. Analyser le ScrollView
            Transform scrollView = leftPanel.Find("ScrollView");
            if (scrollView != null)
            {
                Debug.Log($"📜 ScrollView trouvé:");
                Debug.Log($"   Active: {scrollView.gameObject.activeInHierarchy}");
                
                Transform viewport = scrollView.Find("Viewport");
                if (viewport != null)
                {
                    Debug.Log($"🔍 Viewport trouvé:");
                    Debug.Log($"   Active: {viewport.gameObject.activeInHierarchy}");
                    
                    Transform content = viewport.Find("Content");
                    if (content != null)
                    {
                        Debug.Log($"📄 Content trouvé:");
                        Debug.Log($"   Active: {content.gameObject.activeInHierarchy}");
                        Debug.Log($"   Enfants: {content.childCount}");
                        
                        RectTransform contentRect = content.GetComponent<RectTransform>();
                        Debug.Log($"   Taille Content: {contentRect.rect.width} x {contentRect.rect.height}");
                        Debug.Log($"   sizeDelta: {contentRect.sizeDelta}");
                        
                        // 7. Lister les items
                        for (int i = 0; i < content.childCount; i++)
                        {
                            Transform child = content.GetChild(i);
                            Debug.Log($"   📌 Item {i}: {child.name}");
                            Debug.Log($"      Active: {child.gameObject.activeInHierarchy}");
                            RectTransform childRect = child.GetComponent<RectTransform>();
                            Debug.Log($"      Position: {childRect.anchoredPosition}");
                            Debug.Log($"      Taille: {childRect.rect.width} x {childRect.rect.height}");
                            
                            Image img = child.GetComponent<Image>();
                            if (img != null)
                            {
                                Debug.Log($"      Couleur: {img.color}");
                            }
                            
                            // Vérifier les TextMeshPro
                            TextMeshProUGUI[] texts = child.GetComponentsInChildren<TextMeshProUGUI>();
                            Debug.Log($"      Textes: {texts.Length}");
                            foreach (var text in texts)
                            {
                                Debug.Log($"         → '{text.text}' (couleur: {text.color}, fontSize: {text.fontSize})");
                            }
                        }
                    }
                    else
                    {
                        Debug.LogError("❌ Content introuvable !");
                    }
                }
                else
                {
                    Debug.LogError("❌ Viewport introuvable !");
                }
            }
            else
            {
                Debug.LogError("❌ ScrollView introuvable !");
            }
        }
        else
        {
            Debug.LogError("❌ LeftPanel introuvable !");
        }
        
        // 8. Vérifier tous les Canvas de la scène
        Debug.Log("🎨 TOUS LES CANVAS DE LA SCÈNE:");
        Canvas[] allCanvases = FindObjectsByType<Canvas>(FindObjectsSortMode.None);
        foreach (Canvas c in allCanvases)
        {
            Debug.Log($"   → {c.gameObject.name}: SortingOrder={c.sortingOrder}, OverrideSorting={c.overrideSorting}, Active={c.gameObject.activeInHierarchy}");
        }
        
        // 9. Vérifier les GraphicRaycasters
        Debug.Log("🖱️ TOUS LES GRAPHIC RAYCASTERS:");
        GraphicRaycaster[] raycasters = FindObjectsByType<GraphicRaycaster>(FindObjectsSortMode.None);
        foreach (GraphicRaycaster gr in raycasters)
        {
            Debug.Log($"   → {gr.gameObject.name}: Enabled={gr.enabled}, Active={gr.gameObject.activeInHierarchy}");
        }
        
        Debug.Log("═══════════════════════════════════════════════════");
        Debug.Log("🔍 FIN DU DIAGNOSTIC");
        Debug.Log("═══════════════════════════════════════════════════");
    }
}

