using UnityEngine;
using UnityEngine.UI;
using TMPro;
using System.Text;

/// <summary>
/// Onglet Debug qui affiche toutes les variables utilisateur stockées
/// </summary>
public class DebugTab : SettingsTab
{
    public Transform contentContainer;
    private TextMeshProUGUI debugText;
    
    protected override void Awake()
    {
        base.Awake();
        tabName = "DEBUG";
        tabOrder = 4;
    }
    
    protected override void OnEnable()
    {
        base.OnEnable();
        UpdateDebugInfo();
    }
    
    void UpdateDebugInfo()
    {
        if (contentContainer == null)
        {
            Debug.LogError("[DebugTab] ❌ contentContainer est NULL !");
            return;
        }
        
        // Nettoyer l'ancien contenu
        foreach (Transform child in contentContainer)
        {
            Destroy(child.gameObject);
        }
        debugText = null;
        
        // Construire le texte avec toutes les informations utilisateur
        StringBuilder sb = new StringBuilder();
        sb.AppendLine("═══════════════════════════════════════════════════════════");
        sb.AppendLine("<size=24><b>INFORMATIONS DE DEBUG</b></size>");
        sb.AppendLine("═══════════════════════════════════════════════════════════");
        sb.AppendLine();
        
        if (UserDataManager.Instance != null)
        {
            if (UserDataManager.Instance.IsLoggedIn())
            {
                sb.AppendLine("<b><size=22>UTILISATEUR CONNECTÉ</size></b>");
                sb.AppendLine();
                sb.AppendLine($"<b>User ID:</b> {UserDataManager.Instance.userId}");
                sb.AppendLine($"<b>Email:</b> {UserDataManager.Instance.email}");
                sb.AppendLine();
                sb.AppendLine($"<b>Temps de jeu:</b> {UserDataManager.Instance.FormatPlayTime(UserDataManager.Instance.playTime)}");
                sb.AppendLine();
                sb.AppendLine("───────────────────────────────────────────────────────────");
                sb.AppendLine();
                
                // Récupérer les quêtes depuis MainSceneManager
                AppendQuestsInfo(sb);
            }
            else
            {
                sb.AppendLine("<size=22><b>UTILISATEUR NON CONNECTÉ</b></size>");
                sb.AppendLine();
                sb.AppendLine("<i>Connectez-vous pour voir vos données de progression.</i>");
                sb.AppendLine();
                sb.AppendLine($"<b>Temps de jeu:</b> {UserDataManager.Instance.FormatPlayTime(UserDataManager.Instance.playTime)}");
            }
        }
        else
        {
            Debug.LogWarning("[DebugTab] UserDataManager.Instance est NULL !");
            sb.AppendLine("<size=22><b>ERREUR SYSTÈME</b></size>");
            sb.AppendLine();
            sb.AppendLine("<color=red>UserDataManager non initialisé</color>");
            sb.AppendLine();
            sb.AppendLine("Veuillez redémarrer le jeu.");
        }
        
        sb.AppendLine();
        sb.AppendLine("═══════════════════════════════════════════════════════════");
        
        string finalText = sb.ToString();
        
        // Créer un simple TextMeshProUGUI dans le contentContainer
        GameObject textObj = new GameObject("DebugText");
        textObj.transform.SetParent(contentContainer, false);
        
        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.offsetMin = Vector2.zero;
        textRect.offsetMax = Vector2.zero;
        
        debugText = textObj.AddComponent<TextMeshProUGUI>();
        
        // CRITIQUE : Charger un font asset pour TextMeshPro
        TMP_FontAsset font = Resources.Load<TMP_FontAsset>("Fonts & Materials/LiberationSans SDF");
        if (font == null)
        {
            // Fallback : utiliser le premier font disponible
            TMP_FontAsset[] fonts = Resources.FindObjectsOfTypeAll<TMP_FontAsset>();
            if (fonts.Length > 0) font = fonts[0];
        }
        if (font != null) debugText.font = font;
        
        debugText.text = finalText;
        debugText.fontSize = 20;
        debugText.alignment = TextAlignmentOptions.TopLeft;
        debugText.color = new Color(0.1f, 0.1f, 0.1f, 1f);
        debugText.textWrappingMode = TMPro.TextWrappingModes.Normal;
        debugText.overflowMode = TMPro.TextOverflowModes.Overflow;
    }
    
    
    void AppendQuestsInfo(StringBuilder sb)
    {
        sb.AppendLine("<b><size=22>QUÊTES EN COURS</size></b>");
        sb.AppendLine();
        
        // Essayer de récupérer les quêtes depuis MainSceneManager
        MainSceneManager mainSceneManager = FindFirstObjectByType<MainSceneManager>();
        
        if (mainSceneManager != null)
        {
            // Utiliser la réflexion pour accéder au champ privé sceneConfig
            var sceneConfigField = mainSceneManager.GetType().GetField("sceneConfig", 
                System.Reflection.BindingFlags.NonPublic | System.Reflection.BindingFlags.Instance);
            
            if (sceneConfigField != null)
            {
                var sceneConfig = sceneConfigField.GetValue(mainSceneManager) as MainSceneConfig;
                
                if (sceneConfig?.quests != null)
                {
                    int inProgressCount = 0;
                    
                    foreach (var quest in sceneConfig.quests)
                    {
                        if (quest.user != null)
                        {
                            foreach (var userProgress in quest.user)
                            {
                                // Afficher seulement si completion > 0
                                if (userProgress.completion > 0)
                                {
                                    inProgressCount++;
                                    
                                    sb.AppendLine($"<b>{quest.title}</b>");
                                    sb.AppendLine($"  • <b>Difficulté:</b> <color=#6495ED>{userProgress.difficulty}</color>");
                                    sb.AppendLine($"  • <b>Completion:</b> {userProgress.completion}%");
                                    sb.AppendLine($"  • <b>Score:</b> {userProgress.score}");
                                    sb.AppendLine($"  • <b>Badge:</b> {(string.IsNullOrEmpty(userProgress.badge) ? "Aucun" : userProgress.badge)}");
                                    sb.AppendLine($"  • <b>Dernier jeu complété:</b> {(userProgress.last_completed_game > 0 ? userProgress.last_completed_game.ToString() : "Aucun")}");
                                    sb.AppendLine();
                                }
                            }
                        }
                    }
                    
                    if (inProgressCount == 0)
                    {
                        sb.AppendLine("<i>Aucune quête en cours (completion > 0)</i>");
                        sb.AppendLine();
                    }
                }
                else
                {
                    sb.AppendLine("<color=orange>⚠ Données de quêtes non disponibles</color>");
                    sb.AppendLine();
                }
            }
            else
            {
                sb.AppendLine("<color=orange>⚠ Impossible d'accéder aux données de quêtes</color>");
                sb.AppendLine();
            }
        }
        else
        {
            sb.AppendLine("<color=orange>⚠ MainSceneManager non trouvé</color>");
            sb.AppendLine("<i>(Les données ne sont disponibles que depuis la scène principale)</i>");
            sb.AppendLine();
        }
    }
    
}
