# 🏗️ Architecture des Lanceurs de Jeu

## 📋 Vue d'ensemble

Le projet utilise **deux systèmes** pour lancer les jeux, qui peuvent sembler redondants mais ont des rôles légèrement différents :

---

## 🚀 **GameLauncher** vs **GameConfigLoader**

### **GameLauncher** (`Scripts/GameLauncher.cs`)

**Rôle** : Orchestrateur principal du flow multi-scènes

**Utilisation** :
- Scène : `Player` (dialogues)
- Phases : `Before`, `Shooting`, `After`

**Responsabilités** :
1. ✅ Charge la configuration du jeu depuis JSON
2. ✅ Gère les dialogues groupés (nouveau système)
3. ✅ Supporte les dialogues séparés (ancien système)
4. ✅ Décide quelle scène charger (shooting/calculator)
5. ✅ Gère les transitions Before → Game → After

**Flow typique** :
```
Menu → Map → Zone cliquée → GameLauncher (Player scene)
                                    ↓
                            [Dialogue BEFORE]
                                    ↓
                            Jeu (shooting/calculator)
                                    ↓
                            [Dialogue AFTER]
                                    ↓
                            Retour Map
```

---

### **GameConfigLoader** (`Scripts/GameConfigLoader.cs`)

**Rôle** : Chargeur de configuration simplifié (système alternatif/legacy)

**Utilisation** :
- Scène : `Player` (dialogues)
- Phases : `Before`, `After`

**Responsabilités** :
1. ✅ Charge la configuration du jeu depuis JSON
2. ✅ Gère UNIQUEMENT les dialogues séparés (3 fichiers)
3. ✅ Configure SubtitleManager pour les dialogues
4. ✅ Transition vers shooting/calculator

**Différences clés avec GameLauncher** :
- ❌ Ne supporte PAS les dialogues groupés
- ✅ Plus simple et direct
- ✅ Mieux intégré avec SubtitleManager

---

## 🔀 **Quel système utiliser ?**

### **Utilisez GameLauncher si :**
- ✅ Vous avez des dialogues groupés (`Q*J*_dialogues.json`)
- ✅ Vous voulez le système le plus moderne
- ✅ Vous lancez depuis le nouveau système de cartes

### **Utilisez GameConfigLoader si :**
- ✅ Vous avez des dialogues séparés (intro/success/fail)
- ✅ Vous préférez un système plus simple
- ✅ Vous lancez avec SubtitleManager

---

## 🔧 **Corrections Appliquées**

### **GameConfigLoader - Fix du NullReference**

**Problème** : `GeneralConfigManager.Instance` était null au démarrage

**Solution** : Attente active jusqu'à ce que l'instance soit prête
```csharp
// Attendre que GeneralConfigManager soit prêt
while (GeneralConfigManager.Instance == null)
{
    yield return null;
}

while (!GeneralConfigManager.Instance.IsConfigLoaded())
{
    yield return null;
}
```

**Résultat** : Plus d'erreur "Object reference not set to an instance of an object" ✅

---

## 📊 **Flow Complet**

```
┌─────────────────────────────────────────────────────────────┐
│                     MENU PRINCIPAL                          │
│                  (MainMenuManager)                          │
└──────────────────────┬──────────────────────────────────────┘
                       │
                       ↓ Click sur carte de niveau
┌─────────────────────────────────────────────────────────────┐
│                   LEVEL MANAGER                             │
│              LoadLevel(levelId)                             │
│   • Stocke dans PlayerPrefs                                 │
│   • Lance scène Player                                      │
└──────────────────────┬──────────────────────────────────────┘
                       │
                       ↓
┌─────────────────────────────────────────────────────────────┐
│                   SCÈNE PLAYER                              │
│         GameLauncher OU GameConfigLoader                    │
│   • Charge config JSON                                      │
│   • Lit phase dans PlayerPrefs                              │
└──────────────────────┬──────────────────────────────────────┘
                       │
        ┌──────────────┴──────────────┐
        │                             │
        ↓ Phase = "Before"            ↓ Phase = "After"
┌────────────────┐            ┌────────────────┐
│ DIALOGUE INTRO │            │ DIALOGUE FIN   │
│ SubtitleManager│            │ SubtitleManager│
└───────┬────────┘            └───────┬────────┘
        │                             │
        ↓                             │
┌────────────────┐                    │
│   JEU          │                    │
│ shooting ou    │                    │
│ calculator     │                    │
└───────┬────────┘                    │
        │                             │
        └─────────────┬───────────────┘
                      ↓
              ┌────────────────┐
              │   RETOUR MAP   │
              │  (step suivant)│
              └────────────────┘
```

---

## ⚙️ **Configuration des Logs**

Dans `DebugLogger.cs` :
```csharp
public static LogLevel GameLauncherLogLevel = LogLevel.Warning;
public static LogLevel GameConfigLoaderLogLevel = LogLevel.Warning;
```

Pour débugger les problèmes de lancement :
```csharp
// Temporairement, mettez à Debug
public static LogLevel GameLauncherLogLevel = LogLevel.Debug;
public static LogLevel GameConfigLoaderLogLevel = LogLevel.Debug;
```

---

## 🐛 **Dépannage**

### **Écran noir au lancement d'un jeu**

**Causes possibles :**
1. ❌ `GeneralConfigManager` pas encore chargé → **CORRIGÉ** ✅
2. ❌ URL de config invalide
3. ❌ JSON malformé
4. ❌ Dialogue Before manquant et pas de fallback

**Solution de debug :**
```csharp
// Dans DebugLogger.cs, ligne 28-29
public static LogLevel GameConfigLoaderLogLevel = LogLevel.Debug;
public static LogLevel GameLauncherLogLevel = LogLevel.Debug;
```

### **Conflit entre les deux systèmes**

Les deux systèmes peuvent **coexister** car ils vérifient les mêmes PlayerPrefs :
- `GameConfigUrl`
- `GamePhase`
- `CurrentLevelType`

Mais **un seul doit être actif** dans la scène Player !

---

## ✅ **État Actuel**

- ✅ GameConfigLoader : Corrigé et opérationnel
- ✅ GameLauncher : Opérationnel
- ✅ Logs unifiés avec DebugLogger
- ✅ Attente de GeneralConfigManager
- ✅ Gestion d'erreurs améliorée

---

**Dernière mise à jour** : Fix du NullReferenceException dans GameConfigLoader

