using UnityEditor;
using UnityEditor.Build;
using UnityEngine;

/// <summary>
/// Script pour corriger les erreurs CS2006 liées aux Scripting Define Symbols
/// </summary>
public class FixCompilerDefines : EditorWindow
{
    [MenuItem("Tools/Fix Compiler Defines Errors")]
    public static void Fix()
    {
        Debug.Log("🔧 Correction des Scripting Define Symbols...");
        
        BuildTargetGroup[] targets = new BuildTargetGroup[]
        {
            BuildTargetGroup.Standalone,
            BuildTargetGroup.WebGL,
            BuildTargetGroup.Android,
            BuildTargetGroup.iOS,
            BuildTargetGroup.WSA
        };

        foreach (var target in targets)
        {
            try
            {
                // Récupérer les symboles actuels
#if UNITY_2023_1_OR_NEWER
                var namedTarget = NamedBuildTarget.FromBuildTargetGroup(target);
                string defines = PlayerSettings.GetScriptingDefineSymbols(namedTarget);
#else
                string defines = PlayerSettings.GetScriptingDefineSymbolsForGroup(target);
#endif
                
                if (string.IsNullOrWhiteSpace(defines))
                {
                    Debug.Log($"  ✓ {target}: Aucun symbole (OK)");
                    continue;
                }
                
                // Nettoyer les symboles
                string[] symbols = defines.Split(';', ',');
                var cleanedSymbols = new System.Collections.Generic.List<string>();
                
                foreach (var symbol in symbols)
                {
                    string cleaned = symbol.Trim();
                    if (!string.IsNullOrEmpty(cleaned))
                    {
                        cleanedSymbols.Add(cleaned);
                    }
                }
                
                // Réappliquer les symboles nettoyés
                string cleanedDefines = string.Join(";", cleanedSymbols);
                
#if UNITY_2023_1_OR_NEWER
                PlayerSettings.SetScriptingDefineSymbols(namedTarget, cleanedDefines);
#else
                PlayerSettings.SetScriptingDefineSymbolsForGroup(target, cleanedDefines);
#endif
                
                if (cleanedSymbols.Count > 0)
                {
                    Debug.Log($"  ✓ {target}: {cleanedSymbols.Count} symbole(s) nettoyé(s) - {cleanedDefines}");
                }
                else
                {
                    Debug.Log($"  ✓ {target}: Tous les symboles vides supprimés");
                }
            }
            catch (System.Exception e)
            {
                Debug.LogWarning($"  ⚠ {target}: {e.Message}");
            }
        }
        
        Debug.Log("✅ Correction terminée ! Rechargez les scripts (Assets > Reimport All)");
        AssetDatabase.Refresh();
    }
    
    [MenuItem("Tools/Show Current Defines")]
    public static void ShowDefines()
    {
        Debug.Log("=== SCRIPTING DEFINE SYMBOLS ACTUELS ===");
        
        BuildTargetGroup[] targets = new BuildTargetGroup[]
        {
            BuildTargetGroup.Standalone,
            BuildTargetGroup.WebGL,
            BuildTargetGroup.Android,
            BuildTargetGroup.iOS
        };

        foreach (var target in targets)
        {
            try
            {
#if UNITY_2023_1_OR_NEWER
                var namedTarget = NamedBuildTarget.FromBuildTargetGroup(target);
                string defines = PlayerSettings.GetScriptingDefineSymbols(namedTarget);
#else
                string defines = PlayerSettings.GetScriptingDefineSymbolsForGroup(target);
#endif
                
                if (string.IsNullOrEmpty(defines))
                {
                    Debug.Log($"{target}: (vide)");
                }
                else
                {
                    Debug.Log($"{target}: {defines}");
                }
            }
            catch { }
        }
        
        Debug.Log("==========================================");
    }
}


