using UnityEngine;
using UnityEngine.InputSystem;

/// <summary>
/// Affiche des informations de debug directement dans le build
/// Utile quand on n'a pas accès aux logs Unity
/// </summary>
public class DebugOverlay : MonoBehaviour
{
    [Header("Configuration")]
    [Tooltip("Activer l'affichage debug au démarrage")]
    public bool showOnStart = true;
    
    [Tooltip("Touche pour toggle l'affichage (Cmd+D sur Mac)")]
    public bool useCommandModifier = true;
    
    [Header("Style")]
    public int fontSize = 14;
    public Color textColor = Color.yellow;
    public Color backgroundColor = new Color(0, 0, 0, 0.7f);
    
    private bool isVisible;
    private Vector2 lastMousePosition;
    private string debugInfo = "";
    
    void Start()
    {
        isVisible = showOnStart;
    }
    
    void Update()
    {
        // Toggle avec Cmd+D (Mac) ou Ctrl+D (Windows)
        bool modifierPressed = false;
        
        if (Keyboard.current != null)
        {
            if (useCommandModifier)
            {
                // Mac: Command (Left ou Right)
                modifierPressed = Keyboard.current.leftCommandKey.isPressed || 
                                 Keyboard.current.rightCommandKey.isPressed ||
                                 Keyboard.current.leftCtrlKey.isPressed;
            }
            else
            {
                modifierPressed = true;
            }
            
            if (modifierPressed && Keyboard.current.dKey.wasPressedThisFrame)
            {
                isVisible = !isVisible;
                Debug.Log($"[DebugOverlay] Affichage: {(isVisible ? "ON" : "OFF")}");
            }
            
            // Cmd+F pour toggle fullscreen
            if ((Keyboard.current.leftCommandKey.isPressed || Keyboard.current.rightCommandKey.isPressed || 
                 Keyboard.current.leftCtrlKey.isPressed) && 
                Keyboard.current.fKey.wasPressedThisFrame)
            {
                Screen.fullScreen = !Screen.fullScreen;
                Debug.Log($"[DebugOverlay] Fullscreen: {Screen.fullScreen}");
            }
        }
        
        // Mettre à jour la position de la souris
        if (Mouse.current != null)
        {
            lastMousePosition = Mouse.current.position.ReadValue();
        }
        
        // Construire les infos debug
        UpdateDebugInfo();
    }
    
    void UpdateDebugInfo()
    {
        debugInfo = "=== DEBUG BUILD ===\n";
        debugInfo += $"Résolution: {Screen.width}x{Screen.height}\n";
        debugInfo += $"Ratio: {(float)Screen.width / Screen.height:F3}\n";
        debugInfo += $"Fullscreen: {Screen.fullScreen}\n";
        debugInfo += $"Platform: {Application.platform}\n";
        debugInfo += $"\n";
        debugInfo += $"Souris: ({lastMousePosition.x:F0}, {lastMousePosition.y:F0})\n";
        debugInfo += $"\n";
        debugInfo += $"Touches:\n";
        debugInfo += $"Cmd+D = Toggle debug\n";
        debugInfo += $"Cmd+F = Toggle fullscreen\n";
        debugInfo += $"ESC = Retour menu/Quitter\n";
        
        // Ajouter info sur les zones si on est dans un jeu
        var gameManager = FindFirstObjectByType<GameManager>();
        if (gameManager != null)
        {
            debugInfo += $"\n=== GAME INFO ===\n";
            // On pourrait ajouter plus d'infos ici
        }
    }
    
    void OnGUI()
    {
        if (!isVisible) return;
        
        // Style pour le texte
        GUIStyle textStyle = new GUIStyle(GUI.skin.label);
        textStyle.fontSize = fontSize;
        textStyle.normal.textColor = textColor;
        textStyle.padding = new RectOffset(10, 10, 10, 10);
        
        // Calculer la taille du panneau
        GUIContent content = new GUIContent(debugInfo);
        Vector2 size = textStyle.CalcSize(content);
        size.x += 20;
        size.y += 20;
        
        // Dessiner le fond
        Rect bgRect = new Rect(10, 10, size.x, size.y);
        GUI.color = backgroundColor;
        GUI.DrawTexture(bgRect, Texture2D.whiteTexture);
        
        // Dessiner le texte
        GUI.color = Color.white;
        GUI.Label(bgRect, debugInfo, textStyle);
    }
}

