using System.Collections;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;

/// <summary>
/// Carte de jeu simplifiée pour la mosaïque du menu
/// Affiche uniquement le titre du jeu et lance le jeu au clic via l'API
/// </summary>
public class GameCard : MonoBehaviour, IPointerEnterHandler, IPointerExitHandler, IPointerClickHandler
{
    [Header("UI Components")]
    public TextMeshProUGUI titleText;
    public TextMeshProUGUI typeText;
    public Image backgroundImage;
    public Button cardButton;

    [Header("Visual Effects")]
    public float hoverScale = 1.05f;
    public float animationDuration = 0.2f;

    [Header("Colors by Type")]
    public Color shootingColor = new Color(0.39f, 0.28f, 0.5f, 1f); // Violet
    public Color textHoleColor = new Color(0.8f, 0.58f, 0.26f, 1f); // Doré
    public Color calculatorColor = new Color(0.33f, 0.61f, 0.36f, 1f); // Vert
    public Color defaultColor = new Color(0.5f, 0.5f, 0.5f, 1f); // Gris

    // Données
    private GameEntry gameEntry;
    private Vector3 originalScale;
    private CanvasGroup canvasGroup;

    void Awake()
    {
        // Ne rien créer ici - MainMenuManager s'en charge
    }
    
    void Start()
    {
        // Initialiser la scale après que le RectTransform soit configuré
        if (originalScale == Vector3.zero)
            originalScale = transform.localScale;
        
        if (originalScale == Vector3.zero)
            originalScale = Vector3.one;
            
        // Récupérer les composants existants
        if (canvasGroup == null)
            canvasGroup = GetComponent<CanvasGroup>();
            
        if (cardButton == null)
            cardButton = GetComponent<Button>();
    }

    /// <summary>
    /// Initialise la carte avec les données du jeu
    /// </summary>
    public void Initialize(GameEntry entry)
    {
        gameEntry = entry;

        if (gameEntry == null)
        {
            Debug.LogError("[GameCard] Données de jeu nulles !");
            return;
        }

        SetupCardVisuals();
        SetupCardInteraction();

        Debug.Log($"[GameCard] Carte initialisée : {gameEntry.title} (ID: {gameEntry.id})");
    }

    /// <summary>
    /// Configure l'apparence visuelle de la carte
    /// </summary>
    void SetupCardVisuals()
    {
        // Titre du jeu
        if (titleText != null)
        {
            titleText.text = gameEntry.title;
            titleText.color = Color.white;
        }

        // Type du jeu
        if (typeText != null)
        {
            typeText.text = gameEntry.GetDisplayType();
            typeText.color = new Color(1f, 1f, 1f, 0.7f);
        }

        // Couleur de fond selon le type
        if (backgroundImage != null)
        {
            backgroundImage.color = GetTypeColor(gameEntry.type);
        }
    }

    /// <summary>
    /// Configure l'interaction de la carte
    /// </summary>
    void SetupCardInteraction()
    {
        if (cardButton != null)
        {
            cardButton.interactable = true;
            cardButton.onClick.RemoveAllListeners();
            cardButton.onClick.AddListener(OnCardClicked);
        }
    }

    /// <summary>
    /// Retourne la couleur associée au type de jeu
    /// </summary>
    Color GetTypeColor(string type)
    {
        switch (type?.ToLower())
        {
            case "shooting": return shootingColor;
            case "text_hole": return textHoleColor;
            case "calculator": return calculatorColor;
            default: return defaultColor;
        }
    }

    /// <summary>
    /// Appelé quand la carte est cliquée
    /// </summary>
    void OnCardClicked()
    {
        if (gameEntry == null)
        {
            Debug.LogError("[GameCard] Données de jeu manquantes !");
            return;
        }

        Debug.Log($"[GameCard] 🎮 Clic sur le jeu: {gameEntry.title} (ID: {gameEntry.id})");
        StartCoroutine(ClickAnimation());
        LaunchGame();
    }

    /// <summary>
    /// Lance le jeu via l'API
    /// </summary>
    void LaunchGame()
    {
        Debug.Log($"[GameCard] 🚀 Lancement du jeu ID: {gameEntry.id}");

        // Définir les paramètres dans PlayerPrefs
        string gameType = gameEntry.GetNormalizedType();
        PlayerPrefs.SetString("CurrentLevelType", gameType);
        PlayerPrefs.SetString("CurrentLevelId", $"game_{gameEntry.id}");
        PlayerPrefs.SetInt("CurrentGameId", gameEntry.id);
        PlayerPrefs.SetString("GamePhase", "Before");
        
        // IMPORTANT : Définir la scène de retour au Menu (on vient du menu)
        PlayerPrefs.SetString("ReturnToScene", "Menu");
        Debug.Log($"[GameCard] ✅ ReturnToScene défini à 'Menu'");
        
        PlayerPrefs.Save();

        Debug.Log($"[GameCard] Type de jeu: {gameType}");
        Debug.Log($"[GameCard] GameId: {gameEntry.id}");

        // Charger les données du jeu via l'API puis lancer
        StartCoroutine(LoadGameDataAndLaunch());
    }

    /// <summary>
    /// Charge les données du jeu depuis l'API puis lance la scène
    /// </summary>
    private IEnumerator LoadGameDataAndLaunch()
    {
        // Afficher l'écran de chargement
        if (UnifiedLoadingManager.Instance != null)
        {
            UnifiedLoadingManager.ShowLoading($"Chargement de {gameEntry.title}...", LoadingContext.Game);
        }

        bool loadingComplete = false;
        bool loadingSuccess = false;

        // Charger les données du jeu via GameDataManager
        GameDataManager.Instance.LoadGameData(
            gameEntry.id,
            "Débutant", // TODO: Récupérer depuis les paramètres utilisateur
            onSuccess: (gameData) => {
                Debug.Log($"[GameCard] ✅ Données du jeu {gameEntry.id} chargées avec succès");
                loadingComplete = true;
                loadingSuccess = true;
            },
            onError: (error) => {
                Debug.LogError($"[GameCard] ❌ Erreur lors du chargement du jeu {gameEntry.id}: {error}");
                loadingComplete = true;
                loadingSuccess = false;
            }
        );

        // Attendre la fin du chargement
        float timeout = 10f;
        float elapsed = 0f;
        while (!loadingComplete && elapsed < timeout)
        {
            elapsed += Time.deltaTime;
            yield return null;
        }

        if (!loadingComplete)
        {
            Debug.LogError($"[GameCard] ⏰ Timeout lors du chargement du jeu {gameEntry.id}");
        }

        // Définir le flag pour indiquer qu'on utilise les données API
        if (loadingSuccess)
        {
            PlayerPrefs.SetString("UseApiData", "true");
            PlayerPrefs.Save();
            Debug.Log($"[GameCard] ✅ UseApiData=true défini, transition vers la scène Player...");
        }
        else
        {
            PlayerPrefs.SetString("UseApiData", "false");
            PlayerPrefs.Save();
            Debug.LogWarning($"[GameCard] ⚠️ Lancement malgré l'erreur de chargement API (UseApiData=false)");
        }

        UnityEngine.SceneManagement.SceneManager.LoadScene("Player");
    }

    // ========================================
    // ANIMATIONS
    // ========================================

    IEnumerator ClickAnimation()
    {
        Vector3 smallScale = originalScale * 0.95f;
        yield return StartCoroutine(AnimateScale(smallScale, animationDuration * 0.5f));
        yield return StartCoroutine(AnimateScale(originalScale, animationDuration * 0.5f));
    }

    IEnumerator AnimateScale(Vector3 targetScale, float duration)
    {
        Vector3 startScale = transform.localScale;
        float elapsed = 0f;

        while (elapsed < duration)
        {
            elapsed += Time.deltaTime;
            float t = Mathf.SmoothStep(0f, 1f, elapsed / duration);
            transform.localScale = Vector3.Lerp(startScale, targetScale, t);
            yield return null;
        }

        transform.localScale = targetScale;
    }

    // ========================================
    // POINTER EVENTS
    // ========================================

    public void OnPointerEnter(PointerEventData eventData)
    {
        Vector3 hoverScaleVector = originalScale * hoverScale;
        StartCoroutine(AnimateScale(hoverScaleVector, animationDuration));
        if (canvasGroup != null) canvasGroup.alpha = 1f;
    }

    public void OnPointerExit(PointerEventData eventData)
    {
        StartCoroutine(AnimateScale(originalScale, animationDuration));
    }

    public void OnPointerClick(PointerEventData eventData)
    {
        // Géré par OnCardClicked via le Button component
    }

    // ========================================
    // MÉTHODES PUBLIQUES
    // ========================================

    public GameEntry GetGameEntry() => gameEntry;

    void OnDestroy()
    {
        StopAllCoroutines();
    }
}

