using UnityEngine;
using UnityEngine.UI;
using System.Collections;

/// <summary>
/// Overlay de transition persistant créé par IntroScreenManager
/// Reste visible jusqu'à ce que la scène cible appelle Hide() ou après un timeout
/// </summary>
public class IntroTransitionOverlay : MonoBehaviour
{
    private static IntroTransitionOverlay instance;
    public static IntroTransitionOverlay Instance => instance;
    
    [Header("Configuration")]
    [Tooltip("Délai maximum avant auto-destruction (sécurité)")]
    public float maxWaitTime = 3f;
    
    [Tooltip("Durée du fade out")]
    public float fadeOutDuration = 0.5f;
    
    private CanvasGroup canvasGroup;
    private bool isHiding = false;
    
    void Awake()
    {
        // Singleton - détruire les doublons
        if (instance != null && instance != this)
        {
            Destroy(gameObject);
            return;
        }
        
        instance = this;
        
        // S'assurer qu'on a un CanvasGroup pour le fade
        Canvas canvas = GetComponent<Canvas>();
        if (canvas != null)
        {
            canvasGroup = canvas.GetComponent<CanvasGroup>();
            if (canvasGroup == null)
            {
                canvasGroup = canvas.gameObject.AddComponent<CanvasGroup>();
            }
        }
        
        // Lancer le timeout de sécurité
        StartCoroutine(AutoHideAfterTimeout());
    }
    
    /// <summary>
    /// Timeout de sécurité - masque l'overlay après maxWaitTime secondes
    /// </summary>
    IEnumerator AutoHideAfterTimeout()
    {
        yield return new WaitForSeconds(maxWaitTime);
        
        if (!isHiding && gameObject != null)
        {
            Debug.LogWarning($"[IntroTransitionOverlay] ⚠️ Timeout atteint ({maxWaitTime}s) - Auto-destruction");
            Hide();
        }
    }
    
    /// <summary>
    /// Masque l'overlay avec un fade out puis le détruit
    /// Appelé par la scène cible quand elle est prête
    /// </summary>
    public void Hide()
    {
        if (isHiding) return;
        isHiding = true;
        
        Debug.Log("[IntroTransitionOverlay] 🎬 Masquage de l'overlay de transition");
        StartCoroutine(FadeOutAndDestroy());
    }
    
    /// <summary>
    /// Masque l'overlay immédiatement sans fade
    /// </summary>
    public void HideImmediate()
    {
        Debug.Log("[IntroTransitionOverlay] 🎬 Masquage immédiat de l'overlay");
        
        if (instance == this)
        {
            instance = null;
        }
        
        Destroy(gameObject);
    }
    
    IEnumerator FadeOutAndDestroy()
    {
        if (canvasGroup != null)
        {
            float elapsed = 0f;
            float startAlpha = canvasGroup.alpha;
            
            while (elapsed < fadeOutDuration)
            {
                elapsed += Time.deltaTime;
                canvasGroup.alpha = Mathf.Lerp(startAlpha, 0f, elapsed / fadeOutDuration);
                yield return null;
            }
            
            canvasGroup.alpha = 0f;
        }
        
        if (instance == this)
        {
            instance = null;
        }
        
        Destroy(gameObject);
        Debug.Log("[IntroTransitionOverlay] ✅ Overlay de transition détruit");
    }
    
    /// <summary>
    /// Méthode statique pour masquer l'overlay depuis n'importe où
    /// </summary>
    public static void HideOverlay()
    {
        if (instance != null)
        {
            instance.Hide();
        }
    }
    
    /// <summary>
    /// Méthode statique pour masquer l'overlay immédiatement
    /// </summary>
    public static void HideOverlayImmediate()
    {
        if (instance != null)
        {
            instance.HideImmediate();
        }
    }
    
    /// <summary>
    /// Vérifie si l'overlay est actuellement affiché
    /// </summary>
    public static bool IsOverlayVisible()
    {
        return instance != null && !instance.isHiding;
    }
    
    void OnDestroy()
    {
        if (instance == this)
        {
            instance = null;
        }
    }
}

