using UnityEngine;
using UnityEngine.UI;
using UnityEngine.EventSystems;
using System.Collections;

/// <summary>
/// Composant pour gérer les zones cliquables des boutons de la calculatrice
/// Compatible avec le système de coordonnées de votre JSON
/// Utilise EventTrigger pour une meilleure compatibilité WebGL
/// </summary>
public class CalculatorButtonZone : MonoBehaviour, IPointerClickHandler
{
    [Header("Debug")]
    [SerializeField] private bool showDebugInfo = false;
    
    private CalculatorButtonData buttonConfig;
    private CalculatorGameManager gameManager;
    private RectTransform rectTransform;
    private Image debugImage;
    
    /// <summary>
    /// Initialise la zone de bouton avec sa configuration
    /// </summary>
    /// <param name="config">Configuration du bouton depuis le JSON</param>
    /// <param name="manager">Référence au GameManager principal</param>
    public void Initialize(CalculatorButtonData config, CalculatorGameManager manager)
    {
        buttonConfig = config;
        gameManager = manager;
        rectTransform = GetComponent<RectTransform>();
        debugImage = GetComponent<Image>();
        
        // S'assurer que l'image peut recevoir des événements de clic (important pour WebGL)
        if (debugImage != null)
        {
            debugImage.raycastTarget = true;
        }
        
        if (showDebugInfo)
        {
            Debug.Log($"Bouton initialisé: {config.buttonId} ({config.buttonType}) - Position: {config.position}, Taille: {config.size}");
        }
    }
    
    /// <summary>
    /// Vérifie si un point local est à l'intérieur de cette zone de bouton
    /// </summary>
    /// <param name="localPoint">Point en coordonnées locales du conteneur calculatrice</param>
    /// <returns>True si le point est dans la zone</returns>
    public bool IsPointInside(Vector2 localPoint)
    {
        if (rectTransform == null) return false;
        
        // Obtenir les limites du bouton en coordonnées locales
        Rect buttonRect = new Rect(
            rectTransform.anchoredPosition.x - rectTransform.sizeDelta.x * 0.5f,
            rectTransform.anchoredPosition.y - rectTransform.sizeDelta.y * 0.5f,
            rectTransform.sizeDelta.x,
            rectTransform.sizeDelta.y
        );
        
        bool isInside = buttonRect.Contains(localPoint);
        
        if (showDebugInfo && isInside)
        {
            Debug.Log($"Clic détecté sur bouton {buttonConfig.buttonId} - Point: {localPoint}, Rect: {buttonRect}");
        }
        
        return isInside;
    }
    
    /// <summary>
    /// Méthode alternative utilisant RectTransformUtility (plus précise)
    /// </summary>
    /// <param name="screenPoint">Point en coordonnées écran</param>
    /// <param name="camera">Caméra utilisée pour la conversion</param>
    /// <returns>True si le point écran est dans cette zone</returns>
    public bool IsScreenPointInside(Vector2 screenPoint, Camera camera = null)
    {
        if (rectTransform == null) return false;
        
        if (camera == null)
            camera = Camera.main;
        
        return RectTransformUtility.RectangleContainsScreenPoint(rectTransform, screenPoint, camera);
    }
    
    /// <summary>
    /// Interface IPointerClickHandler - Appelée automatiquement par Unity quand on clique (meilleur pour WebGL)
    /// </summary>
    public void OnPointerClick(PointerEventData eventData)
    {
        if (showDebugInfo)
        {
            Debug.Log($"[WebGL] OnPointerClick reçu pour: {buttonConfig?.buttonId}");
        }
        OnClick();
    }
    
    /// <summary>
    /// Appelée quand ce bouton est cliqué
    /// </summary>
    public void OnClick()
    {
        if (buttonConfig == null || gameManager == null)
        {
            Debug.LogError("CalculatorButtonZone: Configuration ou GameManager manquant!");
            return;
        }
        
        // Effet visuel de clic
        StartCoroutine(ClickVisualEffect());
        
        // Notifier le GameManager
        gameManager.OnCalculatorButtonPressed(buttonConfig);
        
        if (showDebugInfo)
        {
            Debug.Log($"Bouton cliqué: {buttonConfig.buttonId} (Valeur: {buttonConfig.value})");
        }
    }
    
    /// <summary>
    /// Effet visuel rapide quand le bouton est cliqué
    /// </summary>
    /// <returns></returns>
    IEnumerator ClickVisualEffect()
    {
        // Effet sur l'image debug si présente
        if (debugImage != null)
        {
            Color originalColor = debugImage.color;
            debugImage.color = Color.white;
            
            yield return new WaitForSeconds(0.1f);
            
            debugImage.color = originalColor;
        }
        
        // Effet de scale alternatif
        if (rectTransform != null)
        {
            Vector3 originalScale = rectTransform.localScale;
            
            // Scale down puis up rapidement
            rectTransform.localScale = originalScale * 0.95f;
            yield return new WaitForSeconds(0.05f);
            rectTransform.localScale = originalScale;
        }
    }
    
    /// <summary>
    /// Active ou désactive l'affichage debug de cette zone
    /// </summary>
    /// <param name="show">True pour afficher la zone en semi-transparent</param>
    public void SetDebugVisibility(bool show)
    {
        showDebugInfo = show;
        
        if (debugImage != null)
        {
            if (show)
            {
                // Couleur selon le type de bouton
                Color debugColor = GetDebugColorForButtonType();
                debugImage.color = debugColor;
                debugImage.enabled = true;
            }
            else
            {
                debugImage.enabled = false;
            }
        }
    }
    
    /// <summary>
    /// Retourne une couleur de debug selon le type de bouton
    /// </summary>
    /// <returns></returns>
    Color GetDebugColorForButtonType()
    {
        if (buttonConfig == null) return Color.gray;
        
        switch (buttonConfig.buttonType)
        {
            case "number":
                return new Color(0, 1, 0, 0.3f); // Vert pour les chiffres
            case "delete":
                return new Color(1, 1, 0, 0.3f); // Jaune pour delete
            case "validate":
                return new Color(0, 0, 1, 0.3f); // Bleu pour validate
            default:
                return new Color(1, 0, 1, 0.3f); // Magenta pour inconnu
        }
    }
    
    /// <summary>
    /// Méthode debug pour afficher les informations de ce bouton
    /// </summary>
    public void LogButtonInfo()
    {
        if (buttonConfig == null)
        {
            Debug.Log($"CalculatorButtonZone sur {gameObject.name}: Pas de configuration");
            return;
        }
        
        Debug.Log($"=== BOUTON {buttonConfig.buttonId} ===");
        Debug.Log($"Type: {buttonConfig.buttonType}");
        Debug.Log($"Valeur: {buttonConfig.value}");
        Debug.Log($"Position: {buttonConfig.position}");
        Debug.Log($"Taille: {buttonConfig.size}");
        
        if (rectTransform != null)
        {
            Debug.Log($"RectTransform Position: {rectTransform.anchoredPosition}");
            Debug.Log($"RectTransform Taille: {rectTransform.sizeDelta}");
            
            Rect bounds = new Rect(
                rectTransform.anchoredPosition.x - rectTransform.sizeDelta.x * 0.5f,
                rectTransform.anchoredPosition.y - rectTransform.sizeDelta.y * 0.5f,
                rectTransform.sizeDelta.x,
                rectTransform.sizeDelta.y
            );
            Debug.Log($"Zone cliquable: {bounds}");
        }
        
        Debug.Log("========================");
    }
    
    /// <summary>
    /// Retourne les informations du bouton pour debug
    /// </summary>
    /// <returns></returns>
    public string GetButtonInfo()
    {
        if (buttonConfig == null) return "Pas de config";
        
        return $"{buttonConfig.buttonId} ({buttonConfig.buttonType}): {buttonConfig.value}";
    }
    
    /// <summary>
    /// Méthode appelée par Unity quand l'objet est activé
    /// </summary>
    void OnEnable()
    {
        // S'assurer que l'image debug est configurée correctement
        if (debugImage != null && showDebugInfo)
        {
            SetDebugVisibility(true);
        }
    }
    
    /// <summary>
    /// Debug visuel dans l'éditeur Unity
    /// </summary>
    void OnDrawGizmos()
    {
        #if UNITY_EDITOR
        if (rectTransform != null && showDebugInfo)
        {
            // Dessiner les limites du bouton dans la vue Scene
            Gizmos.color = GetDebugColorForButtonType();
            
            Vector3 center = rectTransform.position;
            Vector3 size = new Vector3(rectTransform.sizeDelta.x, rectTransform.sizeDelta.y, 0);
            
            // Prendre en compte le scale du transform parent
            Transform parent = rectTransform.parent;
            if (parent != null)
            {
                size.Scale(parent.lossyScale);
            }
            
            Gizmos.DrawWireCube(center, size);
        }
        #endif
    }
    
    // Méthodes publiques pour l'intégration avec votre architecture existante
    
    /// <summary>
    /// Getter pour la configuration du bouton
    /// </summary>
    public CalculatorButtonData GetButtonConfig()
    {
        return buttonConfig;
    }
    
    /// <summary>
    /// Vérifie si ce bouton est de type numérique
    /// </summary>
    public bool IsNumericButton()
    {
        return buttonConfig != null && buttonConfig.buttonType == "number";
    }
    
    /// <summary>
    /// Vérifie si ce bouton est de type action (delete/validate)
    /// </summary>
    public bool IsActionButton()
    {
        return buttonConfig != null && 
               (buttonConfig.buttonType == "delete" || buttonConfig.buttonType == "validate");
    }
}