using UnityEngine;
using UnityEngine.UI;
using UnityEngine.Video;
using UnityEngine.InputSystem;
using System.Collections;

public class MapFullscreenVideo : MonoBehaviour
{
    private VideoPlayer videoPlayer;
    private RenderTexture renderTexture;
    private RawImage display;
    private CanvasGroup canvasGroup;
    private MapStep stepData;
    private System.Action onComplete;
    private bool canSkip = false;
    private bool hasCompleted = false;

    public void Initialize(MapStep step, System.Action onVideoComplete)
    {
        stepData = step;
        onComplete = onVideoComplete;

        canvasGroup = gameObject.AddComponent<CanvasGroup>();
        canvasGroup.alpha = 0f;

        CreateVideoPlayer();
        StartCoroutine(FadeInAndPlay());
    }

    void CreateVideoPlayer()
    {
        RectTransform rectTransform = GetComponent<RectTransform>();
        rectTransform.anchorMin = Vector2.zero;
        rectTransform.anchorMax = Vector2.one;
        rectTransform.offsetMin = Vector2.zero;
        rectTransform.offsetMax = Vector2.zero;

        GameObject bgObj = new GameObject("BlackBackground");
        bgObj.transform.SetParent(transform, false);
        RectTransform bgRect = bgObj.AddComponent<RectTransform>();
        bgRect.anchorMin = Vector2.zero;
        bgRect.anchorMax = Vector2.one;
        bgRect.offsetMin = Vector2.zero;
        bgRect.offsetMax = Vector2.zero;
        Image bgImage = bgObj.AddComponent<Image>();
        bgImage.color = Color.black;
        bgImage.raycastTarget = false;  // Désactiver les raycast sur le fond noir

        renderTexture = new RenderTexture(1920, 1080, 0);
        renderTexture.Create();

        GameObject videoPlayerObj = new GameObject("VideoPlayer");
        videoPlayerObj.transform.SetParent(transform, false);

        videoPlayer = videoPlayerObj.AddComponent<VideoPlayer>();
        videoPlayer.playOnAwake = false;
        videoPlayer.renderMode = VideoRenderMode.RenderTexture;
        videoPlayer.targetTexture = renderTexture;
        videoPlayer.isLooping = false;
        videoPlayer.aspectRatio = VideoAspectRatio.FitInside;
        
        // Configuration audio AVANT de définir l'URL (important!)
        videoPlayer.audioOutputMode = VideoAudioOutputMode.AudioSource;
        videoPlayer.controlledAudioTrackCount = 1;  // Activer le contrôle de la piste audio
        
        AudioSource audioSource = videoPlayerObj.AddComponent<AudioSource>();
        audioSource.playOnAwake = false;
        audioSource.volume = 1.0f;
        audioSource.spatialBlend = 0f;  // 2D audio (pas de spatialisation 3D)
        audioSource.priority = 0;  // Priorité maximale
        audioSource.bypassEffects = false;
        audioSource.bypassListenerEffects = false;
        audioSource.bypassReverbZones = true;
        
        videoPlayer.EnableAudioTrack(0, true);  // Activer explicitement la piste audio 0
        videoPlayer.SetTargetAudioSource(0, audioSource);
        
        // Définir l'URL APRÈS la configuration audio
        videoPlayer.url = stepData.videoUrl;

        GameObject displayObj = new GameObject("VideoDisplay");
        displayObj.transform.SetParent(transform, false);

        RectTransform displayRect = displayObj.AddComponent<RectTransform>();
        displayRect.anchorMin = Vector2.zero;
        displayRect.anchorMax = Vector2.one;
        displayRect.offsetMin = Vector2.zero;
        displayRect.offsetMax = Vector2.zero;

        display = displayObj.AddComponent<RawImage>();
        display.texture = renderTexture;
        display.color = Color.white;
        display.raycastTarget = false;  // IMPORTANT: Désactiver les raycast pour que les clics ne soient pas détectés sur la vidéo

        videoPlayer.loopPointReached += OnVideoFinished;
        videoPlayer.errorReceived += OnVideoError;
        videoPlayer.prepareCompleted += OnVideoPrepared;

        if (stepData.skipable)
        {
            CreateSkipButton();
        }

        videoPlayer.Prepare();
    }

    void OnVideoPrepared(VideoPlayer vp)
    {
        // Garder seulement un log minimal pour confirmer que la vidéo est prête
        if (vp.audioTrackCount == 0)
        {
            Debug.LogWarning("[MapFullscreenVideo] ⚠️ Cette vidéo n'a AUCUNE piste audio !");
        }
    }

    void OnVideoError(VideoPlayer vp, string message)
    {
        Debug.LogError("========================================");
        Debug.LogError("❌❌❌ VIDEO ERROR ❌❌❌");
        Debug.LogError("========================================");
        Debug.LogError($"ERROR MESSAGE: {message}");
        Debug.LogError($"URL TENTEE: {stepData.videoUrl}");
        Debug.LogError($"VideoPlayer.url: {vp.url}");
        Debug.LogError($"VideoPlayer.source: {vp.source}");
        Debug.LogError($"VideoPlayer.isPrepared: {vp.isPrepared}");
        Debug.LogError($"VideoPlayer.isPlaying: {vp.isPlaying}");
        Debug.LogError("========================================");
        
        // Si erreur, passer au step suivant quand même
        if (!hasCompleted)
        {
            hasCompleted = true;
            Debug.LogWarning("[MapFullscreenVideo] ⚠️ SKIPPING VIDEO - Passage au step suivant");
            StartCoroutine(FadeOutAndFinish());
        }
    }

    void CreateSkipButton()
    {
        GameObject skipObj = new GameObject("SkipButton");
        skipObj.transform.SetParent(transform, false);

        RectTransform skipRect = skipObj.AddComponent<RectTransform>();
        skipRect.anchorMin = new Vector2(1f, 0f);
        skipRect.anchorMax = new Vector2(1f, 0f);
        skipRect.pivot = new Vector2(1f, 0f);
        skipRect.sizeDelta = new Vector2(200f, 60f);
        skipRect.anchoredPosition = new Vector2(-30f, 30f);

        Image skipBg = skipObj.AddComponent<Image>();
        skipBg.color = new Color(0f, 0f, 0f, 0.7f);

        Button skipButton = skipObj.AddComponent<Button>();
        skipButton.targetGraphic = skipBg;
        skipButton.onClick.AddListener(SkipVideo);

        GameObject textObj = new GameObject("Text");
        textObj.transform.SetParent(skipObj.transform, false);

        RectTransform textRect = textObj.AddComponent<RectTransform>();
        textRect.anchorMin = Vector2.zero;
        textRect.anchorMax = Vector2.one;
        textRect.offsetMin = Vector2.zero;
        textRect.offsetMax = Vector2.zero;

        TMPro.TextMeshProUGUI skipText = textObj.AddComponent<TMPro.TextMeshProUGUI>();
        skipText.text = "PASSER";
        skipText.fontSize = 20;
        skipText.alignment = TMPro.TextAlignmentOptions.Center;
        skipText.color = Color.white;
    }

    IEnumerator FadeInAndPlay()
    {
        float timeout = 10f;
        float elapsed = 0f;

        while (!videoPlayer.isPrepared && elapsed < timeout)
        {
            elapsed += Time.deltaTime;
            yield return null;
        }

        if (!videoPlayer.isPrepared)
        {
            Debug.LogError("[MapFullscreenVideo] Video preparation timeout!");
            if (!hasCompleted)
            {
                hasCompleted = true;
                StartCoroutine(FadeOutAndFinish());
            }
            yield break;
        }

        // Vérification AudioListener
        AudioListener listener = FindFirstObjectByType<AudioListener>();
        if (listener == null)
        {
            Debug.LogError("[MapFullscreenVideo] ❌ AUCUN AudioListener trouvé dans la scène !");
        }
        
        videoPlayer.Play();

        float fadeElapsed = 0f;
        float fadeTime = 0.5f;

        while (fadeElapsed < fadeTime)
        {
            fadeElapsed += Time.deltaTime;
            canvasGroup.alpha = Mathf.Lerp(0f, 1f, fadeElapsed / fadeTime);
            yield return null;
        }

        canvasGroup.alpha = 1f;
        canSkip = true;
    }

    void Update()
    {
        if (canSkip && stepData.skipable)
        {
            // Permettre le skip avec la touche Escape seulement
            if (Keyboard.current != null && Keyboard.current.escapeKey.wasPressedThisFrame)
            {
                SkipVideo();
            }
            
            // Le skip par clic souris est maintenant géré uniquement par le bouton UI
            // On ne détecte plus les clics globaux sur la vidéo
            
            // WebGL fallback pour le bouton uniquement
            #if UNITY_WEBGL && !UNITY_EDITOR
            if (WebGLClickReceiver.WasClickedThisFrame())
            {
                // Note: En WebGL, le clic sur le bouton UI sera géré par le système UI normal
                // Cette détection est pour d'autres cas spécifiques si nécessaire
            }
            #endif
        }
    }

    void SkipVideo()
    {
        if (!canSkip || hasCompleted) return;

        canSkip = false;
        hasCompleted = true;
        StartCoroutine(FadeOutAndFinish());
    }
    
    /// <summary>
    /// Appelé par WebGLClickReceiver quand un clic est reçu depuis JavaScript
    /// </summary>
    public void OnWebGLClick()
    {
        if (canSkip && stepData.skipable)
        {
            #if UNITY_WEBGL && !UNITY_EDITOR
            Application.ExternalCall("console.log", "[Unity] MapFullscreenVideo.OnWebGLClick: Skip vidéo");
            #endif
            SkipVideo();
        }
    }

    void OnVideoFinished(VideoPlayer vp)
    {
        if (hasCompleted) return;

        if (stepData.autoAdvance)
        {
            hasCompleted = true;
            StartCoroutine(FadeOutAndFinish());
        }
    }

    IEnumerator FadeOutAndFinish()
    {
        onComplete?.Invoke();

        float elapsed = 0f;
        float fadeTime = 0.5f;

        while (elapsed < fadeTime)
        {
            elapsed += Time.deltaTime;
            canvasGroup.alpha = Mathf.Lerp(1f, 0f, elapsed / fadeTime);
            yield return null;
        }

        canvasGroup.alpha = 0f;

        if (videoPlayer != null && videoPlayer.isPlaying)
        {
            videoPlayer.Stop();
        }

        Destroy(gameObject);
    }

    void OnDestroy()
    {
        if (renderTexture != null)
        {
            renderTexture.Release();
            Destroy(renderTexture);
        }
    }
}