using UnityEngine;
using UnityEngine.SceneManagement;

/// <summary>
/// Utilitaire pour réinitialiser la progression d'une map et rejouer l'intro
/// Usage: Attacher à un bouton ou appeler ResetMapProgress.ResetMap("map-Q1")
/// </summary>
public class ResetMapProgress : MonoBehaviour
{
    [Header("Configuration")]
    [Tooltip("ID de la map à réinitialiser (ex: map-Q1)")]
    public string mapId = "map-Q1";
    
    [Header("Actions")]
    [Tooltip("Réinitialiser au Start ?")]
    public bool resetOnStart = false;
    
    [Tooltip("Réinitialiser et charger la map immédiatement ?")]
    public bool resetAndLoadMap = false;

    void Start()
    {
        if (resetOnStart)
        {
            ResetMap(mapId);
            
            if (resetAndLoadMap)
            {
                LoadMap(mapId);
            }
        }
    }

    /// <summary>
    /// Réinitialise complètement la progression d'une map
    /// </summary>
    public static void ResetMap(string mapId)
    {
        Debug.Log($"[ResetMapProgress] Réinitialisation de la map: {mapId}");
        
        // Supprimer toutes les clés liées à cette map
        PlayerPrefs.DeleteKey($"IntroWatched_{mapId}");
        PlayerPrefs.DeleteKey($"CurrentStepId_{mapId}");
        PlayerPrefs.DeleteKey($"NextStepId_{mapId}");
        PlayerPrefs.Save();
        
        Debug.Log($"[ResetMapProgress] ✅ Map {mapId} réinitialisée - L'intro se jouera au prochain chargement");
    }

    /// <summary>
    /// Réinitialise et charge immédiatement la map
    /// </summary>
    public static void ResetAndLoadMap(string mapId)
    {
        ResetMap(mapId);
        LoadMap(mapId);
    }

    /// <summary>
    /// Charge une map
    /// </summary>
    public static void LoadMap(string mapId)
    {
        PlayerPrefs.SetString("CurrentMapId", mapId);
        PlayerPrefs.Save();
        
        Debug.Log($"[ResetMapProgress] Chargement de la map: {mapId}");
        SceneManager.LoadScene("Map");
    }

    /// <summary>
    /// Méthode appelable depuis un bouton Unity
    /// </summary>
    public void ResetCurrentMap()
    {
        ResetMap(mapId);
    }

    /// <summary>
    /// Méthode appelable depuis un bouton Unity
    /// </summary>
    public void ResetAndLoadCurrentMap()
    {
        ResetAndLoadMap(mapId);
    }

    /// <summary>
    /// Réinitialise TOUTES les maps
    /// </summary>
    public static void ResetAllMaps()
    {
        Debug.Log("[ResetMapProgress] Réinitialisation de TOUTES les maps");
        
        // Liste des maps connues (à adapter selon votre projet)
        string[] knownMaps = { "map-Q0", "map-Q1", "map-Q2", "map-Q3", "map-Q4", "map-Q5", "map-Q6", "map-Q7", "map-Q8", "map-Q9" };
        
        foreach (string mapId in knownMaps)
        {
            PlayerPrefs.DeleteKey($"IntroWatched_{mapId}");
            PlayerPrefs.DeleteKey($"CurrentStepId_{mapId}");
            PlayerPrefs.DeleteKey($"NextStepId_{mapId}");
        }
        
        PlayerPrefs.Save();
        Debug.Log("[ResetMapProgress] ✅ Toutes les maps réinitialisées");
    }
}

