#if UNITY_EDITOR
using UnityEngine;
using UnityEditor;

/// <summary>
/// Script d'éditeur pour vérifier et corriger automatiquement les paramètres du nouveau Input System
/// </summary>
public class InputSystemSettingsChecker : EditorWindow
{
    [MenuItem("Tools/Input System/Check Settings")]
    public static void CheckSettings()
    {
        Debug.Log("═══════════════════════════════════════════════════");
        Debug.Log("🔍 VÉRIFICATION DES PARAMÈTRES INPUT SYSTEM");
        Debug.Log("═══════════════════════════════════════════════════");
        
        // Vérifier Active Input Handling (utiliser l'API obsolète avec pragma warning)
        #pragma warning disable CS0618
        var inputHandling = PlayerSettings.GetPropertyString("activeInputHandler");
        #pragma warning restore CS0618
        Debug.Log($"Active Input Handling: {inputHandling}");
        
        if (inputHandling != "2") // 2 = Input System Package (New)
        {
            Debug.LogWarning("⚠️ Active Input Handling n'est pas configuré sur 'Input System Package (New)'");
            Debug.LogWarning("   Cela peut causer des exceptions InvalidOperationException");
            Debug.LogWarning("   Solution: Edit → Project Settings → Player → Other Settings → Active Input Handling → 'Input System Package (New)'");
        }
        else
        {
            Debug.Log("✅ Active Input Handling est correctement configuré");
        }
        
        // Vérifier si le package Input System est installé
        #if ENABLE_INPUT_SYSTEM
        Debug.Log("✅ Package Input System est activé (ENABLE_INPUT_SYSTEM défini)");
        #else
        Debug.LogError("❌ Package Input System n'est PAS activé (ENABLE_INPUT_SYSTEM non défini)");
        Debug.LogError("   Solution: Installez le package Input System depuis Package Manager");
        #endif
        
        // Vérifier les EventSystem dans les scènes
        var eventSystems = FindObjectsByType<UnityEngine.EventSystems.EventSystem>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        Debug.Log($"\n📊 EventSystem trouvés: {eventSystems.Length}");
        
        foreach (var es in eventSystems)
        {
            var standaloneModule = es.GetComponent<UnityEngine.EventSystems.StandaloneInputModule>();
            var inputSystemModule = es.GetComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
            
            if (standaloneModule != null)
            {
                Debug.LogWarning($"⚠️ EventSystem '{es.gameObject.name}' a un StandaloneInputModule (à supprimer)");
            }
            
            if (inputSystemModule == null)
            {
                Debug.LogWarning($"⚠️ EventSystem '{es.gameObject.name}' n'a PAS d'InputSystemUIInputModule (à ajouter)");
            }
            else
            {
                Debug.Log($"✅ EventSystem '{es.gameObject.name}' a un InputSystemUIInputModule");
            }
        }
        
        Debug.Log("═══════════════════════════════════════════════════");
        Debug.Log("🔍 FIN DE LA VÉRIFICATION");
        Debug.Log("═══════════════════════════════════════════════════");
    }
    
    [MenuItem("Tools/Input System/Fix Settings")]
    public static void FixSettings()
    {
        Debug.Log("🔧 CORRECTION DES PARAMÈTRES INPUT SYSTEM");
        
        // Forcer Active Input Handling sur "Input System Package (New)" (utiliser l'API obsolète avec pragma warning)
        #pragma warning disable CS0618
        PlayerSettings.SetPropertyString("activeInputHandler", "2");
        #pragma warning restore CS0618
        Debug.Log("✅ Active Input Handling configuré sur 'Input System Package (New)'");
        
        // Corriger les EventSystem dans la scène active
        var eventSystems = FindObjectsByType<UnityEngine.EventSystems.EventSystem>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        
        foreach (var es in eventSystems)
        {
            // Supprimer StandaloneInputModule
            var standaloneModule = es.GetComponent<UnityEngine.EventSystems.StandaloneInputModule>();
            if (standaloneModule != null)
            {
                DestroyImmediate(standaloneModule);
                Debug.Log($"✅ StandaloneInputModule supprimé de '{es.gameObject.name}'");
            }
            
            // Ajouter InputSystemUIInputModule si absent
            var inputSystemModule = es.GetComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
            if (inputSystemModule == null)
            {
                es.gameObject.AddComponent<UnityEngine.InputSystem.UI.InputSystemUIInputModule>();
                Debug.Log($"✅ InputSystemUIInputModule ajouté à '{es.gameObject.name}'");
            }
        }
        
        Debug.Log("✅ Correction terminée ! N'oubliez pas de sauvegarder la scène.");
    }
}
#endif

