using UnityEngine;
using UnityEngine.UI;
using System.Collections;

/// <summary>
/// Script de correction automatique pour l'écran noir sur la map Mac
/// À attacher à un GameObject dans la scène Map
/// Détecte et corrige les problèmes de chargement qui bloquent l'écran
/// </summary>
public class FixMacBlackScreenMap : MonoBehaviour
{
    [Header("Configuration")]
    [Tooltip("Forcer la fermeture de l'écran de chargement après ce délai")]
    public float forceHideLoadingAfter = 5f;
    
    [Tooltip("Activer les logs verbeux")]
    public bool verboseLogs = true;
    
    [Header("Diagnostic")]
    [Tooltip("Appuyez sur cette touche pour relancer le diagnostic")]
    public KeyCode diagnosticKey = KeyCode.F12;
    
    private bool fixApplied = false;
    private float startTime;
    
    void Start()
    {
        startTime = Time.time;
        
        if (verboseLogs)
        {
            Debug.Log("═══════════════════════════════════════");
            Debug.Log("🍎 FIX MAC BLACK SCREEN - DÉMARRAGE");
            Debug.Log($"Platform: {Application.platform}");
            Debug.Log("═══════════════════════════════════════");
        }
        
        // Lancer le diagnostic et la correction en parallèle
        StartCoroutine(MonitorAndFixLoadingScreen());
        StartCoroutine(DiagnoseScene());
    }
    
    void Update()
    {
        // Touche F12 pour relancer le diagnostic
        #if ENABLE_INPUT_SYSTEM
        if (UnityEngine.InputSystem.Keyboard.current != null && 
            UnityEngine.InputSystem.Keyboard.current.f12Key.wasPressedThisFrame)
        {
            Debug.Log("🔄 Relancement du diagnostic (F12)...");
            StartCoroutine(DiagnoseScene());
        }
        #else
        if (Input.GetKeyDown(diagnosticKey))
        {
            Debug.Log("🔄 Relancement du diagnostic (F12)...");
            StartCoroutine(DiagnoseScene());
        }
        #endif
    }
    
    /// <summary>
    /// Surveille l'écran de chargement et force sa fermeture si nécessaire
    /// </summary>
    IEnumerator MonitorAndFixLoadingScreen()
    {
        yield return new WaitForSeconds(1f); // Laisser le temps à l'initialisation normale
        
        if (verboseLogs)
        {
            Debug.Log("[FixBlackScreen] 🔍 Surveillance de l'écran de chargement...");
        }
        
        float elapsed = 0f;
        bool loadingHidden = false;
        
        while (elapsed < forceHideLoadingAfter && !loadingHidden)
        {
            // Vérifier si l'écran de chargement est toujours visible
            loadingHidden = CheckLoadingScreenHidden();
            
            if (loadingHidden && verboseLogs)
            {
                Debug.Log($"[FixBlackScreen] ✅ Écran de chargement masqué après {elapsed:F1}s");
                break;
            }
            
            yield return new WaitForSeconds(0.5f);
            elapsed += 0.5f;
        }
        
        // Si l'écran de chargement est toujours visible après le délai, forcer la fermeture
        if (!loadingHidden)
        {
            Debug.LogWarning($"[FixBlackScreen] ⚠️ Écran de chargement toujours visible après {elapsed:F1}s");
            Debug.LogWarning("[FixBlackScreen] 🔧 Application du fix...");
            
            ApplyLoadingScreenFix();
            fixApplied = true;
        }
    }
    
    /// <summary>
    /// Vérifie si l'écran de chargement est masqué
    /// </summary>
    bool CheckLoadingScreenHidden()
    {
        // Chercher les Canvas de chargement
        Canvas[] allCanvases = FindObjectsByType<Canvas>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        
        foreach (Canvas canvas in allCanvases)
        {
            string canvasName = canvas.name.ToLower();
            
            // Détecter les Canvas de chargement
            if (canvasName.Contains("loading") || 
                canvasName.Contains("chargement") || 
                canvas.sortingOrder >= 1000) // Ordre de rendu typique pour loading screens
            {
                if (canvas.gameObject.activeInHierarchy)
                {
                    if (verboseLogs)
                    {
                        Debug.Log($"[FixBlackScreen] Canvas de chargement détecté actif: {canvas.name}");
                    }
                    return false; // Écran de chargement toujours visible
                }
            }
        }
        
        return true; // Aucun écran de chargement visible
    }
    
    /// <summary>
    /// Force la fermeture de l'écran de chargement
    /// </summary>
    void ApplyLoadingScreenFix()
    {
        int fixedCount = 0;
        
        // Méthode 1 : Via UnifiedLoadingManager
        if (UnifiedLoadingManager.Instance != null)
        {
            Debug.Log("[FixBlackScreen] 📍 Méthode 1: UnifiedLoadingManager.HideLoading()");
            UnifiedLoadingManager.HideLoading();
            fixedCount++;
        }
        else
        {
            Debug.LogWarning("[FixBlackScreen] ⚠️ UnifiedLoadingManager.Instance est NULL");
        }
        
        // Méthode 2 : Via LoadingScreenManager
        LoadingScreenManager loadingManager = FindFirstObjectByType<LoadingScreenManager>();
        if (loadingManager != null)
        {
            Debug.Log("[FixBlackScreen] 📍 Méthode 2: LoadingScreenManager.HideLoading()");
            loadingManager.HideLoading(immediate: true);
            fixedCount++;
        }
        else
        {
            Debug.LogWarning("[FixBlackScreen] ⚠️ LoadingScreenManager introuvable");
        }
        
        // Méthode 3 : Désactiver manuellement les Canvas de chargement
        Canvas[] allCanvases = FindObjectsByType<Canvas>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        
        foreach (Canvas canvas in allCanvases)
        {
            string canvasName = canvas.name.ToLower();
            
            if (canvasName.Contains("loading") || 
                canvasName.Contains("chargement") || 
                canvas.sortingOrder >= 1000)
            {
                if (canvas.gameObject.activeInHierarchy)
                {
                    Debug.Log($"[FixBlackScreen] 📍 Méthode 3: Désactivation manuelle de {canvas.name}");
                    canvas.gameObject.SetActive(false);
                    fixedCount++;
                }
            }
        }
        
        if (fixedCount > 0)
        {
            Debug.Log($"[FixBlackScreen] ✅ FIX APPLIQUÉ - {fixedCount} correction(s)");
        }
        else
        {
            Debug.LogError("[FixBlackScreen] ❌ Aucune correction possible - Vérifier manuellement");
        }
    }
    
    /// <summary>
    /// Diagnostic complet de la scène
    /// </summary>
    IEnumerator DiagnoseScene()
    {
        yield return new WaitForSeconds(0.5f); // Petit délai pour laisser la scène s'initialiser
        
        Debug.Log("═══════════════════════════════════════");
        Debug.Log("🔍 DIAGNOSTIC COMPLET - ÉCRAN NOIR MAC");
        Debug.Log($"Temps écoulé: {(Time.time - startTime):F1}s");
        Debug.Log("═══════════════════════════════════════");
        
        // 1. CAMÉRA
        Camera mainCam = Camera.main;
        if (mainCam == null)
        {
            Debug.LogError("❌ [1] Pas de caméra principale !");
        }
        else
        {
            Debug.Log($"✅ [1] Caméra: {mainCam.name}");
            Debug.Log($"    Clear Flags: {mainCam.clearFlags}");
            Debug.Log($"    Background: {mainCam.backgroundColor}");
            
            #if UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX
            // Sur Mac, vérifier si la caméra a un fond noir
            if (mainCam.backgroundColor == Color.black && mainCam.clearFlags == CameraClearFlags.SolidColor)
            {
                Debug.LogWarning("⚠️ [1] Caméra avec fond noir - Peut causer un écran noir si aucun contenu");
            }
            #endif
        }
        
        // 2. GENERAL CONFIG MANAGER
        if (GeneralConfigManager.Instance == null)
        {
            Debug.LogError("❌ [2] GeneralConfigManager.Instance NULL !");
        }
        else
        {
            bool configLoaded = GeneralConfigManager.Instance.IsConfigLoaded();
            Debug.Log($"{(configLoaded ? "✅" : "❌")} [2] GeneralConfigManager - Config chargée: {configLoaded}");
            
            if (!configLoaded)
            {
                Debug.LogError("❌ [2] Config non chargée - MapManager peut être bloqué");
            }
        }
        
        // 3. MAP MANAGER
        MapManager mapManager = FindFirstObjectByType<MapManager>();
        if (mapManager == null)
        {
            Debug.LogError("❌ [3] MapManager introuvable !");
        }
        else
        {
            Debug.Log($"✅ [3] MapManager: {mapManager.name}");
        }
        
        // 4. UNIFIED LOADING MANAGER
        if (UnifiedLoadingManager.Instance == null)
        {
            Debug.LogError("❌ [4] UnifiedLoadingManager.Instance NULL !");
            Debug.LogError("    → L'écran de chargement ne peut pas être masqué !");
        }
        else
        {
            Debug.Log("✅ [4] UnifiedLoadingManager présent");
        }
        
        // 5. LOADING SCREEN MANAGER
        LoadingScreenManager loadingManager = FindFirstObjectByType<LoadingScreenManager>();
        if (loadingManager == null)
        {
            Debug.LogWarning("⚠️ [5] LoadingScreenManager introuvable");
        }
        else
        {
            Debug.Log($"✅ [5] LoadingScreenManager: {loadingManager.name}");
            Debug.Log($"    Is Loading: {loadingManager.IsLoading}");
        }
        
        // 6. CANVAS (tous)
        Canvas[] allCanvases = FindObjectsByType<Canvas>(FindObjectsInactive.Include, FindObjectsSortMode.None);
        Debug.Log($"📊 [6] {allCanvases.Length} Canvas trouvé(s)");
        
        int activeCanvases = 0;
        int loadingCanvases = 0;
        
        foreach (Canvas canvas in allCanvases)
        {
            bool isActive = canvas.gameObject.activeInHierarchy;
            if (isActive) activeCanvases++;
            
            string canvasName = canvas.name.ToLower();
            bool isLoading = canvasName.Contains("loading") || canvasName.Contains("chargement");
            
            if (isLoading)
            {
                loadingCanvases++;
                Debug.Log($"    📍 Loading Canvas: {canvas.name}, Active: {isActive}, Order: {canvas.sortingOrder}");
            }
            else if (verboseLogs)
            {
                Debug.Log($"    Canvas: {canvas.name}, Active: {isActive}, Order: {canvas.sortingOrder}");
            }
        }
        
        Debug.Log($"    Actifs: {activeCanvases}/{allCanvases.Length}");
        Debug.Log($"    Loading: {loadingCanvases}");
        
        if (loadingCanvases == 0)
        {
            Debug.LogWarning("⚠️ [6] Aucun Canvas de chargement détecté !");
        }
        
        // 7. IMAGES VISIBLES
        Image[] allImages = FindObjectsByType<Image>(FindObjectsInactive.Exclude, FindObjectsSortMode.None);
        Debug.Log($"📊 [7] {allImages.Length} Image(s) active(s)");
        
        if (allImages.Length == 0)
        {
            Debug.LogError("❌ [7] Aucune image active - Écran complètement vide !");
        }
        
        // 8. EVENT SYSTEM
        UnityEngine.EventSystems.EventSystem eventSystem = FindFirstObjectByType<UnityEngine.EventSystems.EventSystem>();
        if (eventSystem == null)
        {
            Debug.LogWarning("⚠️ [8] Pas d'EventSystem - Interactions UI impossibles");
        }
        else
        {
            Debug.Log($"✅ [8] EventSystem: {eventSystem.name}");
        }
        
        // 9. VÉRIFICATION MAC SPÉCIFIQUE
        #if UNITY_STANDALONE_OSX || UNITY_EDITOR_OSX
        Debug.Log("═══════════════════════════════════════");
        Debug.Log("🍎 VÉRIFICATIONS MAC SPÉCIFIQUES");
        Debug.Log("═══════════════════════════════════════");
        Debug.Log($"Graphics API: {SystemInfo.graphicsDeviceType}");
        Debug.Log($"Internet: {Application.internetReachability}");
        
        // Vérifier MacCompatibilityFix
        MacCompatibilityFix macFix = FindFirstObjectByType<MacCompatibilityFix>();
        if (macFix == null)
        {
            Debug.LogWarning("⚠️ MacCompatibilityFix absent - Recommandé pour Mac");
        }
        else
        {
            Debug.Log("✅ MacCompatibilityFix présent");
        }
        #endif
        
        // 10. RÉSUMÉ ET RECOMMANDATIONS
        Debug.Log("═══════════════════════════════════════");
        Debug.Log("📋 RÉSUMÉ");
        Debug.Log("═══════════════════════════════════════");
        
        bool hasCamera = mainCam != null;
        bool hasConfig = GeneralConfigManager.Instance != null && GeneralConfigManager.Instance.IsConfigLoaded();
        bool hasMapManager = mapManager != null;
        bool hasLoadingManager = UnifiedLoadingManager.Instance != null;
        bool hasVisibleContent = allImages.Length > 0;
        
        int score = (hasCamera ? 1 : 0) + (hasConfig ? 1 : 0) + (hasMapManager ? 1 : 0) + 
                   (hasLoadingManager ? 1 : 0) + (hasVisibleContent ? 1 : 0);
        
        Debug.Log($"Score: {score}/5");
        
        if (score >= 4)
        {
            Debug.Log("✅ Scène semble correctement configurée");
            
            if (!hasVisibleContent)
            {
                Debug.LogWarning("⚠️ MAIS aucun contenu visible - Vérifier le chargement des assets");
            }
        }
        else
        {
            Debug.LogError("❌ Problèmes de configuration détectés !");
            
            if (!hasCamera) Debug.LogError("   → Ajouter une caméra avec tag 'MainCamera'");
            if (!hasConfig) Debug.LogError("   → GeneralConfigManager ne charge pas la config");
            if (!hasMapManager) Debug.LogError("   → Ajouter MapManager à la scène");
            if (!hasLoadingManager) Debug.LogError("   → Ajouter UnifiedLoadingManager à la scène");
        }
        
        if (fixApplied)
        {
            Debug.Log("🔧 Fix de l'écran noir déjà appliqué");
        }
        
        Debug.Log("═══════════════════════════════════════");
        Debug.Log("FIN DU DIAGNOSTIC");
        Debug.Log("Appuyez sur F12 pour relancer");
        Debug.Log("═══════════════════════════════════════");
    }
}

