using UnityEngine;

/// <summary>
/// Force une résolution 16:9 au démarrage pour garantir la cohérence des zones
/// TEMPORAIRE : Pour diagnostiquer le problème Mac
/// </summary>
public class ForceResolution : MonoBehaviour
{
    [Header("Configuration")]
    [Tooltip("Activer le forçage de résolution")]
    public bool forceResolution = true;
    
    [Tooltip("Largeur de référence")]
    public int targetWidth = 1920;
    
    [Tooltip("Hauteur de référence")]
    public int targetHeight = 1080;
    
    [Tooltip("Mode plein écran")]
    public bool fullscreen = true;
    
    [Header("Debug")]
    public bool showDebugLogs = true;

    void Awake()
    {
        if (!forceResolution) return;
        
        if (showDebugLogs)
        {
            Debug.Log($"[ForceResolution] 🖥️ Résolution actuelle AVANT: {Screen.width}x{Screen.height}");
            Debug.Log($"[ForceResolution] Ratio actuel AVANT: {(float)Screen.width / Screen.height:F3}");
            Debug.Log($"[ForceResolution] Fullscreen AVANT: {Screen.fullScreen}");
        }
        
        // IMPORTANT : Attendre un frame pour que l'écran soit initialisé
        StartCoroutine(ApplyResolutionAfterFrame());
    }
    
    System.Collections.IEnumerator ApplyResolutionAfterFrame()
    {
        // Attendre 2 frames pour s'assurer que tout est initialisé
        yield return null;
        yield return null;
        
        if (showDebugLogs)
        {
            Debug.Log($"[ForceResolution] Application de la résolution {targetWidth}x{targetHeight}...");
        }
        
        // Forcer la résolution
        Screen.SetResolution(targetWidth, targetHeight, fullscreen);
        
        if (showDebugLogs)
        {
            Debug.Log($"[ForceResolution] ✅ SetResolution appelé : {targetWidth}x{targetHeight}, fullscreen={fullscreen}");
        }
    }
    
    void Start()
    {
        if (!forceResolution) return;
        
        // Vérifier que la résolution a bien été appliquée
        if (showDebugLogs)
        {
            Debug.Log($"[ForceResolution] Résolution finale: {Screen.width}x{Screen.height}");
            Debug.Log($"[ForceResolution] Ratio final: {(float)Screen.width / Screen.height:F3}");
            
            float targetAspect = (float)targetWidth / targetHeight;
            float currentAspect = (float)Screen.width / Screen.height;
            
            if (Mathf.Abs(targetAspect - currentAspect) < 0.01f)
            {
                Debug.Log($"[ForceResolution] ✅ Ratio correct !");
            }
            else
            {
                Debug.LogWarning($"[ForceResolution] ⚠️ Ratio différent ! Attendu: {targetAspect:F3}, Obtenu: {currentAspect:F3}");
            }
        }
    }
    
    void OnGUI()
    {
        if (!showDebugLogs) return;
        
        // Afficher les infos en haut à gauche
        GUIStyle style = new GUIStyle();
        style.fontSize = 16;
        style.normal.textColor = Color.yellow;
        
        string info = $"Résolution: {Screen.width}x{Screen.height}\n";
        info += $"Ratio: {(float)Screen.width / Screen.height:F3}\n";
        info += $"Fullscreen: {Screen.fullScreen}";
        
        GUI.Label(new Rect(10, 10, 300, 100), info, style);
    }
}

